// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.mda;

import de.caff.annotation.NotNull;
import de.caff.generics.function.Function1;
import de.caff.generics.function.LongOperator1;

import java.util.Arrays;
import java.util.function.Function;

/**
 * A multi-dimensional long array.
 * <p>
 * There is no enforced limit to the number of dimensions,
 * but internally this is mapped to a standard Java array,
 * which can only hold a limited number of elements.
 * <p>
 * Please note that although this implements
 * {@link MultiDimensionalAccess#getElement(int...)}
 * and
 * {@link MultiDimensionalAccess#setElement(Object, int...)}
 * to be used in generic circumstances the recommended methods
 * for getting and setting are
 * {@link #getValueAt(int...)}
 * and
 * {@link #setValueAt(long, int...)}
 * because they avoid unnecessary boxing.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since November 01, 2019
 */
public class MultiDimensionalLongArray
        extends AbstractBasicMultiDimensionalArray<Long>
{
  @NotNull
  private final long[] array;

  /**
   * Constructor.
   * @param sizes of the dimensions of this array,
   *              e.g. {@code (2, 3, 4}} will create
   *              an oblong array with size 2 in the first,
   *              size 3 in the second, and size 4 in the third
   *              dimension, containing {@code  2*3*4 = 24} elements
   */
  public MultiDimensionalLongArray(int... sizes)
  {
    super(sizes);
    array = new long[(int)getNumElements()];
  }

  /**
   * Sub array constructor.
   * <p>
   * This constructs a new read/write access to the given {@code baseArray}
   * allowing you to do nearly everything. As this gives a lot of power
   * be sure that the {@code indexConverter} does "the right thing",
   * otherwise runtime exceptions will occur. 
   * @param baseArray      basic array to which accesses to this array are mapped
   * @param indexConverter index converter which converts indexes into this array
   *                       to indexes into the basic array. Failing to fulfill
   *                       the restrictions (parameter length has to be 
   *                       the same length as the following {@code sizes}, and 
   *                       fit into their ranges, and the result indexes have
   *                       to fit the restrictions of the {@code baseArray}
   *                       will result in runtime errors
   * @param sizes          sizes of the dimensions of this array
   */
  public MultiDimensionalLongArray(@NotNull MultiDimensionalLongArray baseArray,
                                   @NotNull Function1<int[], int[]> indexConverter,
                                   int... sizes)
  {
    this(baseArray,
         new MappingMultiIndexLinearizer(baseArray.getIndexLinearizer(),
                                         indexConverter,
                                         sizes));
  }

  /**
   * Sub array constructor.
   * @param parent     parent array
   * @param linearizer tweaked linearizer
   */
  private MultiDimensionalLongArray(@NotNull MultiDimensionalLongArray parent,
                                    @NotNull MultiIndexLinearizer linearizer)
  {
    super(linearizer);
    array = parent.array;
  }

  @Override
  @NotNull
  public Long getElement(int... indexes)
  {
    return getValueAt(indexes);
  }

  @Override
  public void setElement(@NotNull Long value, int... indexes)
  {
    setValueAt(value, indexes);
  }

  @Override
  @NotNull
  public Long change(@NotNull Function<? super Long, ? extends Long> operator, int... indexes)
  {
    return changeValueAt(operator::apply, indexes);
  }

  /**
   * Get the element at the given index combination.
   * @param indexes indexes of the element
   * @return element at the given index
   */
  public long getValueAt(int... indexes)
  {
    return array[toLinear(indexes)];
  }

  /**
   * Set the element at the given index combination.
   * @param elem    element to set
   * @param indexes indexes of the element
   */
  public void setValueAt(long elem, int... indexes)
  {
    array[toLinear(indexes)] = elem;
  }

  /**
   * Change the element at the given index combination.
   * @param operator operator to apply
   * @param indexes  indexes of the element
   * @return changed element value
   */
  public long changeValueAt(@NotNull LongOperator1 operator,
                            int... indexes)
  {
    final int index = toLinear(indexes);
    final long value = operator.applyAsLong(array[index]);
    array[index] = value;
    return value;
  }

  /**
   * Set the elements of this array from the given values.
   * <p>
   * This uses {@link MultiIndexLinearizer#getHighFastSequencer()}
   * the internal order to copy values.
   *
   * @param values values, at least {@link #getNumElements()}
   */
  public void setValuesFrom(@NotNull long... values)
  {
    if (values.length < array.length) {
      throw new IllegalArgumentException(String.format("Need at least %d elements for setting, but got %d!",
                                                       getNumElements(), values.length));
    }
    System.arraycopy(values, 0, array, 0, array.length);
  }

  /**
   * Set the elements of this array from the given values
   * using the defined order.
   * @param sequencer sequencer defining the order
   * @param values values, at least
   */
  public void setValuesFrom(@NotNull MultiIndexLinearizer.Sequencer sequencer,
                            @NotNull long... values)
  {
    if (values.length < array.length) {
      throw new IllegalArgumentException(String.format("Need at least %d elements for setting, but got %d!",
                                                       getNumElements(), values.length));
    }

    for (int i = array.length - 1;  i >= 0;  --i) {
      array[toLinear(sequencer.get(i))] = values[i];
    }
  }

  /**
   * Get access to a multi-dimensional sub array of this array.
   * The returned array will forward all operations
   * including setting elements to this array.
   * @param indexes indexes of sub array, negative for open indexes,
   *                {@code 0} or positive for fixed indexes, if
   *                there are not enough indexes the missing ones
   *                are considered open
   * @return multi-dimensional sub array, the number of dimensions is
   *         defined by the number of open indexes
   */
  @NotNull
  public MultiDimensionalLongArray sub(int... indexes)
  {
    return new MultiDimensionalLongArray(this, indexLinearizer.sub(indexes));
  }

  @Override
  public boolean equals(Object o)
  {
    if (this == o) {
      return true;
    }
    if (o == null || getClass() != o.getClass()) {
      return false;
    }
    final MultiDimensionalLongArray that = (MultiDimensionalLongArray)o;
    if (!Arrays.equals(getSizes(), that.getSizes())) {
      return false;
    }
    for (int[] indexes : indexLinearizer.getHighFastSequencer()) {
      if (getValueAt(indexes) != that.getElement(indexes)) {
        return false;
      }
    }
    return true;
  }

  @Override
  public int hashCode()
  {
    return Arrays.deepHashCode(new Object[] { getSizes(), toList(indexLinearizer.getHighFastSequencer())});
  }

  /**
   * Get an independent copy.
   * @return independent copy
   */
  @NotNull
  public MultiDimensionalLongArray getCopy()
  {
    final MultiDimensionalLongArray copy = new MultiDimensionalLongArray(getSizes());
    for (int[] indexes : indexLinearizer.getHighFastSequencer()) {
      copy.setValueAt(getValueAt(indexes), indexes);
    }
    return copy;
  }
}

