// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.mda;

import de.caff.annotation.NotNull;
import de.caff.generics.function.Function1;

import java.util.*;
import java.util.function.Function;

/**
 * Implementation of a basic multi-dimensional array.
 * <p>
 * There is no enforced limit to the number of dimensions,
 * but internally this is mapped to an {@link ArrayList},
 * which can only hold a limited number of elements.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since November 01, 2019
 */
public class MultiDimensionalArray<T>
        extends AbstractBasicMultiDimensionalArray<T>
{
  @NotNull
  private final List<T> elements;

  /**
   * Basic array constructor.
   * @param sizes of the dimensions of this array,
   *              e.g. {@code (2, 3, 4}} will create
   *              an oblong array with size 2 in the first,
   *              size 3 in the second, and size 4 in the third
   *              dimension, containing {@code  2*3*4 = 24} elements
   */
  public MultiDimensionalArray(int... sizes)
  {
    this(new HighFastMultiIndexLinearizer(sizes));
  }

  /**
   * Basic array constructor.
   * @param linearizer multi-index linearizer, defining dimensions and mappings
   */
  public MultiDimensionalArray(@NotNull MultiIndexLinearizer linearizer)
  {
    super(linearizer);
    elements = new ArrayList<>((int)getNumElements());
    for (int i = (int)getNumElements();  i > 0;  --i) {
      elements.add(null);
    }
  }

  /**
   * Sub array constructor.
   * <p>
   * This constructs a new read/write access to the given {@code baseArray}
   * allowing you to do nearly everything. As this gives a lot of power
   * be sure that the {@code indexConverter} does "the right thing",
   * otherwise runtime exceptions will occur.
   * @param baseArray      basic array to which accesses to this array are mapped
   * @param indexConverter index converter which converts indexes into this array
   *                       to indexes into the basic array. Failing to fulfill
   *                       the restrictions (parameter length has to be
   *                       the same length as the following {@code sizes}, and
   *                       fit into their range, and the result indexes have
   *                       to fit the restrictions of the {@code baseArray}
   *                       will result in runtime errors
   * @param sizes          sizes of the dimensions of this array
   */
  public MultiDimensionalArray(@NotNull MultiDimensionalArray<T> baseArray,
                               @NotNull Function1<int[], int[]> indexConverter,
                               int... sizes)
  {
    this(baseArray.elements,
         new MappingMultiIndexLinearizer(baseArray.getIndexLinearizer(),
                                              indexConverter,
                                              sizes));
  }

  /**
   * Sub-array constructor.
   * @param elems      elements of the sub array
   * @param linearizer tweaked linearizer for accessing the sub array
   */
  private MultiDimensionalArray(@NotNull List<T> elems,
                                @NotNull MultiIndexLinearizer linearizer)
  {
    super(linearizer);
    elements = elems;
  }

  /**
   * Sub array constructor.
   * @param elems elements of the sub array
   * @param sizes sizes of the sub array
   */
  private MultiDimensionalArray(@NotNull List<T> elems,
                                int[] sizes)
  {
    super(sizes);
    elements = elems;
  }

  /**
   * Get the element at the given index combination.
   * @param indexes indexes of the element
   * @return element at the given index
   */
  public T getElement(int... indexes)
  {
    return elements.get(indexLinearizer.toLinear(indexes));
  }

  /**
   * Set the element at the given index combination.
   * @param elem    element to set
   * @param indexes indexes of the element
   */
  public void setElement(T elem, int... indexes)
  {
    elements.set(indexLinearizer.toLinear(indexes), elem);
  }

  @Override
  public T change(@NotNull Function<? super T, ? extends T> operator, int... indexes)
  {
    final int index = indexLinearizer.toLinear(indexes);
    final T value = operator.apply(elements.get(index));
    elements.set(index, value);
    return value;
  }

  @Override
  public void setFrom(@NotNull Collection<? extends T> elements)
  {
    if (elements.size() < getNumElements()) {
      throw new IllegalArgumentException(String.format("Need at least %d elements for setting, but got %d!",
                                                       getNumElements(), elements.size()));
    }
    int index = 0;
    for (T e : elements) {
      this.elements.set(index++, e);
    }
  }

  /**
   * Get access to a multi-dimensional sub array of this array.
   * The returned array will forward all operations
   * including setting elements to this array.
   * @param indexes indexes of sub array, negative for open indexes,
   *                {@code 0} or positive for fixed indexes, if
   *                there are not enough indexes the missing ones
   *                are considered open
   * @return multi-dimensional sub array, the number of dimensions is
   *         defined by the number of open indexes
   */
  @NotNull
  public MultiDimensionalArray<T> sub(@NotNull int... indexes)
  {
    if (indexes.length == 0) {
      return this;
    }
    return new MultiDimensionalArray<>(elements,
                                       indexLinearizer.sub(indexes));
  }

  @Override
  public boolean equals(Object o)
  {
    if (this == o) {
      return true;
    }
    if (o == null || getClass() != o.getClass()) {
      return false;
    }
    final MultiDimensionalArray<?> that = (MultiDimensionalArray<?>)o;
    if (!Arrays.equals(getSizes(), that.getSizes())) {
      return false;
    }
    for (int[] indexes : indexLinearizer.getHighFastSequencer()) {
      if (!getElement(indexes).equals(that.getElement(indexes))) {
        return false;
      }
    }
    return true;
  }

  @Override
  public int hashCode()
  {
    return Arrays.deepHashCode(new Object[] { getSizes(), toList(indexLinearizer.getHighFastSequencer())});
  }

  /**
   * Get an independent copy.
   * @return independent copy
   */
  @NotNull
  public MultiDimensionalArray<T> getCopy()
  {
    final MultiDimensionalArray<T> copy = new MultiDimensionalArray<>(getSizes());
    copy.elements.clear();
    copy.elements.addAll(elements);
    return copy;
  }

  /**
   * Get an independent copy with adapted element values.
   * @param mapper mapper for value adaption
   * @param <R> element type of copy
   * @return independent copy with mapped elements
   */
  @NotNull
  public <R> MultiDimensionalArray<R> getCopy(@NotNull Function1<? extends R, ? super T> mapper)
  {
    final MultiDimensionalArray<R> copy = new MultiDimensionalArray<>(getSizes());
    for (int linear = elements.size() - 1;  linear >= 0;  --linear) {
      copy.elements.set(linear, mapper.apply(elements.get(linear)));
    }
    return copy;
  }

}
