// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.mda;

import de.caff.annotation.NotNull;
import de.caff.generics.Indexable;
import de.caff.generics.function.Function2;

import java.util.Collection;
import java.util.Iterator;
import java.util.function.BiFunction;
import java.util.function.Function;

/**
 * Multi-dimensional read-write access.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since November 02, 2019
 */
public interface MultiDimensionalAccess<T>
        extends MultiDimensionalReadAccess<T>
{
  /**
   * Set the element located at the given indexes.
   * @param value   value to set at the location defined by the given indexes
   * @param indexes {@link MultiDimensional#getNumDimensions()} indexes inside the bounds
   *                defined by {@link MultiDimensional#getSizes()}
   */
  void setElement(T value, int... indexes);

  /**
   * Change the value located at the given indexes.
   * <p>
   * This default implementation is only slightly more efficient than
   * a {@link #getElement(int...) get} and {@link #setElement(Object, int...) set} sequence,
   * but implementing classes are encouraged to provide better implementations
   * (e.g. the index linearization has only to happen once).
   *
   * @param operator operator to apply to the value at the location defined by the given indexes.
   *                 When used on a specialized primitive type array the operator
   *                 will never receive {@code null}, and must not return {@code null}.
   *                 In other cases this depends on usage.
   * @param indexes  {@link MultiDimensional#getNumDimensions()} indexes inside the bounds
   *                 defined by {@link MultiDimensional#getSizes()}
   * @return changed value
   */
  default T change(@NotNull Function<? super T, ? extends T> operator,
                   int... indexes)
  {
    T value = operator.apply(getElement(indexes));
    setElement(value, indexes);
    return value;
  }

  /**
   * Change all elements in this multi-dimensional array.
   * This will run over all elements and call the operator on each.
   * @param operator operator to change the value, with the current value as first
   *                 argument and the indexes as second, returns the new value
   */
  default void changeAll(@NotNull BiFunction<? super T, int[], ? extends T> operator)
  {
    final int[] sizes = getSizes();
    if (sizes.length == 0) {
      return;
    }
    final int[] run = new int[sizes.length];
    final int last = sizes.length - 1;
    while (run[0] < sizes[0]) {
      change(Function2.from(operator).partialRight(run.clone()), run);
      for (int i = last;  i > 0;  --i) {
        if (++run[i] == sizes[i]) {
          run[i] = 0;
        }
        else {
          break;
        }
      }
    }
  }

  /**
   * Change all elements in this multi-dimensional array.
   * This will run over all elements and call the operator on each.
   * @param operator operator to change the value, with the current value as argument
   */
  default void changeAll(@NotNull Function<? super T, ? extends T> operator)
  {
    final int[] sizes = getSizes();
    if (sizes.length == 0) {
      return;
    }
    final int[] run = new int[sizes.length];
    final int last = sizes.length - 1;
    while (run[0] < sizes[0]) {
      change(operator, run);
      for (int i = last;  i > 0;  --i) {
        if (++run[i] == sizes[i]) {
          run[i] = 0;
        }
        else {
          break;
        }
      }
    }
  }

  /**
   * Set all elements in this multi-dimensional access from the given collection.
   * @param sequencer sequencer defining the order in which the elements are set
   * @param elements collection with at least {@link #getNumElements()} elements
   * @throws IllegalArgumentException if there are not enough elements in the collection.
   *                                  This access is not changed in this case.
   */
  default void setFrom(@NotNull MultiIndexLinearizer.Sequencer sequencer,
                       @NotNull Collection<? extends T> elements)
  {
    if (elements.size() < getNumElements()) {
      throw new IllegalArgumentException(String.format("Need at least %d elements for setting, but got %d!",
                                                       getNumElements(), elements.size()));
    }
    final Iterator<? extends T> elementIterator = elements.iterator();
    for (int[] indexes : sequencer) {
      setElement(elementIterator.next(), indexes);
    }
  }

  /**
   * Set all elements in this multi-dimensional access from the given indexable collection.
   * @param sequencer sequencer defining the order in which the elements are set
   * @param elements indexable collection with at least {@link #getNumElements()} elements
   * @throws IllegalArgumentException if there are not enough elements in the indexable collection.
   *                                  This access is not changed in this case.
   */
  default void setFrom(@NotNull MultiIndexLinearizer.Sequencer sequencer,
                       @NotNull Indexable<? extends T> elements)
  {
    if (elements.size() < getNumElements()) {
      throw new IllegalArgumentException(String.format("Need at least %d elements for setting, but got %d!",
                                                       getNumElements(), elements.size()));
    }
    final Iterator<? extends T> elementIterator = elements.iterator();
    for (int[] indexes : sequencer) {
      setElement(elementIterator.next(), indexes);
    }
  }

  /**
   * Set all elements in this multi-dimensional access from the given iterable.
   * @param sequencer sequencer
   * @param elements  iteration with at least {@link #getNumElements()} elements
   * @throws IllegalArgumentException if there are not enough elements in the collection.
   *                                  This access is probably already changed in this case.
   */
  default void setFrom(@NotNull MultiIndexLinearizer.Sequencer sequencer,
                       @NotNull Iterable<? extends T> elements)
  {
    final Iterator<? extends T> elementIterator = elements.iterator();
    for (int[] indexes : sequencer) {
      if (!elementIterator.hasNext()) {
        throw new IllegalArgumentException(String.format("Need at least %d elements for setting, but got less!",
                                                         getNumElements()));
      }
      setElement(elementIterator.next(), indexes);
    }
  }

  /**
   * Set all elements.
   * @param provider provider called for each element
   * @return {@code this} for chaining
   */
  default MultiDimensionalAccess<T> setAll(@NotNull Function<int[], ? extends T> provider)
  {
    final int[] sizes = getSizes();
    final int length = sizes.length;
    final int[] run = new int[length];
    final int last = length - 1;
    int carry = last;
    while (carry >= 0) {
      setElement(provider.apply(run.clone()), run);
      for (carry = last;  carry >= 0;  --carry) {
        if (++run[carry] < sizes[carry]) {
          break;
        }
        run[carry] = 0;
      }
    }
    return this;
  }
}
