// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.mda;

import de.caff.annotation.NotNull;
import de.caff.generics.function.Function1;

/**
 * Multi-index linearizer which provides an arbitrary mapping.
 * <p>
 * Not public because automatically created when accessing sub arrays
 * and therefore not fit for base arrays.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since November 04, 2019
 */
class MappingMultiIndexLinearizer
        implements MultiIndexLinearizer
{
  @NotNull
  private final MultiIndexLinearizer baseIndexLinearizer;
  @NotNull
  private final int[] sizes;
  @NotNull
  private final Function1<int[], int[]> mapper;

  private final int numElements;

  /**
   * Constructor.
   * @param baseIndexLinearizer basic multi-index linearizer on which this linearizer maps
   * @param mapper              mapper from incoming indexes to indexes the {@code baseIndexLinearizer}
   *                            requires. Failing to fulfill the restrictions will result in
   *                            runtime errors
   * @param sizes               sizes of the dimensions this multi-index linea provides
   */
  MappingMultiIndexLinearizer(@NotNull MultiIndexLinearizer baseIndexLinearizer,
                              @NotNull Function1<int[], int[]> mapper,
                              int... sizes)
  {
    this.baseIndexLinearizer = baseIndexLinearizer;
    this.sizes = sizes.clone();
    this.mapper = mapper;
    int mult = 1;
    for (int s : sizes) {
      mult *= s;
    }
    numElements = mult;
  }

  @Override
  public int toLinear(int... indexes)
  {
    MultiIndexLinearizer.checkIndexes(sizes, indexes);
    return baseIndexLinearizer.toLinear(mapper.apply(indexes));
  }

  @Override
  public int getNumDimensions()
  {
    return sizes.length;
  }

  @Override
  public int getSize(int dim)
  {
    return sizes[dim];
  }

  @NotNull
  @Override
  public int[] getSizes()
  {
    return sizes.clone();
  }

  @Override
  public long getNumElements()
  {
    return numElements;
  }
}
