// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.mda;

import de.caff.annotation.NotNull;

import java.math.BigInteger;

/**
 * Simple highest-index fast implementation of a multi-index linearizer.
 * It maps indexes of the highest dimension to be separated
 * by {@code 1}. So iterating over these might result in caching effects.
 * <p>
 * This is the default multi-index linearizer used in the array classes
 * in this package.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since November 01, 2019
 */
public class HighFastMultiIndexLinearizer
        implements MultiIndexLinearizer
{
  private final BigInteger MAX_INT = BigInteger.valueOf(Integer.MAX_VALUE);

  /** The sizes of the dimensions of this array. */
  @NotNull
  final int[] sizes;
  private final int numElements;
  private final int[] multiplicators;

  /**
   * Constructor.
   * @param sizes of the dimensions of this array,
   *              e.g. {@code (2, 3, 4}} will create
   *              an oblong array with size 2 in the first,
   *              size 3 in the second, and size 4 in the third
   *              dimension, containing {@code  2*3*4 = 24} elements
   */
  protected HighFastMultiIndexLinearizer(int ... sizes)
  {
    BigInteger elems = BigInteger.ONE;
    multiplicators = new int[sizes.length];
    int mult = 1;
    for (int i = sizes.length - 1;  i >= 0; --i) {
      final int s = sizes[i];
      if (s <= 0) {
        throw new IllegalArgumentException("Dimensional sizes have to be positive!");
      }
      elems = elems.multiply(BigInteger.valueOf(s));
      multiplicators[i] = mult;
      mult *= s;
    }
    if (elems.compareTo(MAX_INT) >= 0) {
      throw new IllegalArgumentException("Too many elements for integer index: "+elems+"!");
    }
    this.sizes = sizes.clone();
    numElements = mult;
  }

  @Override
  public int getNumDimensions()
  {
    return sizes.length;
  }

  @Override
  public int getSize(int dim)
  {
    return sizes[dim];
  }

  @NotNull
  @Override
  public int[] getSizes()
  {
    return sizes.clone();
  }

  @Override
  public int toLinear(int... indexes)
  {
    if (indexes.length != sizes.length) {
      throw new IllegalArgumentException(String.format("Incorrect number of indexes for an array with %d dimensions: %d!",
                                                       sizes.length, indexes.length));
    }
    int linearIndex = 0;
    for (int i = sizes.length - 1;  i >= 0;  --i) {
      final int index = indexes[i];
      MultiIndexLinearizer.checkIndex(i, sizes[i], index);
      linearIndex += multiplicators[i] * index;
    }
    return linearIndex;
  }

  @Override
  public long getNumElements()
  {
    return numElements;
  }

  /**
   * Get the multi-dimensional index mapped to a given linear index.
   * @param linearIndex linear index between {@code 0} (included) and
   *                    {@link #getNumElements()} (excluded)
   * @return multi-dimensional indexes which will be mapped to {@code linearIndex}
   */
  @NotNull
  public int[] fromLinear(int linearIndex)
  {
    if (linearIndex < 0  ||  linearIndex >= numElements) {
      throw new IllegalArgumentException("Index out of range: "+linearIndex);
    }
    final int[] result = new int[sizes.length];
    for (int i = 0;  i < result.length;  ++i) {
      final int mult = multiplicators[i];
      result[i] = linearIndex / mult;
      linearIndex %= mult;
    }
    return result;
  }
}
