// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.mda;

import de.caff.annotation.NotNull;
import de.caff.generics.Indexable;
import de.caff.generics.Types;
import de.caff.generics.function.Function1;

import java.util.*;

/**
 * Basic implementation of a multi-dimensional array.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since November 01, 2019
 */
public abstract class AbstractBasicMultiDimensionalArray<T>
        implements MultiDimensionalAccess<T>,
                   Iterable<T>
{
  @NotNull
  protected final MultiIndexLinearizer indexLinearizer;

  /**
   * Constructor.
   * @param sizes of the dimensions of this array,
   *              e.g. {@code (2, 3, 4}} will create
   *              an oblong array with size 2 in the first,
   *              size 3 in the second, and size 4 in the third
   *              dimension, containing {@code  2*3*4 = 24} elements
   */
  protected AbstractBasicMultiDimensionalArray(int ... sizes)
  {
    this(new HighFastMultiIndexLinearizer(sizes));
  }

  /**
   * Constructor.
   * @param linearizer multi index linearizer
   */
  protected AbstractBasicMultiDimensionalArray(@NotNull MultiIndexLinearizer linearizer)
  {
    indexLinearizer = linearizer;
  }

  /**
   * Get the multi-index linearizer on which this multi-dimensional array is based.
   * <p>
   * Please note that the linearizer for
   * @return multi-index linearizer
   */
  @NotNull
  public MultiIndexLinearizer getIndexLinearizer()
  {
    return indexLinearizer;
  }

  /**
   * Get the linear index for the given multi-dimensional indexes.
   * @param indexes indexes
   * @return linear index
   */
  protected int toLinear(int... indexes)
  {
    return indexLinearizer.toLinear(indexes);
  }

  /**
   * Get the number of dimensions of this array
   * @return number of dimensions of this array
   */
  public int getNumDimensions()
  {
    return indexLinearizer.getNumDimensions();
  }

  /**
   * Get the size of the given dimension.
   * @param dim dimension
   * @return number of elements in the given dimension
   */
  public int getSize(int dim)
  {
    return indexLinearizer.getSize(dim);
  }

  /**
   * Get the sizes of the dimensions of this array.
   * @return array of length {@link #getNumDimensions()}
   */
  @NotNull
  public int[] getSizes()
  {
    return indexLinearizer.getSizes();
  }

  /**
   * Get the combined number of elements in all dimensions.
   * @return number of all elements in this multi-dimensional array
   */
  public long getNumElements()
  {
    return indexLinearizer.getNumElements();
  }

  /**
   * Get an iterator over a linearized view of this multi-dimensional array.
   * This forwards to {@link #iterator(MultiIndexLinearizer.Sequencer)}
   * with a {@link MultiIndexLinearizer#getHighFastSequencer() high fast sequener}.
   * @return iterator
   */
  @NotNull
  @Override
  public Iterator<T> iterator()
  {
    return iterator(indexLinearizer.getHighFastSequencer());
  }

  /**
   * Get an iterator over a linearized view of this multi-dimensional array.
   *
   * @param sequencer sequencer to use for iterating over this multi-dimensional array,
   *                  must provide the same dimensions
   * @return linear iterator over this multi-dimensional array
   */
  @NotNull
  public Iterator<T> iterator(@NotNull MultiIndexLinearizer.Sequencer sequencer)
  {
    if (!Arrays.equals(sequencer.getSizes(), getSizes())) {
      throw new IllegalArgumentException(String.format("Sequencer working on dimensions %s does not fit to this multi-dimensional array with dimensions %s!",
                                                       Arrays.toString(sequencer.getSizes()),
                                                       Arrays.toString(getSizes())));
    }

    final Iterator<int[]> sequenceIterator = sequencer.iterator();

    return new Iterator<T>()
    {
      @Override
      public boolean hasNext()
      {
        return sequenceIterator.hasNext();
      }

      @Override
      public T next()
      {
        return getElement(sequenceIterator.next());
      }
    };
  }

  /**
   * Set all elements in this multi-dimensional array from the given collection.
   * This forwards to {@link #setFrom(MultiIndexLinearizer.Sequencer, Collection)}
   * with a {@link MultiIndexLinearizer#getHighFastSequencer() high fast sequener}.
   * @param elements array elements with at least {@link #getNumElements()} elements
   */
  public void setFrom(@NotNull Collection<? extends T> elements)
  {
    setFrom(indexLinearizer.getHighFastSequencer(), elements);
  }

  /**
   * Set all elements in this multi-dimensional array in high-fast order.
   * @param elements elements in order, at least {@link #getNumElements()}
   * @throws IllegalArgumentException if there are not enough elements in the iterable
   */
  public void setFrom(@NotNull Iterable<? extends T> elements)
  {
    setFrom(indexLinearizer.getHighFastSequencer(), elements);
  }

  /**
   * Set all elements in this multi-dimensional array in the order
   * defined by the sequencer.
   * @param sequencer sequencer defining the order in which elements are set
   * @param elements  elements in order, at least {@link #getNumElements()}
   * @throws IllegalArgumentException if there are not enough elements in the iterable
   */
  public void setFrom(@NotNull MultiIndexLinearizer.Sequencer sequencer,
                      @NotNull Iterable<? extends T> elements)
  {
    final Iterator<? extends T> elementIterator = elements.iterator();
    for (int[] indexes : indexLinearizer.getHighFastSequencer()) {
      if (!elementIterator.hasNext())  {
        throw new IllegalArgumentException("Not enough elements in iterable!");
      }
      setElement(elementIterator.next(), indexes);
    }
  }

  @NotNull
  public Indexable<T> asLinearArray()
  {
    return asLinearArray(v -> v);
  }

  @NotNull
  public Indexable<T> asLinearArray(@NotNull MultiIndexLinearizer.Sequencer sequencer)
  {
    return asLinearArray(sequencer, v -> v);
  }

  @NotNull
  public <R> Indexable<R> asLinearArray(@NotNull Function1<R, T> copier)
  {
    return asLinearArray(indexLinearizer.getHighFastSequencer(), copier);
  }

  /**
   * View this multi-dimensional array as a linear array.
   * This method allows to define the order of elements in the returned array
   * and how elements are copied over when accessed.
   * @param sequencer sequencer defining the element order, its dimensional setup
   *                  has to fit this multi-dimensional array setup
   * @param copier copier used to copy over elements from this multi-dimensional array
   *               when requested
   * @param <R> result array element type
   * @return newly created indexed array in the given order
   */
  @NotNull
  public <R> Indexable<R> asLinearArray(@NotNull MultiIndexLinearizer.Sequencer sequencer,
                                        @NotNull Function1<R, T> copier)
  {
    if (!Arrays.equals(sequencer.getSizes(), getSizes())) {
      throw new IllegalArgumentException(String.format("Sequencer working on dimensions %s does not fit to this multi-dimensional array with dimensions %s!",
                                                       Arrays.toString(sequencer.getSizes()),
                                                       Arrays.toString(getSizes())));
    }
    return new Indexable.Base<R>()
    {
      @Override
      public int size()
      {
        return (int)sequencer.getNumElements();
      }

      @Override
      public R get(int index)
      {
        return copier.apply(AbstractBasicMultiDimensionalArray.this.getElement(sequencer.get(index)));
      }
    };
  }

  /**
   * Convert this multi-dimensional array into an array list.
   *
   * @return newly created list in {@link MultiIndexLinearizer#getHighFastSequencer() high fast order}
   *         with elements copied directly
   */
  @NotNull
  public ArrayList<T> toList()
  {
    return toList(v -> v);
  }

  /**
   * Convert this multi-dimensional array into an array list
   * using a dedicated copier.
   *
   * @param copier copier used to copy over elements from this multi-dimensional array
   *               to the returned list
   * @param <R> result list element type
   * @return newly created array list in {@link MultiIndexLinearizer#getHighFastSequencer() high fast order}
   */
  @NotNull
  public <R> ArrayList<R> toList(@NotNull Function1<R, T> copier)
  {
    return toList(indexLinearizer.getHighFastSequencer(), copier);
  }

  /**
   * Convert this multi-dimensional array into an array list.
   * This method allows to define the order of elements in the list.
   * @param sequencer sequencer defining the element order, its dimensional setup
   *                  has to fit this multi-dimensional array setup
   * @return newly created list in the given order with elements copied directly
   */
  @NotNull
  public ArrayList<T> toList(@NotNull MultiIndexLinearizer.Sequencer sequencer)
  {
    return toList(sequencer, v -> v);
  }

  /**
   * Convert this multi-dimensional array into an array list.
   * This method allows to define the order of elements in the returned list
   * and how elements are copied over.
   * @param sequencer sequencer defining the element order, its dimensional setup
   *                  has to fit this multi-dimensional array setup
   * @param copier copier used to copy over elements from this multi-dimensional array
   *               to the returned list
   * @param <R> result list element type
   * @return newly created array list in the given order with copied elements
   */
  @NotNull
  public <R> ArrayList<R> toList(@NotNull MultiIndexLinearizer.Sequencer sequencer,
                                 @NotNull Function1<R, T> copier)
  {
    if (!Arrays.equals(sequencer.getSizes(), getSizes())) {
      throw new IllegalArgumentException(String.format("Sequencer working on dimensions %s does not fit to this multi-dimensional array with dimensions %s!",
                                                       Arrays.toString(sequencer.getSizes()),
                                                       Arrays.toString(getSizes())));
    }
    return Types.map(new ArrayList<>((int)sequencer.getNumElements()),
                     iterator(sequencer),
                     copier);
  }
}
