// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.matcher;

import de.caff.annotation.NotNull;
import de.caff.generics.Matcher;
import de.caff.generics.UniformMatcher;

import java.util.Comparator;

/**
 * Helper class for {@link Matcher matcher} construction.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public final class Match
{
  /**
   * Avoid construction.
   */
  private Match()
  {
  }

  /**
   * Basic matcher, using {@link Object#equals(Object)} method of first object.
   * The value {@code null} is allowed and handled.
   */
  public static final UniformMatcher<Object> DEFAULT_MATCHER = new BasicMatcher();
  /**
   * Basic matcher, using {@link Object#equals(Object)} method of second object.
   * The value {@code null} is allowed and handled.
   */
  public static final UniformMatcher<Object> REVERTED_DEFAULT_MATCHER =
          new RevertedUniformMatcher<>(DEFAULT_MATCHER);

  /**
   * Matcher which is using reference equality, i.e. each object only matches with itself.
   * The value {@code null} is allowed and handled.
   */
  public static final UniformMatcher<Object> REFERENCE_MATCHER = new ReferenceMatcher();

  /**
   * Get a reverted matcher from a basic one.
   * <p>
   * The reverted matcher compares two objects in the different direction.
   * Basically equality should not depend on the direction, but in real life things are
   * more complicated.
   *
   * @param matcher basic matcher
   * @param <T1> type 1
   * @param <T2> type 2
   * @return matcher with inverted comparison direction
   * @see RevertedMatcher
   */
  public static <T1, T2> Matcher<T1, T2> revert(@NotNull Matcher<T2, T1> matcher)
  {
    return new RevertedMatcher<>(matcher);
  }

  /**
   * Get a matcher which negates the result of a basic one.
   * @param matcher basic matcher
   * @param <T1> type 1
   * @param <T2> type 2
   * @return matcher which returns {@code true} when the basic matcher returns {@code false}
   *         and vice versa
   * @see NegateMatcher
   */
  public static <T1, T2> Matcher<T1, T2> negate(@NotNull Matcher<T1, T2> matcher)
  {
    return new NegateMatcher<>(matcher);
  }

  /**
   * Get a matcher which enhances a given matcher which is not expecting {@code null} values
   * with one which does.
   * @param matcher basic matcher not handling {@code null} values
   * @param <T1> type 1
   * @param <T2> type 2
   * @return matcher handling {@code null} values
   */
  public static <T1, T2> Matcher<T1, T2> withNull(@NotNull Matcher<T1, T2> matcher)
  {
    return new NullCheckMatcher<>(matcher);
  }

  /**
   * Create a matcher from a comparator.
   * It will consider objects equal if the comparator compares them with a {@code 0} result.
   * @param comparator comparator
   * @param <T> type
   * @return matcher which uses the comparator for evaluating equality
   * @see ComparatorMatcher
   */
  public static <T> UniformMatcher<T> from(@NotNull Comparator<T> comparator)
  {
    return new ComparatorMatcher<>(comparator);
  }

  /**
   * Create a matcher for a comparable type.
   * It will consider objects equal if the compareTo() method of the first object compares them with
   * a {@code 0} result.
   * @param type comparable type
   * @param <T>  type
   * @return matcher using the compareTo() method
   * @see ComparableMatcher
   */
  public static <T extends Comparable<T>> UniformMatcher<T> forClass(@NotNull Class<T> type)
  {
    return new ComparableMatcher<>();
  }
}
