// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.matcher;

import de.caff.generics.UniformMatcher;

import java.util.Comparator;

/**
 * A matcher which uses a comparator for equality checking.
 * <p>
 * Objects are considered equal if the comparator returns {@code 0}.
 * <p>
 * How this matcher handles {@code null} values depends on the comparator.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @see Match#from(java.util.Comparator)
 */
public class ComparatorMatcher<T>
        implements UniformMatcher<T>
{
  /** The wrapped comparator. */
  private final Comparator<T> comparator;

  /**
   * Constructor.
   * @param comparator comparator
   */
  public ComparatorMatcher(Comparator<T> comparator)
  {
    this.comparator = comparator;
  }

  /**
   * Are the given two objects considered equal?
   *
   * @param object1 object 1
   * @param object2 object 2
   * @return <code>true</code> if this matcher considers the two object equals<br>
   *         <code>false</code> if this matcher considers them different
   */
  @Override
  public boolean areEqual(T object1, T object2)
  {
    return comparator.compare(object1, object2) == 0;
  }
}
