// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.handler;

import de.caff.annotation.NotNull;
import de.caff.generics.LoopItemHandler;
import de.caff.generics.function.Function1;

/**
 * A loop item handler allowing to do convert on incoming item
 * to a multitude of outgoing items.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @see de.caff.generics.handler.ConvertingLoopItemHandler
 * @see de.caff.generics.handler.DeletingLoopItemHandler
 * @see de.caff.generics.handler.ConvertingFragileLoopItemHandler
 * @see de.caff.generics.handler.DeletingFragileLoopItemHandler
 * @see de.caff.generics.handler.MultiConvertingFragileLoopItemHandler
 * @see de.caff.generics.handler.FilteringLoopItemHandler
 */
public abstract class MultiConvertingLoopItemHandler<T, S>
        implements LoopItemHandler<S>
{
  protected final Function1<? extends Iterable<T>, ? super S> typeConverter;

  /**
   * Constructor.
   * @param typeConverter type converter
   */
  protected MultiConvertingLoopItemHandler(@NotNull Function1<? extends Iterable<T>, ? super S> typeConverter)
  {
    this.typeConverter = typeConverter;
  }

  /**
   * Handle the converted item.
   * @param converted converted item
   * @param incoming  incoming item
   * @return {@code true} go on with loop<br>
   *         {@code false} break loop
   */
  protected abstract boolean handleConverted(T converted, S incoming);

  /**
   * Handle an item.
   *
   * @param item item to handle
   * @return {@code true} go on with loop<br>
   *         {@code false} break loop
   */
  @Override
  public boolean handle(S item)
  {
    for (T converted : typeConverter.apply(item)) {
      if (!handleConverted(converted, item)) {
        return false;
      }
    }
    return true;
  }

  /**
   * Called before the loop is started.
   */
  @Override
  public void beforeStart()
  {
  }

  /**
   * Called after the loop has finished.
   */
  @Override
  public void afterFinish()
  {
  }
}
