// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.handler;

import de.caff.annotation.Nullable;
import de.caff.generics.LoopItemHandler;

/**
 * Handler which finds the index of an item in a loop.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public class FindFirstIndexHandler<T>
        implements LoopItemHandler<T>
{
  private final T object;
  private final int startIndex;
  private int currentIndex;
  private int foundIndex;
  private T foundItem;

  /**
   * Constructor.
   * This searches from the beginning.
   *
   * @param object object to look for
   */
  public FindFirstIndexHandler(@Nullable T object)
  {
    this(object, 0);
  }

  /**
   * Constructor.
   * This searches from the given start index.
   *
   * @param object object to look for
   * @param startIndex start index from where to look
   */
  public FindFirstIndexHandler(@Nullable T object, int startIndex)
  {
    this.object = object;
    this.startIndex = startIndex;
  }

  /**
   * Handle an item.
   *
   * @param item item to handle
   * @return {@code true} go on with loop<br>
   *         {@code false} break loop
   */
  @Override
  public boolean handle(T item)
  {
    if (++currentIndex >= startIndex) {
      boolean match = object == null ? item == null : object.equals(item);
      if (match) {
        foundIndex = currentIndex;
        foundItem = item;
      }
    }
    return foundIndex < 0;
  }

  /**
   * Called before the loop is started.
   */
  @Override
  public void beforeStart()
  {
    currentIndex =
    foundIndex = -1;
    foundItem = null;
  }

  /**
   * Called after the loop has finished.
   */
  @Override
  public void afterFinish()
  {
  }

  /**
   * Get the index where the object was found.
   * @return first found index or {@code -1}
   */
  public int getFoundIndex()
  {
    return foundIndex;
  }

  /**
   * Get the item which was found.
   * @return first found index or {@code null}
   */
  @Nullable
  public T getFoundItem()
  {
    return foundItem;
  }

  /**
   * Get the looked up object.
   * @return first found index or {@code null}
   */
  @Nullable
  public T getSearchObject()
  {
    return object;
  }
}
