// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.handler;

import de.caff.annotation.NotNull;
import de.caff.generics.FragileLoopItemHandler;
import de.caff.generics.FragileTypeConverter;
import de.caff.generics.TypeConverterException;

/**
 * A fragile loop item handler which converts incoming items.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @see de.caff.generics.handler.ConvertingLoopItemHandler
 * @see de.caff.generics.handler.DeletingLoopItemHandler
 * @see de.caff.generics.handler.MultiConvertingLoopItemHandler
 * @see de.caff.generics.handler.DeletingFragileLoopItemHandler
 * @see de.caff.generics.handler.MultiConvertingFragileLoopItemHandler
 * @see de.caff.generics.handler.FilteringLoopItemHandler
 */
public abstract class ConvertingFragileLoopItemHandler<T, S>
        implements FragileLoopItemHandler<S, TypeConverterException>
{
  protected final FragileTypeConverter<T, S> typeConverter;

  /**
   * Constructor.
   * @param typeConverter type converter which converts the incoming types to the collected types.
   */
  protected ConvertingFragileLoopItemHandler(@NotNull FragileTypeConverter<T, S> typeConverter)
  {
    this.typeConverter = typeConverter;
  }

  /**
   * Handle the converted item.
   * @param converted converted item
   * @param incoming  incoming item
   * @return {@code true} go on with loop<br>
   *         {@code false} break loop
   * @throws Exception if converter throws it
   */
  protected abstract boolean handleConverted(T converted, S incoming) throws Exception;

  /**
   * Handle an item.
   *
   * @param item item to handle
   * @return {@code true} go on with loop<br>
   *         {@code false} break loop
   * @throws TypeConverterException on type conversion errors
   */
  @Override
  public boolean handle(S item) throws TypeConverterException
  {
    try {
      return handleConverted(typeConverter.convert(item), item);
    } catch (Exception e) {
      throw new TypeConverterException(e);
    }
  }

  /**
   * Called before the loop is started.
   */
  @Override
  public void beforeStart()
  {
  }

  /**
   * Called after the loop has finished.
   */
  @Override
  public void afterFinish()
  {
  }
}
