// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.handler;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;
import de.caff.generics.FragileLoopItemHandler;
import de.caff.generics.LoopItemHandler;

/**
 * Wrapper allowing to use a {@link de.caff.generics.FragileLoopItemHandler}
 * as a standard {@link de.caff.generics.LoopItemHandler}.
 * <p>
 * For getting such a handler it's recommended to use
 * {@link de.caff.generics.Types#toLoopItemHandler(de.caff.generics.FragileLoopItemHandler)}.
 * <p>
 * The wrapper will end the loop (i,e, return {@code false} from
 * {@link LoopItemHandler#handle(Object)} if an exception was thrown.
 *
 * @param <T> item type
 * @param <E> exception type
 * @param <L> careful loop item handler
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public class CarelessLoopItemHandler<T, E extends Exception, L extends FragileLoopItemHandler<T, E>>
  implements LoopItemHandler<T>
{
  private final L wrapped;
  private E exception;

  /**
   * Constructor.
   * @param wrapped wrapped careless handler
   */
  public CarelessLoopItemHandler(@NotNull L wrapped)
  {
    this.wrapped = wrapped;
  }

  /**
   * Handle an item.
   *
   * @param item item to handle
   * @return <code>true</code> go on with loop<br>
   *         <code>false</code> break loop
   */
  @Override
  @SuppressWarnings("unchecked")
  public boolean handle(T item)
  {
    try {
      return wrapped.handle(item);
    } catch (Exception e) {
      exception = (E)e;
      return false;
    }
  }

  /**
   * Called before the loop is started.
   */
  @Override
  public void beforeStart()
  {
    wrapped.beforeStart();
    exception = null;
  }

  /**
   * Called after the loop has finished.
   */
  @Override
  public void afterFinish()
  {
    wrapped.afterFinish();
  }

  /**
   * Get the wrapped careful loop item handler.
   * @return wrapped handler
   */
  @NotNull
  public L getWrapped()
  {
    return wrapped;
  }

  /**
   * Get the exception thrown during the last loop.
   * @return exception or <code>null</code> if there was no exception thrown
   */
  @Nullable
  public E getException()
  {
    return exception;
  }
}
