// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2014-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================

/**
 * Function support.
 *
 * <h2>General</h2>
 * <ul>
 *   <li>Interfaces with more than one argument provide default methods for partial evaluation.</li>
 *   <li>
 *     Generic functions, procedures (consumers)  and predicates with more than one argument
 *     are also defined as 1-argument equivalents which accept a <a href="../tuple/package-summary.html">tuple</a>.
 *     The tuple is expanded into the expected arguments in order one-by-one. So the tuple's
 *     size has to match the number of parameters, and the tuple's types have to match the parameter types.
 *   </li>
 * </ul>
 *
 * <h2>Simple Functions</h2>
 * <p>
 *   Functions here take up to 9 arguments, and return one result.
 *   For historical reasons (this lib started earlier) the order of type arguments
 *   is different from standard Java {@code Function} and {@code BiFunction}. 
 *   While Java orders the types like lambda arguments ({@code <P1, P2, R>}):
 * </p>
 * <pre>
 * {@code (P1 arg1, P2 arg2) -> R}
 * </pre>
 * this lib follows the order in function definitions ({@code <R, P1, P2>}}:
 * <pre>
 * {@code R function(P1 arg1, P2 arg2)}  
 * </pre>
 * <p>
 * Some of the functions are available in standard Java, and implementations here extend the
 * standard interfaces nowadays, but usually come with a bit of benefit. But especially they allow
 * more than 2 arguments.
 * </p>
 * <p>
 *   Functions with 2 or more arguments can be chained after any function returning a
 *   tuple of the correct size and matching elements.
 *   See eg {@link de.caff.generics.function.Function2#after3(de.caff.generics.function.Function3)}.
 *   Having the analogue to {@link de.caff.generics.function.Function2#andThen(java.util.function.Function)}
 *   is not possible due to the restrictions of the Java generic type system.
 * </p>
 * <ul>
 *   <li>{@link de.caff.generics.function.Function0}: function without argumen like {@code java.util.function.Supplier}.</li>
 *   <li>{@link de.caff.generics.function.Function1}: function with one argument like {@code java.util.function.Function}.</li>
 *   <li>{@link de.caff.generics.function.Function2}: function with two arguments like {@code java.util.function.BiFunction}.</li>
 *   <li>{@link de.caff.generics.function.Function3}: function with three arguments.</li>
 *   <li>{@link de.caff.generics.function.Function4}: function with four arguments.</li>
 *   <li>{@link de.caff.generics.function.Function5}: function with five arguments.</li>
 *   <li>{@link de.caff.generics.function.Function6}: function with six arguments.</li>
 *   <li>{@link de.caff.generics.function.Function7}: function with seven arguments.</li>
 *   <li>{@link de.caff.generics.function.Function8}: function with eight arguments.</li>
 *   <li>{@link de.caff.generics.function.Function9}: function with nine arguments.</li>
 * </ul>
 * 
 * <h2>Procedures</h2>
 * <p>
 *   Procedures are functions with no return values, so they are quite similar to functions:
 * </p>
 * <p>
 *   As functions they can be chain after functions which return tuples of the correct size.
 * </p>
 * <ul>
 *   <li>{@link de.caff.generics.function.Procedure0}: procedure without argument.</li>
 *   <li>{@link de.caff.generics.function.Procedure1}: procedure with one argument.</li>
 *   <li>{@link de.caff.generics.function.Procedure2}: procedure with two arguments.</li>
 *   <li>{@link de.caff.generics.function.Procedure3}: procedure with three arguments.</li>
 *   <li>{@link de.caff.generics.function.Procedure4}: procedure with four arguments.</li>
 *   <li>{@link de.caff.generics.function.Procedure5}: procedure with five arguments.</li>
 *   <li>{@link de.caff.generics.function.Procedure6}: procedure with six arguments.</li>
 *   <li>{@link de.caff.generics.function.Procedure7}: procedure with seven arguments.</li>
 *   <li>{@link de.caff.generics.function.Procedure8}: procedure with eight arguments.</li>
 *   <li>{@link de.caff.generics.function.Procedure9}: procedure with nine arguments.</li>
 * </ul>
 *
 * <h2>Predicates</h2>
 * <p>
 *   Predicates are just functions which return a {@code boolean}. Basically a function 
 *   returning a {@code java.lang.Boolean} is quite the same, but for one thing the predicates
 *   avoid boxing/unboxing, and come with special methods allowing easy boolean combination.
 * </p>
 * <p>
 *   As functions they can be chain after functions which return tuples of the correct size.
 * </p>
 * <ul>
 *   <li>{@link de.caff.generics.function.Predicate1}: predicate with one argument.</li>
 *   <li>{@link de.caff.generics.function.Predicate2}: predicate with one arguments.</li>
 *   <li>{@link de.caff.generics.function.Predicate3}: predicate with three arguments.</li>
 *   <li>{@link de.caff.generics.function.Predicate4}: predicate with four arguments.</li>
 *   <li>{@link de.caff.generics.function.Predicate5}: predicate with five arguments.</li>
 *   <li>{@link de.caff.generics.function.Predicate6}: predicate with six arguments.</li>
 *   <li>{@link de.caff.generics.function.Predicate7}: predicate with seven arguments.</li>
 *   <li>{@link de.caff.generics.function.Predicate8}: predicate with eight arguments.</li>
 *   <li>{@link de.caff.generics.function.Predicate9}: predicate with nine arguments.</li>
 * </ul>
 *
 * <h2>Fragile functions</h2>
 * Fragile functions are like standard functions, but can throw exceptions.
 * This concept is a bit fragile itself, so not supported as well as the above.
 * <ul>
 *   <li>{@link de.caff.generics.function.FragileFunction0}: fragile function without arguments.</li>
 *   <li>{@link de.caff.generics.function.FragileFunction1}: fragile function with one arguments.</li>
 * </ul>
 *
 * <h2>Operators, Functions and Consumers for Primitive Types</h2>
 * <p>
 *   Some of these are lacking in the standard lib, although they are useful to avoid unnecessary boxing/unboxing.
 * </p>
 * <p>
 *   Primitive types are divided in
 * </p>
 * <ul>
 *   <li><b>Consumers</b>: 1-argument procedure accepting the type, standard Java has thes only for {@code int}, {@code long}, and {@code double}.</li>
 *   <li>
 *     <b>Functions</b>: these accept 2 to 3 arguments of the given type, and return a non-primitive type.
 *     The 1-argument case is often already contained in the standard Java library, if not it is also provided here.
 *   </li>
 *   <li><b>Operators</b>: these are functions which accept 0 to 3  arguments of the given type and return the same type.</li>
 *   <li><b>Predicates</b>: these are functions which accept 1 or 2 arguments of the given type and return a {@code boolean}.</li>
 * </ul>
 * <table border="1" summary="Functional Interfaces for Primitive Types">
 *   <tr>
 *     <th>Interface</th>
 *     <th>{@code boolean}</th>
 *     <th>{@code byte}</th>
 *     <th>{@code char}</th>
 *     <th>{@code double}</th>
 *     <th>{@code float}</th>
 *     <th>{@code int}</th>
 *     <th>{@code long}</th>
 *     <th>{@code short}</th>
 *   </tr>
 *   <tr>
 *     <td>Consumer</td>
 *     <td>{@link de.caff.generics.function.BooleanConsumer BooleanConsumer}</td>
 *     <td>{@link de.caff.generics.function.ByteConsumer ByteConsumer}</td>
 *     <td>{@link de.caff.generics.function.CharConsumer CharConsumer}</td>
 *     <td>{@code java.util.function.DoubleConsumer}</td>
 *     <td>{@link de.caff.generics.function.FloatConsumer BooleanConsumer}</td>
 *     <td>{@code java.util.function.IntConsumer}</td>
 *     <td>{@code java.util.function.LongConsumer}</td>
 *     <td>{@link de.caff.generics.function.ShortConsumer ShortConsumer}</td>
 *   </tr>
 *   <tr>
 *     <td>Supplier (0-argument Operator)</td>
 *     <td>{@link de.caff.generics.function.BooleanOperator0 BooleanOperator0}</td>
 *     <td>{@link de.caff.generics.function.ByteOperator0 ByteOperator0}</td>
 *     <td>{@link de.caff.generics.function.CharOperator0 CharOperator0}</td>
 *     <td>{@code java.util.function.DoubleSupplier}</td>
 *     <td>{@link de.caff.generics.function.FloatOperator0 FloatOperator0}</td>
 *     <td>{@code java.util.function.IntSupplier}</td>
 *     <td>{@code java.util.function.LongSupplier}</td>
 *     <td>{@link de.caff.generics.function.ShortOperator0 ShortOperator0}</td>
 *   </tr>
 *   <tr>
 *     <td>Unary (1-argument) Operator</td>
 *     <td>{@link de.caff.generics.function.BooleanOperator1 BooleanOperator1}</td>
 *     <td>{@link de.caff.generics.function.ByteOperator1 ByteOperator1}</td>
 *     <td>{@link de.caff.generics.function.CharOperator1 CharOperator1}</td>
 *     <td>{@code java.util.function.DoubleUnaryOperator}</td>
 *     <td>{@link de.caff.generics.function.FloatOperator1 FloatOperator1}</td>
 *     <td>{@code java.util.function.IntUnaryOperator}</td>
 *     <td>{@code java.util.function.LongUnaryOperator}</td>
 *     <td>{@link de.caff.generics.function.ShortOperator1 ShortOperator1}</td>
 *   </tr>
 *   <tr>
 *     <td>Binary (2-argument) Operator</td>
 *     <td>{@link de.caff.generics.function.BooleanOperator2 BooleanOperator2}</td>
 *     <td>{@link de.caff.generics.function.ByteOperator2 ByteOperator2}</td>
 *     <td>{@link de.caff.generics.function.CharOperator2 CharOperator2}</td>
 *     <td>{@code java.util.function.DoubleBinaryOperator}</td>
 *     <td>{@link de.caff.generics.function.FloatOperator2 FloatOperator2}</td>
 *     <td>{@code java.util.function.IntBinaryOperator}</td>
 *     <td>{@code java.util.function.LongBinaryOperator}</td>
 *     <td>{@link de.caff.generics.function.ShortOperator2 ShortOperator2}</td>
 *   </tr>
 *   <tr>
 *     <td>Ternary (3-argument) Operator</td>
 *     <td>{@link de.caff.generics.function.BooleanOperator3 BooleanOperator3}</td>
 *     <td>{@link de.caff.generics.function.ByteOperator3 ByteOperator3}</td>
 *     <td>{@link de.caff.generics.function.CharOperator3 CharOperator3}</td>
 *     <td>{@link de.caff.generics.function.DoubleOperator3 DoubleOperator3}</td>
 *     <td>{@link de.caff.generics.function.FloatOperator3 FloatOperator3}</td>
 *     <td>{@link de.caff.generics.function.IntOperator3 IntOperator3}</td>
 *     <td>{@link de.caff.generics.function.LongOperator3 LongOperator3}</td>
 *     <td>{@link de.caff.generics.function.ShortOperator3 ShortOperator3}</td>
 *   </tr>
 *   <tr>
 *     <td>Variable Arguments Operator</td>
 *     <td>{@link de.caff.generics.function.VarBooleanOperator VarBooleanOperator}</td>
 *     <td>{@link de.caff.generics.function.VarByteOperator VarByteOperator}</td>
 *     <td>{@link de.caff.generics.function.VarCharOperator VarCharOperator}</td>
 *     <td>{@link de.caff.generics.function.VarDoubleOperator VarDoubleOperator}</td>
 *     <td>{@link de.caff.generics.function.VarFloatOperator VarFloatOperator}</td>
 *     <td>{@link de.caff.generics.function.VarIntOperator VarIntOperator}</td>
 *     <td>{@link de.caff.generics.function.VarLongOperator VarLongOperator}</td>
 *     <td>{@link de.caff.generics.function.VarShortOperator VarShortOperator}</td>
 *   </tr>
 *   <tr>
 *     <td>1-argument Predicate</td>
 *     <td>{@link de.caff.generics.function.BooleanPredicate1 BooleanPredicate1}</td>
 *     <td>{@link de.caff.generics.function.BytePredicate1 BytePredicate1}</td>
 *     <td>{@link de.caff.generics.function.CharPredicate1 CharPredicate1}</td>
 *     <td>{@link de.caff.generics.function.DoublePredicate1 CharPredicate1}</td>
 *     <td>{@link de.caff.generics.function.FloatPredicate1 CharPredicate1}</td>
 *     <td>{@link de.caff.generics.function.IntPredicate1 CharPredicate1}</td>
 *     <td>{@link de.caff.generics.function.LongPredicate1 CharPredicate1}</td>
 *     <td>{@link de.caff.generics.function.ShortPredicate1 CharPredicate1}</td>
 *   </tr>
 *   <tr>
 *     <td>2-argument Predicate</td>
 *     <td>{@link de.caff.generics.function.BooleanPredicate2 BooleanPredicate2}</td>
 *     <td>{@link de.caff.generics.function.BytePredicate2 BytePredicate2}</td>
 *     <td>{@link de.caff.generics.function.CharPredicate2 CharPredicate2}</td>
 *     <td>{@link de.caff.generics.function.DoublePredicate2 CharPredicate2}</td>
 *     <td>{@link de.caff.generics.function.FloatPredicate2 CharPredicate2}</td>
 *     <td>{@link de.caff.generics.function.IntPredicate2 CharPredicate2}</td>
 *     <td>{@link de.caff.generics.function.LongPredicate2 CharPredicate2}</td>
 *     <td>{@link de.caff.generics.function.ShortPredicate2 CharPredicate2}</td>
 *   </tr>
 *   <tr>
 *     <td>3-argument Predicate</td>
 *     <td>{@link de.caff.generics.function.BooleanPredicate3 BooleanPredicate3}</td>
 *     <td>{@link de.caff.generics.function.BytePredicate3 BytePredicate3}</td>
 *     <td>{@link de.caff.generics.function.CharPredicate3 CharPredicate3}</td>
 *     <td>{@link de.caff.generics.function.DoublePredicate3 CharPredicate3}</td>
 *     <td>{@link de.caff.generics.function.FloatPredicate3 CharPredicate3}</td>
 *     <td>{@link de.caff.generics.function.IntPredicate3 CharPredicate3}</td>
 *     <td>{@link de.caff.generics.function.LongPredicate3 CharPredicate3}</td>
 *     <td>{@link de.caff.generics.function.ShortPredicate3 CharPredicate3}</td>
 *   </tr>
 * </table>
 */
package de.caff.generics.function;
