// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2020-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.function;

import de.caff.annotation.NotNull;
import de.caff.generics.IntIndexable;

import java.util.function.IntBinaryOperator;

/**
 * Primitive integer operator with a variable number of arguments.
 * <p>
 * Although this basically works for a variable int array
 * via the {@link #apply(int...)} method implementors
 * have to implement the method {@link #apply(IntIndexable)}
 * with a {@link IntIndexable} argument. This is done
 * to make this class generally more useful without the need
 * to copy arrays.
 * </p>
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since November 03, 2020
 */
@FunctionalInterface
public interface VarIntOperator
{
  /**
   * Apply this operator.
   * This default method forwards execution to {@link #apply(IntIndexable)}.
   *
   * @param values arguments on which this operator is applied
   * @return result of applying this operator to all arguments
   */
  default int apply(int ... values)
  {
    return apply(IntIndexable.viewArray(values));
  }

  /**
   * Apply this operator.
   * @param values indexable providing the values on which this operator operates
   * @return result of applying this operator to all values
   */
  int apply(@NotNull IntIndexable values);

  /**
   * Create an variable args operator from applying a binary operator
   * over and over.
   * <p>
   * Eg
   * <blockquote><pre>
   * VarIntOperator maxOp = VarIntOperator.fromBinary(Math::max, Integer.MIN_VALUE);
   * int max = maxOp.apply(1, -7, -13, 42);
   * </pre></blockquote>
   *
   * On each application the currently accumulated result becomes the left (first)
   * argument of {@code binaryOp}, while the run value becomes the right (second).
   * @param binaryOp     binary operator
   * @param startValue   start value (result of operator if no values are given)
   * @return variable int argument operator which applies the given binary operator over and over
   */
  @NotNull
  static VarIntOperator fromBinary(@NotNull IntBinaryOperator binaryOp,
                                   int startValue)
  {
    return values -> {
      int result = startValue;
      for (int v : values) {
        result = binaryOp.applyAsInt(result, v);
      }
      return result;
    };
  }
}
