// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright ©2023-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.function;

import de.caff.annotation.NotNull;
import de.caff.generics.Order;

import java.io.Serializable;
import java.util.Comparator;

/**
 * Define an order for two primitive short values.
 * <p>
 * Compared to a  {@code Comparator<Short>} this can
 * avoid boxing and unboxing in various situations.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since January 23, 2023
 */
@FunctionalInterface
public interface ShortOrdering
{
  /**
   * Natural order of ascending {@code short} values.
   */
  ShortOrdering ASCENDING = new ShortOrdering.Serial()
  {
    private static final long serialVersionUID = -7646031177018732313L;

    @NotNull
    @Override
    public Order checkShort(short v1, short v2)
    {
      return (v1 < v2)
              ? Order.Ascending
              : (v1 > v2
                         ? Order.Descending
                         : Order.Same);
    }

    @NotNull
    @Override
    public ShortOrdering inverse()
    {
      return ShortOrdering.DESCENDING;
    }

    @NotNull
    @Override
    public Comparator<Short> asShortComparator()
    {
      return Short::compare;
    }
  };
  /** Inverse natural order of {@code short} values. */
  ShortOrdering DESCENDING = new ShortOrdering.Serial()
  {
    private static final long serialVersionUID = 3026851066329505442L;

    @NotNull
    @Override
    public Order checkShort(short v2, short v1)
    {
      return (v1 < v2)
              ? Order.Ascending
              : (v1 > v2
                         ? Order.Descending
                         : Order.Same);
    }

    @NotNull
    @Override
    public ShortOrdering inverse()
    {
      return ShortOrdering.ASCENDING;
    }

    @NotNull
    @Override
    public Comparator<Short> asShortComparator()
    {
      return (Comparator<Short> & Serializable) (v1, v2) -> Short.compare(v2, v1);
    }
  };

  /**
   * Natural order of ascending unsigned {@code short} values.
   * This handles short values as if they are unsigned and not signed.
   */
  ShortOrdering UNSIGNED_ASCENDING = new ShortOrdering.Serial()
  {
    private static final long serialVersionUID = -3495520630473840505L;

    @NotNull
    @Override
    public Order checkShort(short v1, short v2)
    {
      return IntOrdering.ASCENDING.checkInt(v1 & 0xFFFF,
                                            v2 & 0xFFFF);
    }

    @NotNull
    @Override
    public ShortOrdering inverse()
    {
      return ShortOrdering.UNSIGNED_DESCENDING;
    }

    @NotNull
    @Override
    public Comparator<Short> asShortComparator()
    {
      return Comparator.comparingInt(v -> v & 0xFFFF);
    }
  };

  /**
   * Natural order of ascending unsigned {@code short} values.
   * This handles short values as if they are unsigned and not signed.
   */
  ShortOrdering UNSIGNED_DESCENDING = new ShortOrdering.Serial()
  {
    private static final long serialVersionUID = 8393916740777317119L;

    @NotNull
    @Override
    public Order checkShort(short v1, short v2)
    {
      return IntOrdering.DESCENDING.checkInt(v1 & 0xFFFF,
                                             v2 & 0xFFFF);
    }

    @NotNull
    @Override
    public ShortOrdering inverse()
    {
      return ShortOrdering.UNSIGNED_ASCENDING;
    }

    @NotNull
    @Override
    public Comparator<Short> asShortComparator()
    {
      return (Comparator<Short> & Serializable) (v1, v2) -> Integer.compareUnsigned(v2, v1);
    }
  };

  /**
   * Get the order of the two values.
   * @param v1 first value
   * @param v2 second value
   * @return order of the two values
   */
  @NotNull
  Order checkShort(short v1, short v2);

  /**
   * Are the two values represented in strictly ascending order?
   * @param v1 first value
   * @param v2 second value
   * @return {@code true} if {@code v1 < v2} according to this ordering<br>
   *         {@code false} if {@code v1 >= v2} according to this ordering
   * @see #descending(short, short)
   * @see #ascendingOrSame(short, short)
   * @see #descendingOrSame(short, short)
   * @see #same(short, short)
   * @see #different(short, short)
   */
  default boolean ascending(short v1, short v2)
  {
    return checkShort(v1, v2).ascending;
  }

  /**
   * Are the two values represented in ascending order?
   * @param v1 first value
   * @param v2 second value
   * @return {@code true} if {@code v1 <= v2} according to this ordering<br>
   *         {@code false} if {@code v1 > v2} according to this ordering
   * @see #ascending(short, short)
   * @see #descending(short, short)
   * @see #descendingOrSame(short, short)
   * @see #same(short, short)
   * @see #different(short, short)
   */
  default boolean ascendingOrSame(short v1, short v2)
  {
    return checkShort(v1, v2).ascendingOrSame;
  }

  /**
   * Are the two values represented in strictly descending order?
   * @param v1 first value
   * @param v2 second value
   * @return {@code true} if {@code v1 > v2} according to this ordering<br>
   *         {@code false} if {@code v1 <= v2} according to this ordering
   * @see #ascending(short, short)
   * @see #ascendingOrSame(short, short)
   * @see #descendingOrSame(short, short)
   * @see #same(short, short)
   * @see #different(short, short)
   */
  default boolean descending(short v1, short v2)
  {
    return checkShort(v1, v2).descending;
  }

  /**
   * Are the two values represented in descending order?
   * @param v1 first value
   * @param v2 second value
   * @return {@code true} if {@code v1 >= v2} according to this ordering<br>
   *         {@code false} if {@code v1 < v2} according to this ordering
   * @see #ascending(short, short)
   * @see #descending(short, short)
   * @see #ascendingOrSame(short, short)
   * @see #same(short, short)
   * @see #different(short, short)
   */
  default boolean descendingOrSame(short v1, short v2)
  {
    return checkShort(v1, v2).descendingOrSame;
  }

  /**
   * Are the two values considered equal by this order?
   * @param v1 first value
   * @param v2 second value
   * @return {@code true} if {@code v1 == v2} according to this order<br>
   *         {@code false} if {@code v1 != v2} according to this ordering
   * @see #ascending(short, short)
   * @see #descending(short, short)
   * @see #ascendingOrSame(short, short)
   * @see #descendingOrSame(short, short)
   * @see #different(short, short)
   */
  default boolean same(short v1, short v2)
  {
    return checkShort(v1, v2).same;
  }

  /**
   * Are the two values considered different by this order?
   * @param v1 first value
   * @param v2 second value
   * @return {@code true} if {@code v1 != v2} according to this order<br>
   *         {@code false} if {@code v1 == v2} according to this ordering
   * @see #ascending(short, short)
   * @see #descending(short, short)
   * @see #ascendingOrSame(short, short)
   * @see #descendingOrSame(short, short)
   * @see #same(short, short)
   */
  default boolean different(short v1, short v2)
  {
    return checkShort(v1, v2).different;
  }

  /**
   * Invert this order.
   * @return inverse order
   */
  @NotNull
  default ShortOrdering inverse()
  {
    return new ShortOrdering.Serial()
    {
      private static final long serialVersionUID = 3270504639153727362L;

      @Override
      @NotNull
      public Order checkShort(short v1, short v2)
      {
        return ShortOrdering.this.checkShort(v2, v1);
      }

      @NotNull
      @Override
      public ShortOrdering inverse()
      {
        return ShortOrdering.this;
      }
    };
  }

  /**
   * Use this ordering as a comparator.
   * @return comparator of {@code Short}
   */
  @NotNull
  default Comparator<Short> asShortComparator()
  {
    return (Comparator<Short> & Serializable) (v1, v2) -> checkShort(v1, v2).comparison;
  }

  /**
   * View this ordering as a number comparator.
   * @return comparator of {@link Number}
   */
  @NotNull
  default Comparator<Number> asNumberComparator()
  {
    return (Comparator<Number> & Serializable)(v1, v2) -> checkShort(v1.shortValue(), v2.shortValue()).comparison;
  }

  /**
   * Get this ordering as a generic short ordering. 
   * @return generic ordering for shorts
   */
  @NotNull
  default Ordering<Short> asShortOrdering()
  {
    return (Ordering<Short> & Serializable)this::checkShort;
  }

  /**
   * Convert a standard comparator of {@code Double} into a double ordering.
   * @param comparator comparator of {@code Double}
   * @return ordering
   */
  @NotNull
  static ShortOrdering fromComparator(@NotNull Comparator<Short> comparator)
  {
    return new ShortOrdering.Serial()
    {
      private static final long serialVersionUID = 393576352771473074L;

      @NotNull
      @Override
      public Order checkShort(short v1, short v2)
      {
        return Order.fromCompare(comparator.compare(v1, v2));
      }

      @NotNull
      @Override
      public Comparator<Short> asShortComparator()
      {
        return comparator;
      }
    };
  }

  /**
   * A short ordering which is serializable.
   * It is a recommendation that comparators implement {@code java.io.Serializable}
   * to allow serializing {@code TreeMap}s and similar. If an Ordering is expected to be used
   * as a comparator in such areas implement either this interface, or for lambdas
   * use a cast to force the compiler to make the order serializable:
   * <pre>{@code
   *   return (ShortOrdering & Serializable) (v1, v2) -> v1.compareTo(v2);
   * }</pre>
   * You can also use a {@code (ShortOrdering.Serial)} cast to save a bit of typing,
   * but the above is a nice trick to have in your toolbox.
   */
  interface Serial extends ShortOrdering,
                           java.io.Serializable
  {
  }
}
