// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.function;

import de.caff.annotation.NotNull;
import de.caff.generics.WrappedFragileException;
import de.caff.generics.tuple.ITuple4;

import java.util.function.BiFunction;
import java.util.function.Function;

/**
 * Function without return and with four parameters.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @param <P1> first parameter type
 * @param <P2> second parameter type
 * @param <P3> third parameter type
 * @param <P4> fourth parameter type
 * @see Procedure0
 * @see Procedure1
 * @see Procedure2
 * @see Procedure3
 * @see Procedure5
 * @see Procedure6
 * @see Procedure7
 * @see Procedure8
 * @see Procedure9
 */
@FunctionalInterface
public interface Procedure4<P1, P2, P3, P4>
        extends Procedure1<ITuple4<? extends P1, ? extends P2, ? extends P3,
                                   ? extends P4>>
{
  /**
   * Procedure with four arguments which does nothing.
   * Use {@link #empty()} instead.
   */
  @NotNull
  Procedure4<?, ?, ?, ?> NULL = (a, b, c, d) -> {};

  /**
   * Run the procedure.
   * @param arg1 first procedure argument
   * @param arg2 second procedure argument
   * @param arg3 third procedure argument
   * @param arg4 fourth procedure argument
   */
  void apply(P1 arg1, P2 arg2, P3 arg3, P4 arg4);

  /**
   * Expand a tuple into the arguments of this procedure.
   * @param tuple4 tuple to expand
   */
  @Override
  default void accept(@NotNull ITuple4<? extends P1, ? extends P2, ? extends P3, ? extends P4> tuple4)
  {
    tuple4.sendTo(this);
  }

  /**
   * Get a partially applied procedure.
   * @param argument value applied for first argument
   * @return partially evaluated procedure
   */
  @NotNull
  default Procedure3<P2, P3, P4> partial1(P1 argument)
  {
    return (arg2, arg3, arg4) -> apply(argument, arg2, arg3, arg4);
  }

  /**
   * Get a partially applied procedure.
   * @param argument value applied for second argument
   * @return partially evaluated procedure
   */
  @NotNull
  default Procedure3<P1, P3, P4> partial2(P2 argument)
  {
    return (arg1, arg3, arg4) -> apply(arg1, argument, arg3, arg4);
  }

  /**
   * Get a partially applied procedure.
   * @param argument value applied for third argument
   * @return partially evaluated procedure
   */
  @NotNull
  default Procedure3<P1, P2, P4> partial3(P3 argument)
  {
    return (arg1, arg2, arg4) -> apply(arg1, arg2, argument, arg4);
  }

  /**
   * Get a partially applied procedure.
   * @param argument value applied for fourth argument
   * @return partially evaluated procedure
   */
  @NotNull
  default Procedure3<P1, P2, P3> partial4(P4 argument)
  {
    return (arg1, arg2, arg3) -> apply(arg1, arg2, arg3, argument);
  }

  /**
   * Get a chained procedure which first applies the given function, then this procedure.
   * The given function has to return a {@link ITuple4} with matching parametric types
   * which are applied in their order to this procedure.
   * @param before function to apply first, has to return a non-{@code null} result so unpacking the parameters
   *               from the tuple is possible
   * @param <FR> function {@code before}'s return type, has to be a 4-tuple with matching types
   * @param <FP> function {@code before}'s parameter type, same for returned procedure
   * @return chain of function and this procedure
   */
  @NotNull
  default <FR extends ITuple4<? extends P1, ? extends P2, ? extends P3, ? extends P4>, FP>
  Procedure1<FP> after1(@NotNull Function<FP, FR> before)
  {
    return fp -> before.apply(fp).sendTo(this);
  }

  /**
   * Get a chained procedure which first applies the given 2-parameter function, then this procedure.
   * The given function has to return a {@link ITuple4} with matching parametric types
   * which are applied in their order to this procedure.
   * @param before function to apply first, has to return a non-{@code null} result so unpacking the parameters
   *               from the tuple is possible
   * @param <FR> function {@code before}'s return type, has to be a 4-tuple with matching types
   * @param <FP1> function {@code before}'s first parameter type, sane for returned procedure
   * @param <FP2> function {@code before}'s second parameter type, same for returned procedure
   * @return chain of the given function and this procedure
   */
  @NotNull
  default <FR extends ITuple4<? extends P1, ? extends P2, ? extends P3, ? extends P4>, FP1, FP2>
  Procedure2<FP1, FP2> after2(@NotNull BiFunction<FP1, FP2, FR> before)
  {
    return (fp1, fp2) -> before.apply(fp1, fp2).sendTo(this);
  }

  /**
   * Get a chained procedure which first applies the given 3-parameter function, then this procedure.
   * The given function has to return a {@link ITuple4} with matching parametric types
   * which are applied in their order to this procedure.
   * @param before function to apply first, has to return a non-{@code null} result so unpacking the parameters
   *               from the tuple is possible
   * @param <FR> function {@code before}'s return type, has to be a 4-tuple with matching types
   * @param <FP1> function {@code before}'s first parameter type, sane for returned procedure
   * @param <FP2> function {@code before}'s second parameter type, sane for returned procedure
   * @param <FP3> function {@code before}'s third parameter type, sane for returned procedure
   * @return chain of the given function and this procedure
   */
  @NotNull
  default <FR extends ITuple4<? extends P1, ? extends P2, ? extends P3, ? extends P4>, FP1, FP2, FP3>
  Procedure3<FP1, FP2, FP3> after3(@NotNull Function3<FR, FP1, FP2, FP3> before)
  {
    return (fp1, fp2, fp3) -> before.apply(fp1, fp2, fp3).sendTo(this);
  }

  /**
   * Get a chained procedure which first applies the given 5-parameter function, then this procedure.
   * The given function has to return a {@link ITuple4} with matching parametric types
   * which are applied in their order to this function.
   * @param before function to apply first, has to return a non-{@code null} result so unpacking the parameters
   *               from the tuple is possible
   * @param <FR> function {@code before}'s return type, has to be a 4-tuple with matching types
   * @param <FP1> function {@code before}'s first parameter type, sane for returned procedure
   * @param <FP2> function {@code before}'s second parameter type, sane for returned procedure
   * @param <FP3> function {@code before}'s third parameter type, sane for returned procedure
   * @param <FP4> function {@code before}'s fourth parameter type, sane for returned procedure
   * @param <FP5> function {@code before}'s fifth parameter type, sane for returned procedure
   * @return chain of the given function and this procedure
   */
  @NotNull
  default <FR extends ITuple4<? extends P1, ? extends P2, ? extends P3, ? extends P4>, FP1, FP2, FP3, FP4, FP5>
  Procedure5<FP1, FP2, FP3, FP4, FP5> after5(@NotNull Function5<FR, FP1, FP2, FP3, FP4, FP5> before)
  {
    return (fp1, fp2, fp3, fp4, fp5) -> before.apply(fp1, fp2, fp3, fp4, fp5).sendTo(this);
  }

  /**
   * Get a chained procedure which first applies the given 6-parameter function, then this procedure.
   * The given function has to return a {@link ITuple4} with matching parametric types
   * which are applied in their order to this function.
   * @param before function to apply first, has to return a non-{@code null} result so unpacking the parameters
   *               from the tuple is possible
   * @param <FR> function {@code before}'s return type, has to be a 4-tuple with matching types
   * @param <FP1> function {@code before}'s first parameter type, sane for returned procedure
   * @param <FP2> function {@code before}'s second parameter type, sane for returned procedure
   * @param <FP3> function {@code before}'s third parameter type, sane for returned procedure
   * @param <FP4> function {@code before}'s fourth parameter type, sane for returned procedure
   * @param <FP5> function {@code before}'s fifth parameter type, sane for returned procedure
   * @param <FP6> function {@code before}'s sixth parameter type, sane for returned procedure
   * @return chain of the given function and this procedure
   */
  @NotNull
  default <FR extends ITuple4<? extends P1, ? extends P2, ? extends P3, ? extends P4>, FP1, FP2, FP3, FP4, FP5, FP6>
  Procedure6<FP1, FP2, FP3, FP4, FP5, FP6> after6(@NotNull Function6<FR, FP1, FP2, FP3, FP4, FP5, FP6> before)
  {
    return (fp1, fp2, fp3, fp4, fp5, fp6) -> before.apply(fp1, fp2, fp3, fp4, fp5, fp6).sendTo(this);
  }

  /**
   * Get a chained procedure which first applies the given 7-parameter function, then this one.
   * The given function has to return a {@link ITuple4} with matching parametric types
   * which are applied in their order to this procedure.
   * @param before function to apply first, has to return a non-{@code null} result so unpacking the parameters
   *               from the tuple is possible
   * @param <FR> function {@code before}'s return type, has to be a 4-tuple with matching types
   * @param <FP1> function {@code before}'s first parameter type, sane for returned procedure
   * @param <FP2> function {@code before}'s second parameter type, sane for returned procedure
   * @param <FP3> function {@code before}'s third parameter type, sane for returned procedure
   * @param <FP4> function {@code before}'s fourth parameter type, sane for returned procedure
   * @param <FP5> function {@code before}'s fifth parameter type, sane for returned procedure
   * @param <FP6> function {@code before}'s sixth parameter type, sane for returned procedure
   * @param <FP7> function {@code before}'s seventh parameter type, sane for returned procedure
   * @return chain of the given function and this procedure
   */
  @NotNull
  default <FR extends ITuple4<? extends P1, ? extends P2, ? extends P3, ? extends P4>, FP1, FP2, FP3, FP4, FP5, FP6, FP7>
  Procedure7<FP1, FP2, FP3, FP4, FP5, FP6, FP7> after7(@NotNull Function7<FR, FP1, FP2, FP3, FP4, FP5, FP6, FP7> before)
  {
    return (fp1, fp2, fp3, fp4, fp5, fp6, fp7) -> before.apply(fp1, fp2, fp3, fp4, fp5, fp6, fp7).sendTo(this);
  }

  /**
   * Get a chained procedure which first applies the given 8-parameter function, then this one.
   * The given function has to return a {@link ITuple4} with matching parametric types
   * which are applied in their order to this procedure.
   * @param before function to apply first, has to return a non-{@code null} result so unpacking the parameters
   *               from the tuple is possible
   * @param <FR> function {@code before}'s return type, has to be a 4-tuple with matching types
   * @param <FP1> function {@code before}'s first parameter type, sane for returned procedure
   * @param <FP2> function {@code before}'s second parameter type, sane for returned procedure
   * @param <FP3> function {@code before}'s third parameter type, sane for returned procedure
   * @param <FP4> function {@code before}'s fourth parameter type, sane for returned procedure
   * @param <FP5> function {@code before}'s fifth parameter type, sane for returned procedure
   * @param <FP6> function {@code before}'s sixth parameter type, sane for returned procedure
   * @param <FP7> function {@code before}'s seventh parameter type, sane for returned procedure
   * @param <FP8> function {@code before}'s seventh parameter type, sane for returned procedure
   * @return chain of the given function and this procedure
   */
  @NotNull
  default <FR extends ITuple4<? extends P1, ? extends P2, ? extends P3, ? extends P4>, FP1, FP2, FP3, FP4, FP5, FP6, FP7, FP8>
  Procedure8<FP1, FP2, FP3, FP4, FP5, FP6, FP7, FP8> after8(@NotNull Function8<FR, FP1, FP2, FP3, FP4, FP5, FP6, FP7, FP8> before)
  {
    return (fp1, fp2, fp3, fp4, fp5, fp6, fp7, fp8) -> before.apply(fp1, fp2, fp3, fp4, fp5, fp6, fp7, fp8).sendTo(this);
  }

  /**
   * Get a chained procedure which first applies the given 9-parameter function, then this one.
   * The given function has to return a {@link ITuple4} with matching parametric types
   * which are applied in their order to this procedure.
   * @param before function to apply first, has to return a non-{@code null} result so unpacking the parameters
   *               from the tuple is possible
   * @param <FR> function {@code before}'s return type, has to be a 4-tuple with matching types
   * @param <FP1> function {@code before}'s first parameter type, sane for returned procedure
   * @param <FP2> function {@code before}'s second parameter type, sane for returned procedure
   * @param <FP3> function {@code before}'s third parameter type, sane for returned procedure
   * @param <FP4> function {@code before}'s fourth parameter type, sane for returned procedure
   * @param <FP5> function {@code before}'s fifth parameter type, sane for returned procedure
   * @param <FP6> function {@code before}'s sixth parameter type, sane for returned procedure
   * @param <FP7> function {@code before}'s seventh parameter type, sane for returned procedure
   * @param <FP8> function {@code before}'s seventh parameter type, sane for returned procedure
   * @param <FP9> function {@code before}'s ninth parameter type, sane for returned procedure
   * @return chain of the given function and this procedure
   */
  @NotNull
  default <FR extends ITuple4<? extends P1, ? extends P2, ? extends P3, ? extends P4>, FP1, FP2, FP3, FP4, FP5, FP6, FP7, FP8, FP9>
  Procedure9<FP1, FP2, FP3, FP4, FP5, FP6, FP7, FP8, FP9> after9(@NotNull Function9<FR, FP1, FP2, FP3, FP4, FP5, FP6, FP7, FP8, FP9> before)
  {
    return (fp1, fp2, fp3, fp4, fp5, fp6, fp7, fp8, fp9) -> before.apply(fp1, fp2, fp3, fp4, fp5, fp6, fp7, fp8, fp9).sendTo(this);
  }

  /**
   * Return a procedure which does nothing.
   * This just returns {@link #NULL}, but the compiler will accept it smoothly.
   * @param <T1> first argument type
   * @param <T2> second argument type
   * @param <T3> third argument type
   * @param <T4> fourth argument type
   * @return do-nothing procedure
   */
  @NotNull
  @SuppressWarnings("unchecked") // do nothing is always same
  static <T1, T2, T3, T4> Procedure4<T1, T2, T3, T4> empty()
  {
    return (Procedure4<T1, T2, T3, T4>)NULL;
  }

  /**
   * Convert a 4-argument procedure which might throw a checked exception into
   * one which does throw an unchecked exception.
   * The returned function will instead throw an unchecked {@link WrappedFragileException} for any
   * checked exception thrown during {@link #apply(Object)}.
   * @param fragileProc procedure throwing a checked exception
   * @param <E> exception type
   * @param <PP1> type of first parameter
   * @param <PP2> type of second parameter
   * @param <PP3> type of third parameter
   * @param <PP4> type of fourth parameter
   * @return non-fragile procedure
   */
  @NotNull
  static <E extends Exception, PP1, PP2, PP3, PP4>
  Procedure4<PP1, PP2, PP3, PP4> nonFragile(@NotNull FragileProcedure4<E, ? super PP1, ? super PP2, ? super PP3, ? super PP4> fragileProc)
  {
    return (p1, p2, p3, p4) -> {
      try {
        fragileProc.apply(p1, p2, p3, p4);
      } catch (Exception x) {
        throw new WrappedFragileException(x);
      }
    };
  }

  /**
   * Convert a 4-argument procedure which might throw a checked exception into
   * one for which you can decide what happens with the exception.
   * The exception handler allows
   * <ul>
   *   <li>throw an unchecked exception instead (compare {@link #nonFragile(FragileProcedure4)})</li>
   *   <li>do nothing</li>
   *   <li>include logging in one of the above</li>
   *   <li>and more...</li>
   * </ul>
   * @param fragileProc procedure throwing a checked exception
   * @param exceptionHandler exception handler. Its return value will be used as return value of the returned function.
   * @param <E> exception type
   * @param <PP1> type of first parameter
   * @param <PP2> type of second parameter
   * @param <PP3> type of third parameter
   * @param <PP4> type of fourth parameter
   * @return non-fragile function
   */
  @NotNull
  @SuppressWarnings("unchecked") // compiler should take care that only the correct checked exception is used
  static <E extends Exception, PP1, PP2, PP3, PP4>
  Procedure4<PP1, PP2, PP3, PP4> nonFragileX(@NotNull FragileProcedure4<E, ? super PP1, ? super PP2, ? super PP3, ? super PP4> fragileProc,
                                             @NotNull Procedure5<? super E, ? super PP1, ? super PP2, ? super PP3, ? super PP4> exceptionHandler)
  {
    return (p1, p2, p3, p4) -> {
      try {
        fragileProc.apply(p1, p2, p3, p4);
      } catch (Exception x) {
        exceptionHandler.apply((E)x, p1, p2, p3, p4);
      }
    };
  }
}
