// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.function;

import de.caff.annotation.NotNull;
import de.caff.generics.WrappedFragileException;
import de.caff.generics.tuple.ITuple3;

import java.util.function.BiFunction;
import java.util.function.Function;

/**
 * Function without return and with three parameters.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @param <P1> first parameter type
 * @param <P2> second parameter type
 * @param <P3> third parameter type
 * @see Procedure0
 * @see Procedure1
 * @see Procedure2
 * @see Procedure4
 * @see Procedure5
 * @see Procedure6
 * @see Procedure7
 * @see Procedure8
 * @see Procedure9
 */
@FunctionalInterface
public interface Procedure3<P1, P2, P3>
        extends Procedure1<ITuple3<? extends P1, ? extends P2, ? extends P3>>
{
  /**
   * Procedure with three arguments which does nothing.
   * Use {@link #empty()} instead.
   */
  @NotNull
  Procedure3<?, ?, ?> NULL = (a, b, c) -> {};

  /**
   * Run the procedure.
   * @param arg1 first procedure argument
   * @param arg2 second procedure argument
   * @param arg3 third procedure argument
   */
  void apply(P1 arg1, P2 arg2, P3 arg3);

  /**
   * Expand a tuple into the arguments of this procedure.
   * @param tuple3 tuple to expand
   */
  @Override
  default void accept(@NotNull ITuple3<? extends P1, ? extends P2, ? extends P3> tuple3)
  {
    tuple3.sendTo(this);
  }

  /**
   * Get a partially applied procedure.
   * @param argument value applied for first argument
   * @return partially evaluated procedure
   */
  @NotNull
  default Procedure2<P2, P3> partialLeft(P1 argument)
  {
    return (arg2, arg3) -> apply(argument, arg2, arg3);
  }

  /**
   * Get a partially applied procedure.
   * This is the same as {@link #partialLeft(Object)}, but with a name which is more
   * consistent with functions having more arguments.
   * @param argument value applied for first argument
   * @return partially evaluated procedure
   */
  @NotNull
  default Procedure2<P2, P3> partial1(P1 argument)
  {
    return (arg2, arg3) -> apply(argument, arg2, arg3);
  }

  /**
   * Get a partially applied procedure.
   * @param argument value applied for second argument
   * @return partially evaluated procedure
   */
  @NotNull
  default Procedure2<P1, P3> partialMid(P2 argument)
  {
    return (arg1, arg3) -> apply(arg1, argument, arg3);
  }

  /**
   * Get a partially applied procedure.
   * This is the same as {@link #partialMid(Object)}, but with a name which is more
   * consistent with functions having more arguments.
   * @param argument value applied for second argument
   * @return partially evaluated procedure
   */
  @NotNull
  default Procedure2<P1, P3> partial2(P2 argument)
  {
    return (arg1, arg3) -> apply(arg1, argument, arg3);
  }

  /**
   * Get a partially applied procedure.
   * @param argument value applied for third argument
   * @return partially evaluated procedure
   */
  @NotNull
  default Procedure2<P1, P2> partialRight(P3 argument)
  {
    return (arg1, arg2) -> apply(arg1, arg2, argument);
  }

  /**
   * Get a partially applied procedure.
   * This is the same as {@link #partialRight(Object)}, but with a name which is more
   * consistent with functions having more arguments.
   * @param argument value applied for third argument
   * @return partially evaluated procedure
   */
  @NotNull
  default Procedure2<P1, P2> partial3(P3 argument)
  {
    return (arg1, arg2) -> apply(arg1, arg2, argument);
  }

  /**
   * Get a chained procedure which first applies the given function, then this procedure.
   * The given function has to return a {@link ITuple3} with matching parametric types
   * which are applied in their order to this procedure.
   * @param before function to apply first, has to return a non-{@code null} result so unpacking the parameters
   *               from the tuple is possible
   * @param <FR> function {@code before}'s return type, has to be a 3-tuple with matching types
   * @param <FP> function {@code before}'s parameter type, same for returned procedure
   * @return chain of function and this procedure
   */
  @NotNull
  default <FR extends ITuple3<? extends P1, ? extends P2, ? extends P3>, FP>
  Procedure1<FP> after1(@NotNull Function<FP, FR> before)
  {
    return fp -> before.apply(fp).sendTo(this);
  }

  /**
   * Get a chained procedure which first applies the given 2-parameter function, then this procedure.
   * The given function has to return a {@link ITuple3} with matching parametric types
   * which are applied in their order to this procedure.
   * @param before function to apply first, has to return a non-{@code null} result so unpacking the parameters
   *               from the tuple is possible
   * @param <FR> function {@code before}'s return type, has to be a 3-tuple with matching types
   * @param <FP1> function {@code before}'s first parameter type, sane for returned procedure
   * @param <FP2> function {@code before}'s second parameter type, same for returned procedure
   * @return chain of the given function and this procedure
   */
  @NotNull
  default <FR extends ITuple3<? extends P1, ? extends P2, ? extends P3>, FP1, FP2>
  Procedure2<FP1, FP2> after2(@NotNull BiFunction<FP1, FP2, FR> before)
  {
    return (fp1, fp2) -> before.apply(fp1, fp2).sendTo(this);
  }

  /**
   * Get a chained procedure which first applies the given 3-parameter function, then this procedure.
   * The given function has to return a {@link ITuple3} with matching parametric types
   * which are applied in their order to this procedure.
   * @param before function to apply first, has to return a non-{@code null} result so unpacking the parameters
   *               from the tuple is possible
   * @param <FR> function {@code before}'s return type, has to be a 3-tuple with matching types
   * @param <FP1> function {@code before}'s first parameter type, sane for returned procedure
   * @param <FP2> function {@code before}'s second parameter type, sane for returned procedure
   * @param <FP3> function {@code before}'s third parameter type, sane for returned procedure
   * @return chain of the given function and this procedure
   */
  @NotNull
  default <FR extends ITuple3<? extends P1, ? extends P2, ? extends P3>, FP1, FP2, FP3>
  Procedure3<FP1, FP2, FP3> after3(@NotNull Function3<FR, FP1, FP2, FP3> before)
  {
    return (fp1, fp2, fp3) -> before.apply(fp1, fp2, fp3).sendTo(this);
  }

  /**
   * Get a chained procedure which first applies the given 5-parameter function, then this procedure.
   * The given function has to return a {@link ITuple3} with matching parametric types
   * which are applied in their order to this function.
   * @param before function to apply first, has to return a non-{@code null} result so unpacking the parameters
   *               from the tuple is possible
   * @param <FR> function {@code before}'s return type, has to be a 3-tuple with matching types
   * @param <FP1> function {@code before}'s first parameter type, sane for returned procedure
   * @param <FP2> function {@code before}'s second parameter type, sane for returned procedure
   * @param <FP3> function {@code before}'s third parameter type, sane for returned procedure
   * @param <FP4> function {@code before}'s fourth parameter type, sane for returned procedure
   * @param <FP5> function {@code before}'s fifth parameter type, sane for returned procedure
   * @return chain of the given function and this procedure
   */
  @NotNull
  default <FR extends ITuple3<? extends P1, ? extends P2, ? extends P3>, FP1, FP2, FP3, FP4, FP5>
  Procedure5<FP1, FP2, FP3, FP4, FP5> after5(@NotNull Function5<FR, FP1, FP2, FP3, FP4, FP5> before)
  {
    return (fp1, fp2, fp3, fp4, fp5) -> before.apply(fp1, fp2, fp3, fp4, fp5).sendTo(this);
  }

  /**
   * Get a chained procedure which first applies the given 6-parameter function, then this procedure.
   * The given function has to return a {@link ITuple3} with matching parametric types
   * which are applied in their order to this function.
   * @param before function to apply first, has to return a non-{@code null} result so unpacking the parameters
   *               from the tuple is possible
   * @param <FR> function {@code before}'s return type, has to be a 3-tuple with matching types
   * @param <FP1> function {@code before}'s first parameter type, sane for returned procedure
   * @param <FP2> function {@code before}'s second parameter type, sane for returned procedure
   * @param <FP3> function {@code before}'s third parameter type, sane for returned procedure
   * @param <FP4> function {@code before}'s fourth parameter type, sane for returned procedure
   * @param <FP5> function {@code before}'s fifth parameter type, sane for returned procedure
   * @param <FP6> function {@code before}'s sixth parameter type, sane for returned procedure
   * @return chain of the given function and this procedure
   */
  @NotNull
  default <FR extends ITuple3<? extends P1, ? extends P2, ? extends P3>, FP1, FP2, FP3, FP4, FP5, FP6>
  Procedure6<FP1, FP2, FP3, FP4, FP5, FP6> after6(@NotNull Function6<FR, FP1, FP2, FP3, FP4, FP5, FP6> before)
  {
    return (fp1, fp2, fp3, fp4, fp5, fp6) -> before.apply(fp1, fp2, fp3, fp4, fp5, fp6).sendTo(this);
  }

  /**
   * Get a chained procedure which first applies the given 7-parameter function, then this one.
   * The given function has to return a {@link ITuple3} with matching parametric types
   * which are applied in their order to this procedure.
   * @param before function to apply first, has to return a non-{@code null} result so unpacking the parameters
   *               from the tuple is possible
   * @param <FR> function {@code before}'s return type, has to be a 3-tuple with matching types
   * @param <FP1> function {@code before}'s first parameter type, sane for returned procedure
   * @param <FP2> function {@code before}'s second parameter type, sane for returned procedure
   * @param <FP3> function {@code before}'s third parameter type, sane for returned procedure
   * @param <FP4> function {@code before}'s fourth parameter type, sane for returned procedure
   * @param <FP5> function {@code before}'s fifth parameter type, sane for returned procedure
   * @param <FP6> function {@code before}'s sixth parameter type, sane for returned procedure
   * @param <FP7> function {@code before}'s seventh parameter type, sane for returned procedure
   * @return chain of the given function and this procedure
   */
  @NotNull
  default <FR extends ITuple3<? extends P1, ? extends P2, ? extends P3>, FP1, FP2, FP3, FP4, FP5, FP6, FP7>
  Procedure7<FP1, FP2, FP3, FP4, FP5, FP6, FP7> after7(@NotNull Function7<FR, FP1, FP2, FP3, FP4, FP5, FP6, FP7> before)
  {
    return (fp1, fp2, fp3, fp4, fp5, fp6, fp7) -> before.apply(fp1, fp2, fp3, fp4, fp5, fp6, fp7).sendTo(this);
  }

  /**
   * Get a chained procedure which first applies the given 8-parameter function, then this one.
   * The given function has to return a {@link ITuple3} with matching parametric types
   * which are applied in their order to this procedure.
   * @param before function to apply first, has to return a non-{@code null} result so unpacking the parameters
   *               from the tuple is possible
   * @param <FR> function {@code before}'s return type, has to be a 3-tuple with matching types
   * @param <FP1> function {@code before}'s first parameter type, sane for returned procedure
   * @param <FP2> function {@code before}'s second parameter type, sane for returned procedure
   * @param <FP3> function {@code before}'s third parameter type, sane for returned procedure
   * @param <FP4> function {@code before}'s fourth parameter type, sane for returned procedure
   * @param <FP5> function {@code before}'s fifth parameter type, sane for returned procedure
   * @param <FP6> function {@code before}'s sixth parameter type, sane for returned procedure
   * @param <FP7> function {@code before}'s seventh parameter type, sane for returned procedure
   * @param <FP8> function {@code before}'s seventh parameter type, sane for returned procedure
   * @return chain of the given function and this procedure
   */
  @NotNull
  default <FR extends ITuple3<? extends P1, ? extends P2, ? extends P3>, FP1, FP2, FP3, FP4, FP5, FP6, FP7, FP8>
  Procedure8<FP1, FP2, FP3, FP4, FP5, FP6, FP7, FP8> after8(@NotNull Function8<FR, FP1, FP2, FP3, FP4, FP5, FP6, FP7, FP8> before)
  {
    return (fp1, fp2, fp3, fp4, fp5, fp6, fp7, fp8) -> before.apply(fp1, fp2, fp3, fp4, fp5, fp6, fp7, fp8).sendTo(this);
  }

  /**
   * Get a chained procedure which first applies the given 9-parameter function, then this one.
   * The given function has to return a {@link ITuple3} with matching parametric types
   * which are applied in their order to this procedure.
   * @param before function to apply first, has to return a non-{@code null} result so unpacking the parameters
   *               from the tuple is possible
   * @param <FR> function {@code before}'s return type, has to be a 3-tuple with matching types
   * @param <FP1> function {@code before}'s first parameter type, sane for returned procedure
   * @param <FP2> function {@code before}'s second parameter type, sane for returned procedure
   * @param <FP3> function {@code before}'s third parameter type, sane for returned procedure
   * @param <FP4> function {@code before}'s fourth parameter type, sane for returned procedure
   * @param <FP5> function {@code before}'s fifth parameter type, sane for returned procedure
   * @param <FP6> function {@code before}'s sixth parameter type, sane for returned procedure
   * @param <FP7> function {@code before}'s seventh parameter type, sane for returned procedure
   * @param <FP8> function {@code before}'s seventh parameter type, sane for returned procedure
   * @param <FP9> function {@code before}'s ninth parameter type, sane for returned procedure
   * @return chain of the given function and this procedure
   */
  @NotNull
  default <FR extends ITuple3<? extends P1, ? extends P2, ? extends P3>, FP1, FP2, FP3, FP4, FP5, FP6, FP7, FP8, FP9>
  Procedure9<FP1, FP2, FP3, FP4, FP5, FP6, FP7, FP8, FP9> after9(@NotNull Function9<FR, FP1, FP2, FP3, FP4, FP5, FP6, FP7, FP8, FP9> before)
  {
    return (fp1, fp2, fp3, fp4, fp5, fp6, fp7, fp8, fp9) -> before.apply(fp1, fp2, fp3, fp4, fp5, fp6, fp7, fp8, fp9).sendTo(this);
  }

  /**
   * Return a procedure which does nothing.
   * This just returns {@link #NULL}, but the compiler will accept it smoothly.
   * @param <T1> first argument type
   * @param <T2> second argument type
   * @param <T3> third argument type
   * @return do-nothing procedure
   */
  @NotNull
  @SuppressWarnings("unchecked") // do nothing is always same
  static <T1, T2, T3> Procedure3<T1, T2, T3> empty()
  {
    return (Procedure3<T1, T2, T3>)NULL;
  }

  /**
   * Convert a 3-argument procedure which might throw a checked exception into
   * one which does throw an unchecked exception.
   * The returned function will instead throw an unchecked {@link WrappedFragileException} for any
   * checked exception thrown during {@link #apply(Object)}.
   * @param fragileProc procedure throwing a checked exception
   * @param <E> exception type
   * @param <PP1> type of first parameter
   * @param <PP2> type of second parameter
   * @param <PP3> type of third parameter
   * @return non-fragile procedure
   */
  @NotNull
  static <E extends Exception, PP1, PP2, PP3>
  Procedure3<PP1, PP2, PP3> nonFragile(@NotNull FragileProcedure3<E, ? super PP1, ? super PP2, ? super PP3> fragileProc)
  {
    return (p1, p2, p3) -> {
      try {
        fragileProc.apply(p1, p2, p3);
      } catch (Exception x) {
        throw new WrappedFragileException(x);
      }
    };
  }

  /**
   * Convert a 3-argument procedure which might throw a checked exception into
   * one for which you can decide what happens with the exception.
   * The exception handler allows
   * <ul>
   *   <li>throw an unchecked exception instead (compare {@link #nonFragile(FragileProcedure3)})</li>
   *   <li>do nothing</li>
   *   <li>include logging in one of the above</li>
   *   <li>and more...</li>
   * </ul>
   * @param fragileProc procedure throwing a checked exception
   * @param exceptionHandler exception handler. Its return value will be used as return value of the returned function.
   * @param <E> exception type
   * @param <PP1> type of first parameter
   * @param <PP2> type of second parameter
   * @param <PP3> type of third parameter
   * @return non-fragile function
   */
  @NotNull
  @SuppressWarnings("unchecked") // compiler should take care that only the correct checked exception is used
  static <E extends Exception, PP1, PP2, PP3>
  Procedure3<PP1, PP2, PP3> nonFragileX(@NotNull FragileProcedure3<E, ? super PP1, ? super PP2, ? super PP3> fragileProc,
                                        @NotNull Procedure4<? super E, ? super PP1, ? super PP2, ? super PP3> exceptionHandler)
  {
    return (p1, p2, p3) -> {
      try {
        fragileProc.apply(p1, p2, p3);
      } catch (Exception x) {
        exceptionHandler.apply((E)x, p1, p2, p3);
      }
    };
  }
}
