// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.function;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;
import de.caff.generics.WrappedFragileException;

import java.util.function.BiConsumer;
import java.util.function.Consumer;

/**
 * Function without return and with one parameter.
 * This provides the same functional interface as modern Java's {@link java.util.function.Consumer Consumer},
 * but in a more extensible way.
 * As it is extending {@code Consumer} it can be used anywhere a {@code Consumer} is expected.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @param <P> parameter type
 * @see Procedure0
 * @see Procedure2
 * @see Procedure3
 * @see Procedure4
 * @see Procedure5
 * @see Procedure6
 * @see Procedure7
 * @see Procedure8
 * @see Procedure9
 */
@FunctionalInterface
public interface Procedure1<P>
        extends Consumer<P>
{
  /**
   * Procedure with one argument which does nothing.
   * Use {@link #empty()} instead.
   */
  @NotNull
  Procedure1<?> NULL = a -> {};

  /**
   * Run the function.
   * @param arg function argument
   */
  default void apply(P arg)
  {
    accept(arg);
  }

  /**
   * Apply this procedure only if the argument is not {@code null}.
   *<p>
   * Using this invocation it is guaranteed that {@link #apply(Object)} will always be called with
   * a non-null argument.
   * </p>
   * <p>
   *   Use {@link de.caff.generics.Types#notNull(Object, Object)} if yo want to invoke this
   *   with a default argument when {@code argument} is {@code null}.
   * </p>
   * @param argument argument of this function, possibly {@code null}
   */
  default void applyNonNull(@Nullable P argument)
  {
    if (argument != null) {
      apply(argument);
    }
  }

  /**
   * Return a partial evaluation of this procedure.
   *
   * @param argument constant argument use for the evaluation
   * @return zero argument procedure
   */
  @NotNull
  default Procedure0 partial(P argument)
  {
    return () -> apply(argument);
  }

  /**
   * Make a {@link Consumer} usable as a {@link Procedure1}.
   * @param consumer consumer
   * @param <T> argument type of both
   * @return {@code Procedure1} view of the given consumer
   */
  @NotNull
  static <T> Procedure1<T> from(@NotNull Consumer<T> consumer)
  {
    return consumer instanceof  Procedure1
            ? (Procedure1<T>)consumer
            : consumer::accept;
  }

  /**
   * Return a procedure which does nothing.
   * This just returns {@link #NULL}, but the compiler will accept it smoothly.
   * @param <T> argument type
   * @return do-nothing procedure
   */
  @NotNull
  @SuppressWarnings("unchecked") // do nothing is always same
  static <T> Procedure1<T> empty()
  {
    return (Procedure1<T>)NULL;
  }

  /**
   * Convert a 1-argument procedure which might throw a checked exception into
   * one which does throw an unchecked exception.
   * The returned function will instead throw an unchecked {@link WrappedFragileException} for any
   * checked exception thrown during {@link #apply(Object)}.
   * @param fragileProc procedure throwing a checked exception
   * @param <E> exception type
   * @param <PP> parameter type
   * @return non-fragile procedure
   */
  @NotNull
  static <E extends Exception, PP> Procedure1<PP> nonFragile(@NotNull FragileProcedure1<E, ? super PP> fragileProc)
  {
    return p -> {
      try {
        fragileProc.apply(p);
      } catch (Exception x) {
        throw new WrappedFragileException(x);
      }
    };
  }

  /**
   * Convert a 1-argument procedure which might throw a checked exception into
   * one for which you can decide what happens with the exception.
   * The exception handler allows
   * <ul>
   *   <li>throw an unchecked exception instead (compare {@link #nonFragile(FragileProcedure1)})</li>
   *   <li>do nothing</li>
   *   <li>include logging in one of the above</li>
   *   <li>and more...</li>
   * </ul>
   * @param fragileProc procedure throwing a checked exception
   * @param exceptionHandler exception handler. Its return value will be used as return value of the returned function.
   * @param <E> exception type
   * @param <PP> parameter type
   * @return non-fragile function
   */
  @NotNull
  @SuppressWarnings("unchecked") // compiler should take care that only the correct checked exception is used
  static <E extends Exception, PP> Procedure1<PP> nonFragileX(@NotNull FragileProcedure1<E, ? super PP> fragileProc,
                                                              @NotNull BiConsumer<? super E, ? super PP> exceptionHandler)
  {
    return p -> {
      try {
        fragileProc.apply(p);
      } catch (Exception x) {
        exceptionHandler.accept((E)x, p);
      }
    };
  }
}
