// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.function;

import de.caff.annotation.NotNull;
import de.caff.generics.WrappedFragileException;

import java.util.function.Consumer;

/**
 * Function without return and with no parameter.
 * In a pure functional world this is useless, but in the real world it can be useful sometimes.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @see Procedure1
 * @see Procedure2
 * @see Procedure3
 * @see Procedure4
 * @see Procedure5
 * @see Procedure6
 * @see Procedure7
 * @see Procedure8
 * @see Procedure9
 */
@FunctionalInterface
public interface Procedure0
{
  /**
   * Procedure with no arguments which does nothing.
   */
  @NotNull
  Procedure0 NULL = () -> {};

  /**
   * Run the function.
   */
  void apply();

  /**
   * Convert a 0-argument procedure which might throw a checked exception into
   * one which does throw an unchecked exception.
   * The returned function will instead throw an unchecked {@link WrappedFragileException} for any
   * checked exception thrown during {@link #apply()}.
   * @param fragileProc procedure throwing a checked exception
   * @param <E> exception type
   * @return non-fragile procedure
   */
  @NotNull
  static <E extends Exception> Procedure0 nonFragile(@NotNull FragileProcedure0<E> fragileProc)
  {
    return () -> {
      try {
        fragileProc.apply();
      } catch (Exception x) {
        throw new WrappedFragileException(x);
      }
    };
  }

  /**
   * Convert a 0-argument procedure which might throw a checked exception into
   * one for which you can decide what happens with the exception.
   * The exception handler allows
   * <ul>
   *   <li>throw an unchecked exception instead (compare {@link #nonFragile(FragileProcedure0)})</li>
   *   <li>do nothing</li>
   *   <li>include logging in one of the above</li>
   *   <li>and more...</li>
   * </ul>
   * @param fragileProc procedure throwing a checked exception
   * @param exceptionHandler exception handler. Its return value will be used as return value of the returned function.
   * @param <E> exception type
   * @return non-fragile function
   */
  @NotNull
  @SuppressWarnings("unchecked") // compiler should take care that only the correct checked exception is used
  static <E extends Exception> Procedure0 nonFragileX(@NotNull FragileProcedure0<E> fragileProc,
                                                      @NotNull Consumer<? super E> exceptionHandler)
  {
    return () -> {
      try {
        fragileProc.apply();
      } catch (Exception x) {
        exceptionHandler.accept((E)x);
      }
    };
  }
}
