// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.function;

import de.caff.annotation.NotNull;

import java.util.Objects;
import java.util.function.BiPredicate;
import java.util.function.Predicate;

/**
 * Helper for predicates.
 * Currently only used to store non-public constants.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since February 21, 2019
 */
final class Predicates
{
  /**
   * One argument predicate which always evaluates to {@code true}.
   * @see Predicate1#alwaysTrue()
   */
  static Predicate1<Object> TRUE1 = new Predicate1<Object>() {
    @Override
    public boolean test(Object arg)
    {
      return true;
    }

    @NotNull
    @Override
    public Predicate1<Object> and(@NotNull Predicate<? super Object> other)
    {
      return Predicate1.from(other);
    }

    @NotNull
    @Override
    public <T> Predicate1<T> and_(@NotNull Predicate<T> other)
    {
      return Predicate1.from(other);
    }

    @NotNull
    @Override
    public Predicate1<Object> or(@NotNull Predicate<? super Object> other)
    {
      return this;
    }

    @NotNull
    @Override
    public <T> Predicate1<T> or_(@NotNull Predicate<T> other)
    {
      return Predicate1.alwaysTrue();
    }

    @NotNull
    @Override
    public Predicate1<Object> xor(@NotNull Predicate<? super Object> other)
    {
      return Predicate1.from(other).negate();
    }

    @NotNull
    @Override
    public <T> Predicate1<T> xor_(@NotNull Predicate<T> other)
    {
      return Predicate1.from(other).negate();
    }

    @NotNull
    @Override
    public Predicate1<Object> negate()
    {
      return FALSE1;
    }
  };
  /**
   * One argument predicate which always evaluates to {@code false}.
   * @see Predicate1#alwaysFalse()
   */
  static Predicate1<Object> FALSE1 = new Predicate1<Object>() {
    @Override
    public boolean test(Object arg)
    {
      return false;
    }

    @NotNull
    @Override
    public Predicate1<Object> and(@NotNull Predicate<? super Object> other)
    {
      return this;
    }

    @NotNull
    @Override
    public <T> Predicate1<T> and_(@NotNull Predicate<T> other)
    {
      return Predicate1.alwaysFalse();
    }

    @NotNull
    @Override
    public Predicate1<Object> or(@NotNull Predicate<? super Object> other)
    {
      return Predicate1.from(other);
    }

    @NotNull
    @Override
    public <T> Predicate1<T> or_(@NotNull Predicate<T> other)
    {
      return Predicate1.from(other);
    }

    @NotNull
    @Override
    public Predicate1<Object> xor(@NotNull Predicate<? super Object> other)
    {
      return Predicate1.from(other);
    }

    @NotNull
    @Override
    public <T> Predicate1<T> xor_(@NotNull Predicate<T> other)
    {
      return Predicate1.from(other);
    }

    @NotNull
    @Override
    public Predicate1<Object> negate()
    {
      return TRUE1;
    }
  };

  /**
   * Two argument predicate which always evaluates to {@code true}.
   * @see Predicate2#alwaysTrue()
   */
  static Predicate2<Object, Object> TRUE2 = new Predicate2<Object, Object>() {
    @Override
    public boolean test(Object arg1, Object arg2)
    {
      return true;
    }

    @NotNull
    @Override
    public Predicate2<Object, Object> and(@NotNull BiPredicate<? super Object, ? super Object> other)
    {
      return Predicate2.from(other);
    }

    @NotNull
    @Override
    public Predicate2<Object, Object> or(@NotNull BiPredicate<? super Object, ? super Object> other)
    {
      return this;
    }

    @NotNull
    @Override
    public Predicate2<Object, Object> xor(@NotNull BiPredicate<? super Object, ? super Object> other)
    {
      return Predicate2.from(other).negate();
    }

    @NotNull
    @Override
    public Predicate2<Object, Object> negate()
    {
      return FALSE2;
    }

    @NotNull
    @Override
    public Predicate1<Object> partialLeft(Object argument)
    {
      return TRUE1;
    }

    @NotNull
    @Override
    public Predicate1<Object> partial1(Object argument)
    {
      return TRUE1;
    }

    @NotNull
    @Override
    public Predicate1<Object> partialRight(Object argument)
    {
      return TRUE1;
    }

    @NotNull
    @Override
    public Predicate1<Object> partial2(Object argument)
    {
      return TRUE1;
    }

    @NotNull
    @Override
    public Predicate2<Object, Object> reverseOrder()
    {
      return this;
    }
  };
  /**
   * Two argument predicate which always evaluates to {@code false}.
   * @see Predicate2#alwaysFalse()
   */
  static Predicate2<Object, Object> FALSE2 = new Predicate2<Object, Object>() {
    @Override
    public boolean test(Object arg1, Object arg2)
    {
      return false;
    }

    @NotNull
    @Override
    public Predicate2<Object, Object> and(@NotNull BiPredicate<? super Object, ? super Object> other)
    {
      return this;
    }

    @NotNull
    @Override
    public Predicate2<Object, Object> or(@NotNull BiPredicate<? super Object, ? super Object> other)
    {
      return Predicate2.from(other);
    }

    @NotNull
    @Override
    public Predicate2<Object, Object> xor(@NotNull BiPredicate<? super Object, ? super Object> other)
    {
      return Predicate2.from(other);
    }

    @NotNull
    @Override
    public Predicate2<Object, Object> negate()
    {
      return TRUE2;
    }

    @NotNull
    @Override
    public Predicate1<Object> partialLeft(Object argument)
    {
      return FALSE1;
    }

    @NotNull
    @Override
    public Predicate1<Object> partial1(Object argument)
    {
      return FALSE1;
    }

    @NotNull
    @Override
    public Predicate1<Object> partialRight(Object argument)
    {
      return FALSE1;
    }

    @NotNull
    @Override
    public Predicate2<Object, Object> reverseOrder()
    {
      return this;
    }
  };

  /**
   * Three argument predicate which always evaluates to {@code true}.
   * @see Predicate3#alwaysTrue()
   */
  static Predicate3<Object, Object, Object> TRUE3 = new Predicate3<Object, Object, Object>() {
    @Override
    public boolean test(Object arg1, Object arg2, Object arg)
    {
      return true;
    }

    @NotNull
    @Override
    public Predicate3<Object, Object, Object> and(@NotNull Predicate3<? super Object, ? super Object, ? super Object> other)
    {
      return other;
    }

    @NotNull
    @Override
    public Predicate3<Object, Object, Object> or(@NotNull Predicate3<? super Object, ? super Object, ? super Object> other)
    {
      return this;
    }

    @NotNull
    @Override
    public Predicate3<Object, Object, Object> xor(@NotNull Predicate3<? super Object, ? super Object, ? super Object> other)
    {
      return other.negate();
    }

    @NotNull
    @Override
    public Predicate3<Object, Object, Object> negate()
    {
      return FALSE3;
    }

    @NotNull
    @Override
    public Predicate2<Object, Object> partial1(Object argument)
    {
      return TRUE2;
    }

    @NotNull
    @Override
    public Predicate2<Object, Object> partial2(Object argument)
    {
      return TRUE2;
    }

    @NotNull
    @Override
    public Predicate2<Object, Object> partial3(Object argument)
    {
      return TRUE2;
    }

    @NotNull
    @Override
    public Predicate3<Object, Object, Object> order231()
    {
      return this;
    }

    @NotNull
    @Override
    public Predicate3<Object, Object, Object> order312()
    {
      return this;
    }

    @NotNull
    @Override
    public Predicate3<Object, Object, Object> order321()
    {
      return this;
    }

    @NotNull
    @Override
    public Predicate3<Object, Object, Object> order132()
    {
      return this;
    }

    @NotNull
    @Override
    public Predicate3<Object, Object, Object> order213()
    {
      return this;
    }
  };
  /**
   * Three argument predicate which always evaluates to {@code false}.
   * @see Predicate3#alwaysFalse()
   */
  static Predicate3<Object, Object, Object> FALSE3 = new Predicate3<Object, Object, Object>() {
    @Override
    public boolean test(Object arg1, Object arg2, Object arg)
    {
      return false;
    }

    @NotNull
    @Override
    public Predicate3<Object, Object, Object> and(@NotNull Predicate3<? super Object, ? super Object, ? super Object> other)
    {
      return this;
    }

    @NotNull
    @Override
    public Predicate3<Object, Object, Object> or(@NotNull Predicate3<? super Object, ? super Object, ? super Object> other)
    {
      return other;
    }

    @NotNull
    @Override
    public Predicate3<Object, Object, Object> xor(@NotNull Predicate3<? super Object, ? super Object, ? super Object> other)
    {
      return other;
    }

    @NotNull
    @Override
    public Predicate3<Object, Object, Object> negate()
    {
      return TRUE3;
    }

    @NotNull
    @Override
    public Predicate2<Object, Object> partial1(Object argument)
    {
      return FALSE2;
    }

    @NotNull
    @Override
    public Predicate2<Object, Object> partial2(Object argument)
    {
      return FALSE2;
    }

    @NotNull
    @Override
    public Predicate2<Object, Object> partial3(Object argument)
    {
      return FALSE2;
    }

    @NotNull
    @Override
    public Predicate3<Object, Object, Object> order231()
    {
      return this;
    }

    @NotNull
    @Override
    public Predicate3<Object, Object, Object> order312()
    {
      return this;
    }

    @NotNull
    @Override
    public Predicate3<Object, Object, Object> order321()
    {
      return this;
    }

    @NotNull
    @Override
    public Predicate3<Object, Object, Object> order132()
    {
      return this;
    }

    @NotNull
    @Override
    public Predicate3<Object, Object, Object> order213()
    {
      return this;
    }
  };

  /**
   * Four argument predicate which always evaluates to {@code true}.
   * @see Predicate4#alwaysTrue()
   */
  static Predicate4<Object, Object, Object, Object> TRUE4 = new Predicate4<Object, Object, Object, Object>()
  {
    @Override
    public boolean test(Object arg1, Object arg2, Object arg3, Object arg4)
    {
      return true;
    }

    @NotNull
    @Override
    public Predicate4<Object, Object, Object, Object> and(@NotNull Predicate4<? super Object, ? super Object, ? super Object, ? super Object> other)
    {
      return other;
    }

    @NotNull
    @Override
    public Predicate4<Object, Object, Object, Object> or(@NotNull Predicate4<? super Object, ? super Object, ? super Object, ? super Object> other)
    {
      return this;
    }

    @NotNull
    @Override
    public Predicate4<Object, Object, Object, Object> xor(@NotNull Predicate4<? super Object, ? super Object, ? super Object, ? super Object> other)
    {
      return other.negate();
    }

    @NotNull
    @Override
    public Predicate4<Object, Object, Object, Object> negate()
    {
      return FALSE4;
    }

    @NotNull
    @Override
    public Predicate3<Object, Object, Object> partial1(Object argument)
    {
      return TRUE3;
    }

    @NotNull
    @Override
    public Predicate3<Object, Object, Object> partial2(Object argument)
    {
      return TRUE3;
    }

    @NotNull
    @Override
    public Predicate3<Object, Object, Object> partial3(Object argument)
    {
      return TRUE3;
    }

    @NotNull
    @Override
    public Predicate3<Object, Object, Object> partial4(Object argument)
    {
      return TRUE3;
    }
  };
  /**
   * Four argument predicate which always evaluates to {@code false}.
   * @see Predicate4#alwaysFalse()
   */
  static Predicate4<Object, Object, Object, Object> FALSE4 = new Predicate4<Object, Object, Object, Object>()
  {
    @Override
    public boolean test(Object arg1, Object arg2, Object arg3, Object arg4)
    {
      return false;
    }

    @NotNull
    @Override
    public Predicate4<Object, Object, Object, Object> and(@NotNull Predicate4<? super Object, ? super Object, ? super Object, ? super Object> other)
    {
      return this;
    }

    @NotNull
    @Override
    public Predicate4<Object, Object, Object, Object> or(@NotNull Predicate4<? super Object, ? super Object, ? super Object, ? super Object> other)
    {
      return other;
    }

    @NotNull
    @Override
    public Predicate4<Object, Object, Object, Object> xor(@NotNull Predicate4<? super Object, ? super Object, ? super Object, ? super Object> other)
    {
      return other;
    }

    @NotNull
    @Override
    public Predicate4<Object, Object, Object, Object> negate()
    {
      return TRUE4;
    }

    @NotNull
    @Override
    public Predicate3<Object, Object, Object> partial1(Object argument)
    {
      return FALSE3;
    }

    @NotNull
    @Override
    public Predicate3<Object, Object, Object> partial2(Object argument)
    {
      return FALSE3;
    }

    @NotNull
    @Override
    public Predicate3<Object, Object, Object> partial3(Object argument)
    {
      return FALSE3;
    }

    @NotNull
    @Override
    public Predicate3<Object, Object, Object> partial4(Object argument)
    {
      return FALSE3;
    }
  };

  /**
   * Five argument predicate which always evaluates to {@code true}.
   * @see Predicate5#alwaysTrue()
   */
  static Predicate5<Object, Object, Object, Object, Object> TRUE5 = new Predicate5<Object, Object, Object, Object, Object>()
  {
    @Override
    public boolean test(Object arg1, Object arg2, Object arg3, Object arg4, Object arg5)
    {
      return true;
    }

    @NotNull
    @Override
    public Predicate5<Object, Object, Object, Object, Object> and(@NotNull Predicate5<? super Object, ? super Object, ? super Object, ? super Object, ? super Object> other)
    {
      return other;
    }

    @NotNull
    @Override
    public Predicate5<Object, Object, Object, Object, Object> or(@NotNull Predicate5<? super Object, ? super Object, ? super Object, ? super Object, ? super Object> other)
    {
      return this;
    }

    @NotNull
    @Override
    public Predicate5<Object, Object, Object, Object, Object> xor(@NotNull Predicate5<? super Object, ? super Object, ? super Object, ? super Object, ? super Object> other)
    {
      return other.negate();
    }

    @NotNull
    @Override
    public Predicate5<Object, Object, Object, Object, Object> negate()
    {
      return FALSE5;
    }

    @NotNull
    @Override
    public Predicate4<Object, Object, Object, Object> partial1(Object argument)
    {
      return TRUE4;
    }

    @NotNull
    @Override
    public Predicate4<Object, Object, Object, Object> partial2(Object argument)
    {
      return TRUE4;
    }

    @NotNull
    @Override
    public Predicate4<Object, Object, Object, Object> partial3(Object argument)
    {
      return TRUE4;
    }

    @NotNull
    @Override
    public Predicate4<Object, Object, Object, Object> partial4(Object argument)
    {
      return TRUE4;
    }

    @NotNull
    @Override
    public Predicate4<Object, Object, Object, Object> partial5(Object argument)
    {
      return TRUE4;
    }
  };
  /**
   * Five argument predicate which always evaluates to {@code false}.
   * @see Predicate5#alwaysFalse()
   */
  static Predicate5<Object, Object, Object, Object, Object> FALSE5 = new Predicate5<Object, Object, Object, Object, Object>()
  {
    @Override
    public boolean test(Object arg1, Object arg2, Object arg3, Object arg4, Object arg5)
    {
      return false;
    }

    @NotNull
    @Override
    public Predicate5<Object, Object, Object, Object, Object> and(@NotNull Predicate5<? super Object, ? super Object, ? super Object, ? super Object, ? super Object> other)
    {
      return this;
    }

    @NotNull
    @Override
    public Predicate5<Object, Object, Object, Object, Object> or(@NotNull Predicate5<? super Object, ? super Object, ? super Object, ? super Object, ? super Object> other)
    {
      return other;
    }

    @NotNull
    @Override
    public Predicate5<Object, Object, Object, Object, Object> xor(@NotNull Predicate5<? super Object, ? super Object, ? super Object, ? super Object, ? super Object> other)
    {
      return other;
    }

    @NotNull
    @Override
    public Predicate5<Object, Object, Object, Object, Object> negate()
    {
      return TRUE5;
    }

    @NotNull
    @Override
    public Predicate4<Object, Object, Object, Object> partial1(Object argument)
    {
      return FALSE4;
    }

    @NotNull
    @Override
    public Predicate4<Object, Object, Object, Object> partial2(Object argument)
    {
      return FALSE4;
    }

    @NotNull
    @Override
    public Predicate4<Object, Object, Object, Object> partial3(Object argument)
    {
      return FALSE4;
    }

    @NotNull
    @Override
    public Predicate4<Object, Object, Object, Object> partial4(Object argument)
    {
      return FALSE4;
    }

    @NotNull
    @Override
    public Predicate4<Object, Object, Object, Object> partial5(Object argument)
    {
      return FALSE4;
    }
  };

  /**
   * Six argument predicate which always evaluates to {@code true}.
   * @see Predicate6#alwaysTrue()
   */
  static Predicate6<Object, Object, Object, Object, Object, Object> TRUE6 = new Predicate6<Object, Object, Object, Object, Object, Object>()
  {
    @Override
    public boolean test(Object arg1, Object arg2, Object arg3, Object arg4, Object arg5, Object arg6)
    {
      return true;
    }

    @NotNull
    @Override
    public Predicate6<Object, Object, Object, Object, Object, Object> and(@NotNull Predicate6<? super Object, ? super Object, ? super Object, ? super Object, ? super Object, ? super Object> other)
    {
      return other;
    }

    @NotNull
    @Override
    public Predicate6<Object, Object, Object, Object, Object, Object> or(@NotNull Predicate6<? super Object, ? super Object, ? super Object, ? super Object, ? super Object, ? super Object> other)
    {
      return this;
    }

    @NotNull
    @Override
    public Predicate6<Object, Object, Object, Object, Object, Object> xor(@NotNull Predicate6<? super Object, ? super Object, ? super Object, ? super Object, ? super Object, ? super Object> other)
    {
      return other.negate();
    }

    @NotNull
    @Override
    public Predicate6<Object, Object, Object, Object, Object, Object> negate()
    {
      return FALSE6;
    }

    @NotNull
    @Override
    public Predicate5<Object, Object, Object, Object, Object> partial1(Object argument)
    {
      return TRUE5;
    }

    @NotNull
    @Override
    public Predicate5<Object, Object, Object, Object, Object> partial2(Object argument)
    {
      return TRUE5;
    }

    @NotNull
    @Override
    public Predicate5<Object, Object, Object, Object, Object> partial3(Object argument)
    {
      return TRUE5;
    }

    @NotNull
    @Override
    public Predicate5<Object, Object, Object, Object, Object> partial4(Object argument)
    {
      return TRUE5;
    }

    @NotNull
    @Override
    public Predicate5<Object, Object, Object, Object, Object> partial5(Object argument)
    {
      return TRUE5;
    }

    @NotNull
    @Override
    public Predicate5<Object, Object, Object, Object, Object> partial6(Object argument)
    {
      return TRUE5;
    }
  };
  /**
   * Six argument predicate which always evaluates to {@code false}.
   * @see Predicate6#alwaysFalse()
   */
  static Predicate6<Object, Object, Object, Object, Object, Object> FALSE6 = new Predicate6<Object, Object, Object, Object, Object, Object>()
  {
    @Override
    public boolean test(Object arg1, Object arg2, Object arg3, Object arg4, Object arg5, Object arg6)
    {
      return false;
    }

    @NotNull
    @Override
    public Predicate6<Object, Object, Object, Object, Object, Object> and(@NotNull Predicate6<? super Object, ? super Object, ? super Object, ? super Object, ? super Object, ? super Object> other)
    {
      return this;
    }

    @NotNull
    @Override
    public Predicate6<Object, Object, Object, Object, Object, Object> or(@NotNull Predicate6<? super Object, ? super Object, ? super Object, ? super Object, ? super Object, ? super Object> other)
    {
      return other;
    }

    @NotNull
    @Override
    public Predicate6<Object, Object, Object, Object, Object, Object> xor(@NotNull Predicate6<? super Object, ? super Object, ? super Object, ? super Object, ? super Object, ? super Object> other)
    {
      return other;
    }

    @NotNull
    @Override
    public Predicate6<Object, Object, Object, Object, Object, Object> negate()
    {
      return TRUE6;
    }

    @NotNull
    @Override
    public Predicate5<Object, Object, Object, Object, Object> partial1(Object argument)
    {
      return FALSE5;
    }

    @NotNull
    @Override
    public Predicate5<Object, Object, Object, Object, Object> partial2(Object argument)
    {
      return FALSE5;
    }

    @NotNull
    @Override
    public Predicate5<Object, Object, Object, Object, Object> partial3(Object argument)
    {
      return FALSE5;
    }

    @NotNull
    @Override
    public Predicate5<Object, Object, Object, Object, Object> partial4(Object argument)
    {
      return FALSE5;
    }

    @NotNull
    @Override
    public Predicate5<Object, Object, Object, Object, Object> partial5(Object argument)
    {
      return FALSE5;
    }

    @NotNull
    @Override
    public Predicate5<Object, Object, Object, Object, Object> partial6(Object argument)
    {
      return FALSE5;
    }
  };

  /**
   * Seven argument predicate which always evaluates to {@code true}.
   * @see Predicate7#alwaysTrue()
   */
  static Predicate7<Object, Object, Object, Object, Object, Object, Object> TRUE7 = new Predicate7<Object, Object, Object, Object, Object, Object, Object>()
  {
    @Override
    public boolean test(Object arg1, Object arg2, Object arg3, Object arg4, Object arg5, Object arg6, Object arg7)
    {
      return true;
    }

    @NotNull
    @Override
    public Predicate7<Object, Object, Object, Object, Object, Object, Object> and(@NotNull Predicate7<? super Object, ? super Object, ? super Object, ? super Object, ? super Object, ? super Object, ? super Object> other)
    {
      return other;
    }

    @NotNull
    @Override
    public Predicate7<Object, Object, Object, Object, Object, Object, Object> or(@NotNull Predicate7<? super Object, ? super Object, ? super Object, ? super Object, ? super Object, ? super Object, ? super Object> other)
    {
      return this;
    }

    @NotNull
    @Override
    public Predicate7<Object, Object, Object, Object, Object, Object, Object> xor(@NotNull Predicate7<? super Object, ? super Object, ? super Object, ? super Object, ? super Object, ? super Object, ? super Object> other)
    {
      return other.negate();
    }

    @NotNull
    @Override
    public Predicate7<Object, Object, Object, Object, Object, Object, Object> negate()
    {
      return FALSE7;
    }

    @NotNull
    @Override
    public Predicate6<Object, Object, Object, Object, Object, Object> partial1(Object argument)
    {
      return TRUE6;
    }

    @NotNull
    @Override
    public Predicate6<Object, Object, Object, Object, Object, Object> partial2(Object argument)
    {
      return TRUE6;
    }

    @NotNull
    @Override
    public Predicate6<Object, Object, Object, Object, Object, Object> partial3(Object argument)
    {
      return TRUE6;
    }

    @NotNull
    @Override
    public Predicate6<Object, Object, Object, Object, Object, Object> partial4(Object argument)
    {
      return TRUE6;
    }

    @NotNull
    @Override
    public Predicate6<Object, Object, Object, Object, Object, Object> partial5(Object argument)
    {
      return TRUE6;
    }

    @NotNull
    @Override
    public Predicate6<Object, Object, Object, Object, Object, Object> partial6(Object argument)
    {
      return TRUE6;
    }

    @NotNull
    @Override
    public Predicate6<Object, Object, Object, Object, Object, Object> partial7(Object argument)
    {
      return TRUE6;
    }
  };
  /**
   * Seven argument predicate which always evaluates to {@code false}.
   * @see Predicate7#alwaysFalse()
   */
  static Predicate7<Object, Object, Object, Object, Object, Object, Object> FALSE7 = new Predicate7<Object, Object, Object, Object, Object, Object, Object>()
  {
    @Override
    public boolean test(Object arg1, Object arg2, Object arg3, Object arg4, Object arg5, Object arg6, Object arg7)
    {
      return false;
    }

    @NotNull
    @Override
    public Predicate7<Object, Object, Object, Object, Object, Object, Object> and(@NotNull Predicate7<? super Object, ? super Object, ? super Object, ? super Object, ? super Object, ? super Object, ? super Object> other)
    {
      return this;
    }

    @NotNull
    @Override
    public Predicate7<Object, Object, Object, Object, Object, Object, Object> or(@NotNull Predicate7<? super Object, ? super Object, ? super Object, ? super Object, ? super Object, ? super Object, ? super Object> other)
    {
      return other;
    }

    @NotNull
    @Override
    public Predicate7<Object, Object, Object, Object, Object, Object, Object> xor(@NotNull Predicate7<? super Object, ? super Object, ? super Object, ? super Object, ? super Object, ? super Object, ? super Object> other)
    {
      return other;
    }

    @NotNull
    @Override
    public Predicate7<Object, Object, Object, Object, Object, Object, Object> negate()
    {
      return TRUE7;
    }

    @NotNull
    @Override
    public Predicate6<Object, Object, Object, Object, Object, Object> partial1(Object argument)
    {
      return FALSE6;
    }

    @NotNull
    @Override
    public Predicate6<Object, Object, Object, Object, Object, Object> partial2(Object argument)
    {
      return FALSE6;
    }

    @NotNull
    @Override
    public Predicate6<Object, Object, Object, Object, Object, Object> partial3(Object argument)
    {
      return FALSE6;
    }

    @NotNull
    @Override
    public Predicate6<Object, Object, Object, Object, Object, Object> partial4(Object argument)
    {
      return FALSE6;
    }

    @NotNull
    @Override
    public Predicate6<Object, Object, Object, Object, Object, Object> partial5(Object argument)
    {
      return FALSE6;
    }

    @NotNull
    @Override
    public Predicate6<Object, Object, Object, Object, Object, Object> partial6(Object argument)
    {
      return FALSE6;
    }

    @NotNull
    @Override
    public Predicate6<Object, Object, Object, Object, Object, Object> partial7(Object argument)
    {
      return FALSE6;
    }
  };

  /**
   * Eight argument predicate which always evaluates to {@code true}.
   * @see Predicate8#alwaysTrue()
   */
  static Predicate8<Object, Object, Object, Object, Object, Object, Object, Object> TRUE8 = new Predicate8<Object, Object, Object, Object, Object, Object, Object, Object>()
  {
    @Override
    public boolean test(Object arg1, Object arg2, Object arg3, Object arg4, Object arg5, Object arg6, Object arg7, Object arg8)
    {
      return true;
    }

    @NotNull
    @Override
    public Predicate8<Object, Object, Object, Object, Object, Object, Object, Object> and(@NotNull Predicate8<? super Object, ? super Object, ? super Object, ? super Object, ? super Object, ? super Object, ? super Object, ? super Object> other)
    {
      return other;
    }

    @NotNull
    @Override
    public Predicate8<Object, Object, Object, Object, Object, Object, Object, Object> or(@NotNull Predicate8<? super Object, ? super Object, ? super Object, ? super Object, ? super Object, ? super Object, ? super Object, ? super Object> other)
    {
      return this;
    }

    @NotNull
    @Override
    public Predicate8<Object, Object, Object, Object, Object, Object, Object, Object> xor(@NotNull Predicate8<? super Object, ? super Object, ? super Object, ? super Object, ? super Object, ? super Object, ? super Object, ? super Object> other)
    {
      return other.negate();
    }

    @NotNull
    @Override
    public Predicate8<Object, Object, Object, Object, Object, Object, Object, Object> negate()
    {
      return FALSE8;
    }

    @NotNull
    @Override
    public Predicate7<Object, Object, Object, Object, Object, Object, Object> partial1(Object argument)
    {
      return TRUE7;
    }

    @NotNull
    @Override
    public Predicate7<Object, Object, Object, Object, Object, Object, Object> partial2(Object argument)
    {
      return TRUE7;
    }

    @NotNull
    @Override
    public Predicate7<Object, Object, Object, Object, Object, Object, Object> partial3(Object argument)
    {
      return TRUE7;
    }

    @NotNull
    @Override
    public Predicate7<Object, Object, Object, Object, Object, Object, Object> partial4(Object argument)
    {
      return TRUE7;
    }

    @NotNull
    @Override
    public Predicate7<Object, Object, Object, Object, Object, Object, Object> partial5(Object argument)
    {
      return TRUE7;
    }

    @NotNull
    @Override
    public Predicate7<Object, Object, Object, Object, Object, Object, Object> partial6(Object argument)
    {
      return TRUE7;
    }

    @NotNull
    @Override
    public Predicate7<Object, Object, Object, Object, Object, Object, Object> partial7(Object argument)
    {
      return TRUE7;
    }

    @NotNull
    @Override
    public Predicate7<Object, Object, Object, Object, Object, Object, Object> partial8(Object argument)
    {
      return TRUE7;
    }
  };
  /**
   * Eight argument predicate which always evaluates to {@code false}.
   * @see Predicate8#alwaysFalse()
   */
  static Predicate8<Object, Object, Object, Object, Object, Object, Object, Object> FALSE8 = new Predicate8<Object, Object, Object, Object, Object, Object, Object, Object>()
  {
    @Override
    public boolean test(Object arg1, Object arg2, Object arg3, Object arg4, Object arg5, Object arg6, Object arg7, Object arg8)
    {
      return false;
    }

    @NotNull
    @Override
    public Predicate8<Object, Object, Object, Object, Object, Object, Object, Object> and(@NotNull Predicate8<? super Object, ? super Object, ? super Object, ? super Object, ? super Object, ? super Object, ? super Object, ? super Object> other)
    {
      return this;
    }

    @NotNull
    @Override
    public Predicate8<Object, Object, Object, Object, Object, Object, Object, Object> or(@NotNull Predicate8<? super Object, ? super Object, ? super Object, ? super Object, ? super Object, ? super Object, ? super Object, ? super Object> other)
    {
      return other;
    }

    @NotNull
    @Override
    public Predicate8<Object, Object, Object, Object, Object, Object, Object, Object> xor(@NotNull Predicate8<? super Object, ? super Object, ? super Object, ? super Object, ? super Object, ? super Object, ? super Object, ? super Object> other)
    {
      return other;
    }

    @NotNull
    @Override
    public Predicate8<Object, Object, Object, Object, Object, Object, Object, Object> negate()
    {
      return TRUE8;
    }

    @NotNull
    @Override
    public Predicate7<Object, Object, Object, Object, Object, Object, Object> partial1(Object argument)
    {
      return FALSE7;
    }

    @NotNull
    @Override
    public Predicate7<Object, Object, Object, Object, Object, Object, Object> partial2(Object argument)
    {
      return FALSE7;
    }

    @NotNull
    @Override
    public Predicate7<Object, Object, Object, Object, Object, Object, Object> partial3(Object argument)
    {
      return FALSE7;
    }

    @NotNull
    @Override
    public Predicate7<Object, Object, Object, Object, Object, Object, Object> partial4(Object argument)
    {
      return FALSE7;
    }

    @NotNull
    @Override
    public Predicate7<Object, Object, Object, Object, Object, Object, Object> partial5(Object argument)
    {
      return FALSE7;
    }

    @NotNull
    @Override
    public Predicate7<Object, Object, Object, Object, Object, Object, Object> partial6(Object argument)
    {
      return FALSE7;
    }

    @NotNull
    @Override
    public Predicate7<Object, Object, Object, Object, Object, Object, Object> partial7(Object argument)
    {
      return FALSE7;
    }

    @NotNull
    @Override
    public Predicate7<Object, Object, Object, Object, Object, Object, Object> partial8(Object argument)
    {
      return FALSE7;
    }
  };

  /**
   * Nine argument predicate which always evaluates to {@code true}.
   * @see Predicate9#alwaysTrue()
   */
  static Predicate9<Object, Object, Object, Object, Object, Object, Object, Object, Object> TRUE9 = new Predicate9<Object, Object, Object, Object, Object, Object, Object, Object, Object>()
  {
    @Override
    public boolean test(Object arg1, Object arg2, Object arg3, Object arg4, Object arg5, Object arg6, Object arg7, Object arg8, Object arg9)
    {
      return true;
    }

    @NotNull
    @Override
    public Predicate9<Object, Object, Object, Object, Object, Object, Object, Object, Object> and(@NotNull Predicate9<? super Object, ? super Object, ? super Object, ? super Object, ? super Object, ? super Object, ? super Object, ? super Object, ? super Object> other)
    {
      return other;
    }

    @NotNull
    @Override
    public Predicate9<Object, Object, Object, Object, Object, Object, Object, Object, Object> or(@NotNull Predicate9<? super Object, ? super Object, ? super Object, ? super Object, ? super Object, ? super Object, ? super Object, ? super Object, ? super Object> other)
    {
      return this;
    }

    @NotNull
    @Override
    public Predicate9<Object, Object, Object, Object, Object, Object, Object, Object, Object> xor(@NotNull Predicate9<? super Object, ? super Object, ? super Object, ? super Object, ? super Object, ? super Object, ? super Object, ? super Object, ? super Object> other)
    {
      return other.negate();
    }

    @NotNull
    @Override
    public Predicate9<Object, Object, Object, Object, Object, Object, Object, Object, Object> negate()
    {
      return FALSE9;
    }

    @NotNull
    @Override
    public Predicate8<Object, Object, Object, Object, Object, Object, Object, Object> partial1(Object argument)
    {
      return TRUE8;
    }

    @NotNull
    @Override
    public Predicate8<Object, Object, Object, Object, Object, Object, Object, Object> partial2(Object argument)
    {
      return TRUE8;
    }

    @NotNull
    @Override
    public Predicate8<Object, Object, Object, Object, Object, Object, Object, Object> partial3(Object argument)
    {
      return TRUE8;
    }

    @NotNull
    @Override
    public Predicate8<Object, Object, Object, Object, Object, Object, Object, Object> partial4(Object argument)
    {
      return TRUE8;
    }

    @NotNull
    @Override
    public Predicate8<Object, Object, Object, Object, Object, Object, Object, Object> partial5(Object argument)
    {
      return TRUE8;
    }

    @NotNull
    @Override
    public Predicate8<Object, Object, Object, Object, Object, Object, Object, Object> partial6(Object argument)
    {
      return TRUE8;
    }

    @NotNull
    @Override
    public Predicate8<Object, Object, Object, Object, Object, Object, Object, Object> partial7(Object argument)
    {
      return TRUE8;
    }

    @NotNull
    @Override
    public Predicate8<Object, Object, Object, Object, Object, Object, Object, Object> partial8(Object argument)
    {
      return TRUE8;
    }

    @NotNull
    @Override
    public Predicate8<Object, Object, Object, Object, Object, Object, Object, Object> partial9(Object argument)
    {
      return TRUE8;
    }
  };
  /**
   * Nine argument predicate which always evaluates to {@code false}.
   * @see Predicate9#alwaysFalse()
   */
  static Predicate9<Object, Object, Object, Object, Object, Object, Object, Object, Object> FALSE9 = new Predicate9<Object, Object, Object, Object, Object, Object, Object, Object, Object>()
  {
    @Override
    public boolean test(Object arg1, Object arg2, Object arg3, Object arg4, Object arg5, Object arg6, Object arg7, Object arg8, Object arg9)
    {
      return false;
    }

    @NotNull
    @Override
    public Predicate9<Object, Object, Object, Object, Object, Object, Object, Object, Object> and(@NotNull Predicate9<? super Object, ? super Object, ? super Object, ? super Object, ? super Object, ? super Object, ? super Object, ? super Object, ? super Object> other)
    {
      return this;
    }

    @NotNull
    @Override
    public Predicate9<Object, Object, Object, Object, Object, Object, Object, Object, Object> or(@NotNull Predicate9<? super Object, ? super Object, ? super Object, ? super Object, ? super Object, ? super Object, ? super Object, ? super Object, ? super Object> other)
    {
      return other;
    }

    @NotNull
    @Override
    public Predicate9<Object, Object, Object, Object, Object, Object, Object, Object, Object> xor(@NotNull Predicate9<? super Object, ? super Object, ? super Object, ? super Object, ? super Object, ? super Object, ? super Object, ? super Object, ? super Object> other)
    {
      return other;
    }

    @NotNull
    @Override
    public Predicate9<Object, Object, Object, Object, Object, Object, Object, Object, Object> negate()
    {
      return TRUE9;
    }

    @NotNull
    @Override
    public Predicate8<Object, Object, Object, Object, Object, Object, Object, Object> partial1(Object argument)
    {
      return FALSE8;
    }

    @NotNull
    @Override
    public Predicate8<Object, Object, Object, Object, Object, Object, Object, Object> partial2(Object argument)
    {
      return FALSE8;
    }

    @NotNull
    @Override
    public Predicate8<Object, Object, Object, Object, Object, Object, Object, Object> partial3(Object argument)
    {
      return FALSE8;
    }

    @NotNull
    @Override
    public Predicate8<Object, Object, Object, Object, Object, Object, Object, Object> partial4(Object argument)
    {
      return FALSE8;
    }

    @NotNull
    @Override
    public Predicate8<Object, Object, Object, Object, Object, Object, Object, Object> partial5(Object argument)
    {
      return FALSE8;
    }

    @NotNull
    @Override
    public Predicate8<Object, Object, Object, Object, Object, Object, Object, Object> partial6(Object argument)
    {
      return FALSE8;
    }

    @NotNull
    @Override
    public Predicate8<Object, Object, Object, Object, Object, Object, Object, Object> partial7(Object argument)
    {
      return FALSE8;
    }

    @NotNull
    @Override
    public Predicate8<Object, Object, Object, Object, Object, Object, Object, Object> partial8(Object argument)
    {
      return FALSE8;
    }

    @NotNull
    @Override
    public Predicate8<Object, Object, Object, Object, Object, Object, Object, Object> partial9(Object argument)
    {
      return FALSE8;
    }
  };

  /** Predicate2 which uses {@link Objects#deepEquals(Object, Object)} for checking for equality. */
  @NotNull
  static Predicate2<Object, Object> DEEP_EQUALS = Objects::deepEquals;

  /** Predicate2 which checks whether two objects represent the same object. */
  @NotNull
  static Predicate2<Object, Object> IDENTICAL = (o1, o2) -> o1 == o2;

  /** Don't create. */
  private Predicates() {}


}
