// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.function;

import de.caff.annotation.NotNull;
import de.caff.generics.tuple.ITuple8;

import java.util.function.BiFunction;
import java.util.function.Function;

/**
 * A function with eight arguments which returns a boolean value.
 * <p>
 * Predicates can be chained to simulate boolean logic by
 * {@link #and(Predicate8)}, {@link #or(Predicate8)},
 * {@link #xor(Predicate8)}, and {@link #negate()}.
 * <p>
 * There is a general problem with combining types which don't
 * allow to use the above combination methods in all cases
 * similar to standard Java's logical expressions. The following
 * static methods of this interface may be used in all cases:
 * {@link #and(Predicate8, Predicate8)},
 * {@link #or(Predicate8, Predicate8)},
 * and {@link #xor(Predicate8, Predicate8)}.
 * <p>
 * Useful for start values in algorithms or as default values
 * are the two values returned by {@link #alwaysTrue()}
 * (a predicate which always returns {@code true} regardless
 * the argument applied to the {@link #test(Object, Object, Object, Object, Object, Object, Object, Object)} method)
 * and {@link #alwaysFalse()} (same for {@code false}).
 * <p>
 * Partial evaluation is also possible.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @param <P1> parameter type 1
 * @param <P2> parameter type 2
 * @param <P3> parameter type 3
 * @param <P4> parameter type 4
 * @param <P5> parameter type 5
 * @param <P6> parameter type 6
 * @param <P7> parameter type 7
 * @param <P8> parameter type 8
 * @see Predicate1
 * @see Predicate2
 * @see Predicate3
 * @see Predicate4
 * @see Predicate5
 * @see Predicate6
 * @see Predicate7
 * @see Predicate9
 */
@FunctionalInterface
public interface Predicate8<P1, P2, P3, P4, P5, P6, P7, P8>
        extends Predicate1<ITuple8<? extends P1, ? extends P2, ? extends P3,
                                   ? extends P4, ? extends P5, ? extends P6,
                                   ? extends P7, ? extends P8>>
{
  /**
   * Test the given values.
   * @param arg1 function argument 1
   * @param arg2 function argument 2
   * @param arg3 function argument 3
   * @param arg4 function argument 4
   * @param arg5 function argument 5
   * @param arg6 function argument 6
   * @param arg7 function argument 7
   * @param arg8 function argument 8
   * @return {@code true} or {@code false}
   */
  boolean test(P1 arg1, P2 arg2, P3 arg3, P4 arg4, P5 arg5, P6 arg6, P7 arg7, P8 arg8);

  /**
   * Expand a tuple into the arguments of this predicate.
   * @param tuple8 tuple to expand
   * @return the result of testing the expended eleeents of the tuple
   */
  @Override
  default boolean test(@NotNull ITuple8<? extends P1, ? extends P2, ? extends P3, ? extends P4, ? extends P5, ? extends P6, ? extends P7, ? extends P8> tuple8)
  {
    return tuple8.testBy(this);
  }

  /**
   * Combine this and another predicate with logical and.
   * The result will use shortcut evaluation, so if this
   * predicate already returns {@code false} the other one is
   * not evaluated.
   * <p>
   * This method has a problem if either one of the argument types
   * pf the parameter is not a super class of this predicate's
   * parameter types. See {@link #and(Predicate8, Predicate8)}
   * for such cases.
   *
   * @param other other predicate to be chained with logical and
   * @return predicates combined with logical and
   */
  @NotNull
  default Predicate8<P1, P2, P3, P4, P5, P6, P7, P8> and(@NotNull Predicate8<? super P1, ? super P2, ? super P3, ? super P4, ? super P5, ? super P6, ? super P7, ? super P8> other)
  {
    return (p1, p2, p3, p4, p5, p6, p7, p8) -> test(p1, p2, p3, p4, p5, p6, p7, p8) && other.test(p1, p2, p3, p4, p5, p6, p7, p8);
  }

  /**
   * Combine this and another predicate with logical or.
   * The result will use shortcut evaluation, so if this
   * predicate already returns {@code true} the other one is
   * not evaluated.
   * <p>
   * This method has a problem if either one of the argument types
   * pf the parameter is not a super class of this predicate's
   * parameter types. See {@link #or(Predicate8, Predicate8)}
   * for such cases.
   *
   * @param other other predicate to be chained with logical or
   * @return predicates combined with logical or
   */
  @NotNull
  default Predicate8<P1, P2, P3, P4, P5, P6, P7, P8> or(@NotNull Predicate8<? super P1, ? super P2, ? super P3, ? super P4, ? super P5, ? super P6, ? super P7, ? super P8> other)
  {
    return (p1, p2, p3, p4, p5, p6, p7, p8) -> test(p1, p2, p3, p4, p5, p6, p7, p8) || other.test(p1, p2, p3, p4, p5, p6, p7, p8);
  }

  /**
   * Combine this and another predicate with logical exclusive or.
   * <p>
   * This method has a problem if either one of the argument types
   * pf the parameter is not a super class of this predicate's
   * parameter types. See {@link #xor(Predicate8, Predicate8)}
   * for such cases.
   *
   * @param other other predicate to be chained with logical exclusive or
   * @return predicates combined with logical exclusive or
   */
  @NotNull
  default Predicate8<P1, P2, P3, P4, P5, P6, P7, P8> xor(@NotNull Predicate8<? super P1, ? super P2, ? super P3, ? super P4, ? super P5, ? super P6, ? super P7, ? super P8> other)
  {
    return (p1, p2, p3, p4, p5, p6, p7, p8) -> test(p1, p2, p3, p4, p5, p6, p7, p8) ^ other.test(p1, p2, p3, p4, p5, p6, p7, p8);
  }

  /**
   * Get the negation of this predicate.
   * @return negation of this predicate
   */
  @NotNull
  default Predicate8<P1, P2, P3, P4, P5, P6, P7, P8> negate()
  {
    return new Predicate8<P1, P2, P3, P4, P5, P6, P7, P8>()
    {
      @Override
      public boolean test(P1 arg1, P2 arg2, P3 arg3, P4 arg4, P5 arg5, P6 arg6, P7 arg7, P8 arg8)
      {
        return !Predicate8.this.test(arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8);
      }

      @NotNull
      @Override
      public Predicate8<P1, P2, P3, P4, P5, P6, P7, P8> negate()
      {
        return Predicate8.this;
      }
    };
  }

  /**
   * Get a partially applied predicate with a fix first argument.
   * @param argument value applied for first argument
   * @return partially evaluated two argument predicate
   */
  @NotNull
  default Predicate7<P2, P3, P4, P5, P6, P7, P8> partial1(P1 argument)
  {
    return (arg2, arg3, arg4, arg5, arg6, arg7, arg8) -> test(argument, arg2, arg3, arg4, arg5, arg6, arg7, arg8);
  }

  /**
   * Get a partially applied predicate with a fix second argument.
   * @param argument value applied for second argument
   * @return partially evaluated two argument predicate
   */
  @NotNull
  default Predicate7<P1, P3, P4, P5, P6, P7, P8> partial2(P2 argument)
  {
    return (arg1, arg3, arg4, arg5, arg6, arg7, arg8) -> test(arg1, argument, arg3, arg4, arg5, arg6, arg7, arg8);
  }

  /**
   * Get a partially applied predicate with a fix third argument.
   * @param argument value applied for third argument
   * @return partially evaluated two argument predicate
   */
  @NotNull
  default Predicate7<P1, P2, P4, P5, P6, P7, P8> partial3(P3 argument)
  {
    return (arg1, arg2, arg4, arg5, arg6, arg7, arg8) -> test(arg1, arg2, argument, arg4, arg5, arg6, arg7, arg8);
  }

  /**
   * Get a partially applied predicate with a fix fourth argument.
   * @param argument value applied for fourth argument
   * @return partially evaluated two argument predicate
   */
  @NotNull
  default Predicate7<P1, P2, P3, P5, P6, P7, P8> partial4(P4 argument)
  {
    return (arg1, arg2, arg3, arg5, arg6, arg7, arg8) -> test(arg1, arg2, arg3, argument, arg5, arg6, arg7, arg8);
  }

  /**
   * Get a partially applied predicate with a fix fifth argument.
   * @param argument value applied for fifth argument
   * @return partially evaluated two argument predicate
   */
  @NotNull
  default Predicate7<P1, P2, P3, P4, P6, P7, P8> partial5(P5 argument)
  {
    return (arg1, arg2, arg3, arg4, arg6, arg7, arg8) -> test(arg1, arg2, arg3, arg4, argument, arg6, arg7, arg8);
  }

  /**
   * Get a partially applied predicate with a fix sixth argument.
   * @param argument value applied for sixth argument
   * @return partially evaluated two argument predicate
   */
  @NotNull
  default Predicate7<P1, P2, P3, P4, P5, P7, P8> partial6(P6 argument)
  {
    return (arg1, arg2, arg3, arg4, arg5, arg7, arg8) -> test(arg1, arg2, arg3, arg4, arg5, argument, arg7, arg8);
  }

  /**
   * Get a partially applied predicate with a fix seventh argument.
   * @param argument value applied for seventh argument
   * @return partially evaluated two argument predicate
   */
  @NotNull
  default Predicate7<P1, P2, P3, P4, P5, P6, P8> partial7(P7 argument)
  {
    return (arg1, arg2, arg3, arg4, arg5, arg6, arg8) -> test(arg1, arg2, arg3, arg4, arg5, arg6, argument, arg8);
  }

  /**
   * Get a partially applied predicate with a fix eighth argument.
   * @param argument value applied for eighth argument
   * @return partially evaluated two argument predicate
   */
  @NotNull
  default Predicate7<P1, P2, P3, P4, P5, P6, P7> partial8(P8 argument)
  {
    return (arg1, arg2, arg3, arg4, arg5, arg6, arg7) -> test(arg1, arg2, arg3, arg4, arg5, arg6, arg7, argument);
  }

  /**
   * Get a predicate which always evaluates to {@code true}.
   * @param <T1> type of first predicate parameter
   * @param <T2> type of second predicate parameter
   * @param <T3> type of third predicate parameter
   * @param <T4> type of fourth predicate parameter
   * @param <T5> type of fifth predicate parameter
   * @param <T6> type of sixth predicate parameter
   * @param <T7> type of seventh predicate parameter
   * @param <T8> type of eighth predicate parameter
   * @return predicate always returning {@code true}
   */
  @NotNull
  @SuppressWarnings("unchecked") // predicate ignores the object
  static <T1, T2, T3, T4, T5, T6, T7, T8> Predicate8<T1, T2, T3, T4, T5, T6, T7, T8> alwaysTrue()
  {
    return (Predicate8<T1, T2, T3, T4, T5, T6, T7, T8>)Predicates.TRUE8;
  }

  /**
   * Get a predicate which always evaluates to {@code false}.
   * @param <T1> type of first predicate parameter
   * @param <T2> type of second predicate parameter
   * @param <T3> type of third predicate parameter
   * @param <T4> type of fourth predicate parameter
   * @param <T5> type of fifth predicate parameter
   * @param <T6> type of sixth predicate parameter
   * @param <T7> type of seventh predicate parameter
   * @param <T8> type of eighth predicate parameter
   * @return predicate always returning {@code false}
   */
  @NotNull
  @SuppressWarnings("unchecked") // predicate ignores the object
  static <T1, T2, T3, T4, T5, T6, T7, T8> Predicate8<T1, T2, T3, T4, T5, T6, T7, T8> alwaysFalse()
  {
    return (Predicate8<T1, T2, T3, T4, T5, T6, T7, T8>)Predicates.FALSE8;
  }

  /**
   * Get a predicate which is the combination of two predicates
   * with a logical and.
   * <p>
   * The result will use shortcut evaluation, so if this
   * predicate already returns {@code false} the other one is
   * not evaluated.
   *
   * @param pred1 first predicate
   * @param pred2 second predicate
   * @param <T1> resulting type of first argument,
   *            which is the most extending type of both {@code pred1}'s
   *            first argument type and {@code pred2}'s first argument type,
   *            so if neither type is extending the
   *            other this method will not compile
   * @param <T2> resulting type of second argument, see {@code <T1>} for the fine print
   * @param <T3> resulting type of third argument, see {@code <T1>} for the fine print
   * @param <T4> resulting type of fourth argument, see {@code <T1>} for the fine print
   * @param <T5> resulting type of fifth argument, see {@code <T1>} for the fine print
   * @param <T6> resulting type of sixth argument, see {@code <T1>} for the fine print
   * @param <T7> resulting type of seventh argument, see {@code <T1>} for the fine print
   * @param <T8> resulting type of eighth argument, see {@code <T1>} for the fine print
   * @return combination of both predicates with logical and
   * @see #and(Predicate8)
   */
  @NotNull
  static <T1, T2, T3, T4, T5, T6, T7, T8> Predicate8<T1, T2, T3, T4, T5, T6, T7, T8>
  and(@NotNull Predicate8<? super T1, ? super T2, ? super T3, ? super T4, ? super T5, ? super T6, ? super T7, ? super T8> pred1,
      @NotNull Predicate8<? super T1, ? super T2, ? super T3, ? super T4, ? super T5, ? super T6, ? super T7, ? super T8> pred2)
  {
    return (p1, p2, p3, p4, p5, p6, p7, p8) -> pred1.test(p1, p2, p3, p4, p5, p6, p7, p8) && pred2.test(p1, p2, p3, p4, p5, p6, p7, p8);
  }

  /**
   * Get a predicate which is the combination of two predicates
   * with a logical or.
   * <p>
   * The result will use shortcut evaluation, so if this
   * predicate already returns {@code true} the other one is
   * not evaluated.
   * @param pred1 first predicate
   * @param pred2 second predicate
   * @param <T1> resulting type of first argument,
   *            which is the most extending type of both {@code pred1}'s
   *            first argument type and {@code pred2}'s first argument type,
   *            so if neither type is extending the
   *            other this method will not compile
   * @param <T2> resulting type of second argument, see {@code <T1>} for the fine print
   * @param <T3> resulting type of third argument, see {@code <T1>} for the fine print
   * @param <T4> resulting type of fourth argument, see {@code <T1>} for the fine print
   * @param <T5> resulting type of fifth argument, see {@code <T1>} for the fine print
   * @param <T6> resulting type of sixth argument, see {@code <T1>} for the fine print
   * @param <T7> resulting type of seventh argument, see {@code <T1>} for the fine print
   * @param <T8> resulting type of eighth argument, see {@code <T1>} for the fine print
   * @return combination of both predicates with logical or
   * @see #or(Predicate8)
   */
  @NotNull
  static <T1, T2, T3, T4, T5, T6, T7, T8> Predicate8<T1, T2, T3, T4, T5, T6, T7, T8>
  or(@NotNull Predicate8<? super T1, ? super T2, ? super T3, ? super T4, ? super T5, ? super T6, ? super T7, ? super T8> pred1,
     @NotNull Predicate8<? super T1, ? super T2, ? super T3, ? super T4, ? super T5, ? super T6, ? super T7, ? super T8> pred2)
  {
    return (p1, p2, p3, p4, p5, p6, p7, p8) -> pred1.test(p1, p2, p3, p4, p5, p6, p7, p8) || pred2.test(p1, p2, p3, p4, p5, p6, p7, p8);
  }

  /**
   * Get a predicate which is the combination of two predicates
   * with a logical exclusive or.
   * <p>
   * The result will use shortcut evaluation, so if this
   * predicate already returns {@code true} the other one is
   * not evaluated.
   *
   * @param pred1 first predicate
   * @param pred2 second predicate
   * @param <T1> resulting type of first argument,
   *            which is the most extending type of both {@code pred1}'s
   *            first argument type and {@code pred2}'s first argument type,
   *            so if neither type is extending the
   *            other this method will not compile
   * @param <T2> resulting type of second argument, see {@code <T1>} for the fine print
   * @param <T3> resulting type of third argument, see {@code <T1>} for the fine print
   * @param <T4> resulting type of fourth argument, see {@code <T1>} for the fine print
   * @param <T5> resulting type of fifth argument, see {@code <T1>} for the fine print
   * @param <T6> resulting type of sixth argument, see {@code <T1>} for the fine print
   * @param <T7> resulting type of seventh argument, see {@code <T1>} for the fine print
   * @param <T8> resulting type of eighth argument, see {@code <T1>} for the fine print
   * @return combination of both predicates with logical exclusive or
   * @see #xor(Predicate8)
   */
  @NotNull
  static <T1, T2, T3, T4, T5, T6, T7, T8> Predicate8<T1, T2, T3, T4, T5, T6, T7, T8>
  xor(@NotNull Predicate8<? super T1, ? super T2, ? super T3, ? super T4, ? super T5, ? super T6, ? super T7, ? super T8> pred1,
      @NotNull Predicate8<? super T1, ? super T2, ? super T3, ? super T4, ? super T5, ? super T6, ? super T7, ? super T8> pred2)
  {
    return (p1, p2, p3, p4, p5, p6, p7, p8) -> pred1.test(p1, p2, p3, p4, p5, p6, p7, p8) ^ pred2.test(p1, p2, p3, p4, p5, p6, p7, p8);
  }

  /**
   * Get a chained predicate which first applies the given function, then this predicate.
   * The given function has to return a {@link ITuple8} with matching parametric types
   * which are applied in their order to this predicate.
   * @param before function to apply first, has to return a non-{@code null} result so unpacking the parameters
   *               from the tuple is possible
   * @param <FR> function {@code before}'s return type, has to be a 8-tuple with matching types
   * @param <FP> function {@code before}'s parameter type, same for returned predicate
   * @return chain of function and this predicate
   */
  @NotNull
  default <FR extends ITuple8<? extends P1, ? extends P2, ? extends P3, ? extends P4, ? extends P5, ? extends P6, ? extends P7, ? extends P8>, FP>
  Predicate1<FP> after1(@NotNull Function<FP, FR> before)
  {
    return fp -> before.apply(fp).testBy(this);
  }

  /**
   * Get a chained predicate which first applies the given 2-parameter function, then this predicate.
   * The given function has to return a {@link ITuple8} with matching parametric types
   * which are applied in their order to this predicate.
   * @param before function to apply first, has to return a non-{@code null} result so unpacking the parameters
   *               from the tuple is possible
   * @param <FR> function {@code before}'s return type, has to be a 8-tuple with matching types
   * @param <FP1> function {@code before}'s first parameter type, sane for returned predicate
   * @param <FP2> function {@code before}'s second parameter type, same for returned predicate
   * @return chain of the given function and this predicate
   */
  @NotNull
  default <FR extends ITuple8<? extends P1, ? extends P2, ? extends P3, ? extends P4, ? extends P5, ? extends P6, ? extends P7, ? extends P8>, FP1, FP2>
  Predicate2<FP1, FP2> after2(@NotNull BiFunction<FP1, FP2, FR> before)
  {
    return (fp1, fp2) -> before.apply(fp1, fp2).testBy(this);
  }

  /**
   * Get a chained predicate which first applies the given 3-parameter function, then this predicate.
   * The given function has to return a {@link ITuple8} with matching parametric types
   * which are applied in their order to this predicate.
   * @param before function to apply first, has to return a non-{@code null} result so unpacking the parameters
   *               from the tuple is possible
   * @param <FR> function {@code before}'s return type, has to be a 8-tuple with matching types
   * @param <FP1> function {@code before}'s first parameter type, sane for returned predicate
   * @param <FP2> function {@code before}'s second parameter type, sane for returned predicate
   * @param <FP3> function {@code before}'s third parameter type, sane for returned predicate
   * @return chain of the given function and this predicate
   */
  @NotNull
  default <FR extends ITuple8<? extends P1, ? extends P2, ? extends P3, ? extends P4, ? extends P5, ? extends P6, ? extends P7, ? extends P8>, FP1, FP2, FP3>
  Predicate3<FP1, FP2, FP3> after3(@NotNull Function3<FR, FP1, FP2, FP3> before)
  {
    return (fp1, fp2, fp3) -> before.apply(fp1, fp2, fp3).testBy(this);
  }

  /**
   * Get a chained predicate which first applies the given 5-parameter function, then this predicate.
   * The given function has to return a {@link ITuple8} with matching parametric types
   * which are applied in their order to this function.
   * @param before function to apply first, has to return a non-{@code null} result so unpacking the parameters
   *               from the tuple is possible
   * @param <FR> function {@code before}'s return type, has to be a 8-tuple with matching types
   * @param <FP1> function {@code before}'s first parameter type, sane for returned predicate
   * @param <FP2> function {@code before}'s second parameter type, sane for returned predicate
   * @param <FP3> function {@code before}'s third parameter type, sane for returned predicate
   * @param <FP4> function {@code before}'s fourth parameter type, sane for returned predicate
   * @param <FP5> function {@code before}'s fifth parameter type, sane for returned predicate
   * @return chain of the given function and this predicate
   */
  @NotNull
  default <FR extends ITuple8<? extends P1, ? extends P2, ? extends P3, ? extends P4, ? extends P5, ? extends P6, ? extends P7, ? extends P8>, FP1, FP2, FP3, FP4, FP5>
  Predicate5<FP1, FP2, FP3, FP4, FP5> after5(@NotNull Function5<FR, FP1, FP2, FP3, FP4, FP5> before)
  {
    return (fp1, fp2, fp3, fp4, fp5) -> before.apply(fp1, fp2, fp3, fp4, fp5).testBy(this);
  }

  /**
   * Get a chained predicate which first applies the given 6-parameter function, then this predicate.
   * The given function has to return a {@link ITuple8} with matching parametric types
   * which are applied in their order to this function.
   * @param before function to apply first, has to return a non-{@code null} result so unpacking the parameters
   *               from the tuple is possible
   * @param <FR> function {@code before}'s return type, has to be a 8-tuple with matching types
   * @param <FP1> function {@code before}'s first parameter type, sane for returned predicate
   * @param <FP2> function {@code before}'s second parameter type, sane for returned predicate
   * @param <FP3> function {@code before}'s third parameter type, sane for returned predicate
   * @param <FP4> function {@code before}'s fourth parameter type, sane for returned predicate
   * @param <FP5> function {@code before}'s fifth parameter type, sane for returned predicate
   * @param <FP6> function {@code before}'s sixth parameter type, sane for returned predicate
   * @return chain of the given function and this predicate
   */
  @NotNull
  default <FR extends ITuple8<? extends P1, ? extends P2, ? extends P3, ? extends P4, ? extends P5, ? extends P6, ? extends P7, ? extends P8>, FP1, FP2, FP3, FP4, FP5, FP6>
  Predicate6<FP1, FP2, FP3, FP4, FP5, FP6> after6(@NotNull Function6<FR, FP1, FP2, FP3, FP4, FP5, FP6> before)
  {
    return (fp1, fp2, fp3, fp4, fp5, fp6) -> before.apply(fp1, fp2, fp3, fp4, fp5, fp6).testBy(this);
  }

  /**
   * Get a chained predicate which first applies the given 7-parameter function, then this one.
   * The given function has to return a {@link ITuple8} with matching parametric types
   * which are applied in their order to this predicate.
   * @param before function to apply first, has to return a non-{@code null} result so unpacking the parameters
   *               from the tuple is possible
   * @param <FR> function {@code before}'s return type, has to be a 8-tuple with matching types
   * @param <FP1> function {@code before}'s first parameter type, sane for returned predicate
   * @param <FP2> function {@code before}'s second parameter type, sane for returned predicate
   * @param <FP3> function {@code before}'s third parameter type, sane for returned predicate
   * @param <FP4> function {@code before}'s fourth parameter type, sane for returned predicate
   * @param <FP5> function {@code before}'s fifth parameter type, sane for returned predicate
   * @param <FP6> function {@code before}'s sixth parameter type, sane for returned predicate
   * @param <FP7> function {@code before}'s seventh parameter type, sane for returned predicate
   * @return chain of the given function and this predicate
   */
  @NotNull
  default <FR extends ITuple8<? extends P1, ? extends P2, ? extends P3, ? extends P4, ? extends P5, ? extends P6, ? extends P7, ? extends P8>, FP1, FP2, FP3, FP4, FP5, FP6, FP7>
  Predicate7<FP1, FP2, FP3, FP4, FP5, FP6, FP7> after7(@NotNull Function7<FR, FP1, FP2, FP3, FP4, FP5, FP6, FP7> before)
  {
    return (fp1, fp2, fp3, fp4, fp5, fp6, fp7) -> before.apply(fp1, fp2, fp3, fp4, fp5, fp6, fp7).testBy(this);
  }

  /**
   * Get a chained predicate which first applies the given 8-parameter function, then this one.
   * The given function has to return a {@link ITuple8} with matching parametric types
   * which are applied in their order to this predicate.
   * @param before function to apply first, has to return a non-{@code null} result so unpacking the parameters
   *               from the tuple is possible
   * @param <FR> function {@code before}'s return type, has to be a 8-tuple with matching types
   * @param <FP1> function {@code before}'s first parameter type, sane for returned predicate
   * @param <FP2> function {@code before}'s second parameter type, sane for returned predicate
   * @param <FP3> function {@code before}'s third parameter type, sane for returned predicate
   * @param <FP4> function {@code before}'s fourth parameter type, sane for returned predicate
   * @param <FP5> function {@code before}'s fifth parameter type, sane for returned predicate
   * @param <FP6> function {@code before}'s sixth parameter type, sane for returned predicate
   * @param <FP7> function {@code before}'s seventh parameter type, sane for returned predicate
   * @param <FP8> function {@code before}'s seventh parameter type, sane for returned predicate
   * @return chain of the given function and this predicate
   */
  @NotNull
  default <FR extends ITuple8<? extends P1, ? extends P2, ? extends P3, ? extends P4, ? extends P5, ? extends P6, ? extends P7, ? extends P8>, FP1, FP2, FP3, FP4, FP5, FP6, FP7, FP8>
  Predicate8<FP1, FP2, FP3, FP4, FP5, FP6, FP7, FP8> after8(@NotNull Function8<FR, FP1, FP2, FP3, FP4, FP5, FP6, FP7, FP8> before)
  {
    return (fp1, fp2, fp3, fp4, fp5, fp6, fp7, fp8) -> before.apply(fp1, fp2, fp3, fp4, fp5, fp6, fp7, fp8).testBy(this);
  }

  /**
   * Get a chained predicate which first applies the given 9-parameter function, then this one.
   * The given function has to return a {@link ITuple8} with matching parametric types
   * which are applied in their order to this predicate.
   * @param before function to apply first, has to return a non-{@code null} result so unpacking the parameters
   *               from the tuple is possible
   * @param <FR> function {@code before}'s return type, has to be a 8-tuple with matching types
   * @param <FP1> function {@code before}'s first parameter type, sane for returned predicate
   * @param <FP2> function {@code before}'s second parameter type, sane for returned predicate
   * @param <FP3> function {@code before}'s third parameter type, sane for returned predicate
   * @param <FP4> function {@code before}'s fourth parameter type, sane for returned predicate
   * @param <FP5> function {@code before}'s fifth parameter type, sane for returned predicate
   * @param <FP6> function {@code before}'s sixth parameter type, sane for returned predicate
   * @param <FP7> function {@code before}'s seventh parameter type, sane for returned predicate
   * @param <FP8> function {@code before}'s seventh parameter type, sane for returned predicate
   * @param <FP9> function {@code before}'s ninth parameter type, sane for returned predicate
   * @return chain of the given function and this predicate
   */
  @NotNull
  default <FR extends ITuple8<? extends P1, ? extends P2, ? extends P3, ? extends P4, ? extends P5, ? extends P6, ? extends P7, ? extends P8>, FP1, FP2, FP3, FP4, FP5, FP6, FP7, FP8, FP9>
  Predicate9<FP1, FP2, FP3, FP4, FP5, FP6, FP7, FP8, FP9> after9(@NotNull Function9<FR, FP1, FP2, FP3, FP4, FP5, FP6, FP7, FP8, FP9> before)
  {
    return (fp1, fp2, fp3, fp4, fp5, fp6, fp7, fp8, fp9) -> before.apply(fp1, fp2, fp3, fp4, fp5, fp6, fp7, fp8, fp9).testBy(this);
  }
}
