// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.function;

import de.caff.annotation.NotNull;
import de.caff.generics.tuple.ITuple2;

import java.util.Objects;
import java.util.function.BiFunction;
import java.util.function.BiPredicate;
import java.util.function.Function;

/**
 * A function with two arguments which returns a boolean value.
 * <p>
 * Predicates can be chained to simulate boolean logic by
 * {@link #and(BiPredicate)}, {@link #or(BiPredicate)},
 * {@link #xor(BiPredicate)}, and {@link #negate()}.
 * <p>
 * There is a general problem with combining types which don't
 * allow to use the above combination methods in all cases
 * similar to standard Java's logical expressions. The following
 * static methods of this interface may be used in all cases:
 * {@link #and(BiPredicate, BiPredicate)},
 * {@link #or(BiPredicate, BiPredicate)},
 * and {@link #xor(BiPredicate, BiPredicate)}.
 * <p>
 * Useful for start values in algorithms or as default values
 * are the two values returned by {@link #alwaysTrue()}
 * (a predicate which always returns {@code true} regardless
 * the argument applied to the {@link #test(Object, Object)} method)
 * and {@link #alwaysFalse()} (same for {@code false}).
 * <p>
 * Partial evaluation is possible via {@link #partialLeft(Object)}
 * which allows to provide a fix first argument and
 * {@link #partialLeft(Object)} which provides a fix second argument.
 * Both return a {@link Predicate1} with the type of the other
 * argument.
 * <p>
 * If a predicate with arguments in reverse order is required
 * {@link #reverseOrder()} comes to help.
 * <p>
 * Compared to the standard Java {@link BiPredicate}
 * this implementation has an
 * improved {@link #negate()} and provides optimized
 * {@link #alwaysTrue()} and {@link #alwaysFalse()}
 * implementations plus an additional {@link #xor(BiPredicate)}.
 * It also allows to use a predicate where the arguments are expected
 * in {@link #reverseOrder() reverse order} or to use this
 * 2 argument predicate where only 1 argument is expected by providing
 * a fix value for either {@link #partialLeft(Object) first}
 * or the {@link #partialRight(Object) second} argument.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @param <P1> parameter type 1
 * @param <P2> parameter type 2
 * @see Predicate1
 * @see Predicate3
 * @see Predicate4
 * @see Predicate5
 * @see Predicate6
 * @see Predicate7
 * @see Predicate8
 * @see Predicate9
 */
@FunctionalInterface
public interface Predicate2<P1, P2>
        extends BiPredicate<P1, P2>,
                Predicate1<ITuple2<? extends P1, ? extends P2>>
{
  /**
   * Expand a tuple into the arguments of this predicate.
   * @param tuple2 tuple to expand
   * @return the result of testing the expended eleeents of the tuple
   */
  @Override
  default boolean test(@NotNull ITuple2<? extends P1, ? extends P2> tuple2)
  {
    return tuple2.testBy(this);
  }

  /**
   * Test the given.
   * This is outdated.
   * @param arg1 function argument 1
   * @param arg2 function argument 2
   * @return {@code true} or {@code false}
   * @deprecated This has been renamed to {@link #test(Object, Object)} to be more in sync with standard Java
   */
  @Deprecated
  default boolean apply(P1 arg1, P2 arg2)
  {
    return test(arg1, arg2);
  }

  /**
   * Combine this and another predicate with logical and.
   * The result will use shortcut evaluation, so if this
   * predicate already returns {@code false} the other one is
   * not evaluated.
   * <p>
   * This method has a problem if either one of the argument types
   * pf the parameter is not a super class of this predicate's
   * parameter types. See {@link #and(BiPredicate, BiPredicate)}
   * for such cases.
   *
   * @param other other predicate to be chained with logical and
   * @return predicates combined with logical and
   */
  @NotNull
  default Predicate2<P1, P2> and(@NotNull BiPredicate<? super P1, ? super P2> other)
  {
    return (p1, p2) -> test(p1, p2) && other.test(p1, p2);
  }

  /**
   * Combine this and another predicate with logical or.
   * The result will use shortcut evaluation, so if this
   * predicate already returns {@code true} the other one is
   * not evaluated.
   * <p>
   * This method has a problem if either one of the argument types
   * pf the parameter is not a super class of this predicate's
   * parameter types. See {@link #or(BiPredicate, BiPredicate)}
   * for such cases.
   *
   * @param other other predicate to be chained with logical or
   * @return predicates combined with logical or
   */
  @NotNull
  default Predicate2<P1, P2> or(@NotNull BiPredicate<? super P1, ? super P2> other)
  {
    return (p1, p2) -> test(p1, p2) || other.test(p1, p2);
  }

  /**
   * Combine this and another predicate with logical exclusive or.
   * <p>
   * This method has a problem if either one of the argument types
   * pf the parameter is not a super class of this predicate's
   * parameter types. See {@link #xor(BiPredicate, BiPredicate)}
   * for such cases.
   *
   * @param other other predicate to be chained with logical exclusive or
   * @return predicates combined with logical exclusive or
   */
  @NotNull
  default Predicate2<P1, P2> xor(@NotNull BiPredicate<? super P1, ? super P2> other)
  {
    return (p1, p2) -> test(p1, p2) ^ other.test(p1, p2);
  }

  /**
   * Get the negation of this predicate.
   * @return negation of this predicate
   */
  @NotNull
  default Predicate2<P1, P2> negate()
  {
    return new Predicate2<P1, P2>()
    {
      @Override
      public boolean test(P1 arg1, P2 arg2)
      {
        return !Predicate2.this.test(arg1, arg2);
      }

      @NotNull
      @Override
      public Predicate2<P1, P2> negate()
      {
        return Predicate2.this;
      }
    };
  }

  /**
   * Get a partially applied predicate.
   * @param argument value applied for first argument
   * @return partially evaluated predicate
   */
  @NotNull
  default Predicate1<P2> partialLeft(P1 argument)
  {
    return partial1(argument);
  }

  /**
   * Get a partially applied predicate using a fix first argument.
   * This is the same as {@link #partialLeft(Object)} but more consistently named.
   * @param argument value applied for first argument
   * @return partially evaluated predicate
   */
  @NotNull
  default Predicate1<P2> partial1(P1 argument)
  {
    return arg -> test(argument, arg);
  }

  /**
   * Get a partially applied predicate.
   * @param argument value applied for second argument
   * @return partially evaluated predicate
   */
  @NotNull
  default Predicate1<P1> partialRight(P2 argument)
  {
    return partial2(argument);
  }

  /**
   * Get a partially applied predicate using a fix second argument.
   * This is the same as {@link #partialRight(Object)} but more consistently named.
   * @param argument value applied for second argument
   * @return partially evaluated predicate
   */
  @NotNull
  default Predicate1<P1> partial2(P2 argument)
  {
    return arg -> test(arg, argument);
  }

  /**
   * Get a two argument predicate which expects the arguments in reverse order.
   * @return predicate with reversed order of arguments
   */
  @NotNull
  default Predicate2<P2, P1> reverseOrder()
  {
    return (arg2, arg1) -> test(arg1, arg2);
  }

  /**
   * Make a standard {@link BiPredicate} usable as a {@code Predicate2}.
   * @param bipred bipredicate
   * @param <T1> type of first argument
   * @param <T2> type of second argument
   * @return {@code Predicate2} view of the given bipredicatge
   */
  @NotNull
  static <T1, T2> Predicate2<T1, T2> from(@NotNull BiPredicate<T1, T2> bipred)
  {
    return bipred instanceof Predicate2
            ? (Predicate2<T1, T2>) bipred
            : bipred::test;
  }

  /**
   * Get a predicate which always evaluates to {@code true}.
   * @param <T1> type of first predicate parameter
   * @param <T2> type of second predicate parameter
   * @return predicate always returning {@code true}
   */
  @NotNull
  @SuppressWarnings("unchecked") // predicate ignores the object
  static <T1, T2> Predicate2<T1, T2> alwaysTrue()
  {
    return (Predicate2<T1, T2>)Predicates.TRUE2;
  }

  /**
   * Get a predicate which always evaluates to {@code false}.
   * @param <T1> type of first predicate parameter
   * @param <T2> type of second predicate parameter
   * @return predicate always returning {@code false}
   */
  @NotNull
  @SuppressWarnings("unchecked") // predicate ignores the object
  static <T1, T2> Predicate2<T1, T2> alwaysFalse()
  {
    return (Predicate2<T1, T2>)Predicates.FALSE2;
  }

  /**
   * Get a predicate which compares 2 objects for deep equality.
   * <p>
   * This internally uses {@link Objects#deepEquals(Object, Object)} for the equality test.
   * @param <T> object type
   * @return predicate for deep equality
   */
  @NotNull
  @SuppressWarnings("unchecked") // always using Object internally
  static <T> Predicate2<T, T> isEqual()
  {
    return (Predicate2<T, T>)Predicates.DEEP_EQUALS;
  }

  /**
   * Get a predicate which checks whether 2 objects represent the same object.
   * @return predicate for checking whether 2 objects represent the same object
   * @param <T> object type
   */
  @NotNull
  @SuppressWarnings("unchecked") // always using Object internally
  static <T> Predicate2<T, T> areIdentical()
  {
    return (Predicate2<T, T>)Predicates.IDENTICAL;
  }

  /**
   * Get a predicate which is the combination of two predicates
   * with a logical and.
   * <p>
   * The result will use shortcut evaluation, so if this
   * predicate already returns {@code false} the other one is
   * not evaluated.
   *
   * @param pred1 first predicate
   * @param pred2 second predicate
   * @param <T1> resulting type of first argument,
   *            which is the most extending type of both {@code pred1}'s
   *            first argument type and {@code pred2}'s first argument type,
   *            so if neither type is extending the
   *            other this method will not compile
   * @param <T2> resulting type of second argument, see {@code <T1>} for the fine print
   * @return combination of both predicates with logical and
   * @see #and(BiPredicate)
   */
  @NotNull
  static <T1, T2> Predicate2<T1, T2> and(@NotNull BiPredicate<? super T1, ? super T2> pred1,
                                         @NotNull BiPredicate<? super T1, ? super T2> pred2)
  {
    return (p1, p2) -> pred1.test(p1, p2) && pred2.test(p1, p2);
  }

  /**
   * Get a predicate which is the combination of two predicates
   * with a logical or.
   * <p>
   * The result will use shortcut evaluation, so if this
   * predicate already returns {@code true} the other one is
   * not evaluated.
   *
   * @param pred1 first predicate
   * @param pred2 second predicate
   * @param <T1> resulting type of first argument,
   *            which is the most extending type of both {@code pred1}'s
   *            first argument type and {@code pred2}'s first argument type,
   *            so if neither type is extending the
   *            other this method will not compile
   * @param <T2> resulting type of second argument, see {@code <T1>} for the fine print
   * @return combination of both predicates with logical or
   * @see #or(BiPredicate)
   */
  @NotNull
  static <T1, T2> Predicate2<T1, T2> or(@NotNull BiPredicate<? super T1, ? super T2> pred1,
                                        @NotNull BiPredicate<? super T1, ? super T2> pred2)
  {
    return (p1, p2) -> pred1.test(p1, p2) || pred2.test(p1, p2);
  }

  /**
   * Get a predicate which is the combination of two predicates
   * with a logical exclusive or.
   * <p>
   * The result will use shortcut evaluation, so if this
   * predicate already returns {@code true} the other one is
   * not evaluated.
   *
   * @param pred1 first predicate
   * @param pred2 second predicate
   * @param <T1> resulting type of first argument,
   *            which is the most extending type of both {@code pred1}'s
   *            first argument type and {@code pred2}'s first argument type,
   *            so if neither type is extending the
   *            other this method will not compile
   * @param <T2> resulting type of second argument, see {@code <T1>} for the fine print
   * @return combination of both predicates with logical exclusive or
   * @see #xor(BiPredicate)
   */
  @NotNull
  static <T1, T2> Predicate2<T1, T2> xor(@NotNull BiPredicate<? super T1, ? super T2> pred1,
                                         @NotNull BiPredicate<? super T1, ? super T2> pred2)
  {
    return (p1, p2) -> pred1.test(p1, p2) ^ pred2.test(p1, p2);
  }
  /**
   * Get a chained predicate which first applies the given function, then this predicate.
   * The given function has to return a {@link ITuple2} with matching parametric types
   * which are applied in their order to this predicate.
   * @param before function to apply first, has to return a non-{@code null} result so unpacking the parameters
   *               from the tuple is possible
   * @param <FR> function {@code before}'s return type, has to be a 2-tuple with matching types
   * @param <FP> function {@code before}'s parameter type, same for returned predicate
   * @return chain of function and this predicate
   */
  @NotNull
  default <FR extends ITuple2<? extends P1, ? extends P2>, FP>
  Predicate1<FP> after1(@NotNull Function<FP, FR> before)
  {
    return fp -> before.apply(fp).testBy(this);
  }

  /**
   * Get a chained predicate which first applies the given 2-parameter function, then this predicate.
   * The given function has to return a {@link ITuple2} with matching parametric types
   * which are applied in their order to this predicate.
   * @param before function to apply first, has to return a non-{@code null} result so unpacking the parameters
   *               from the tuple is possible
   * @param <FR> function {@code before}'s return type, has to be a 2-tuple with matching types
   * @param <FP1> function {@code before}'s first parameter type, sane for returned predicate
   * @param <FP2> function {@code before}'s second parameter type, same for returned predicate
   * @return chain of the given function and this predicate
   */
  @NotNull
  default <FR extends ITuple2<? extends P1, ? extends P2>, FP1, FP2>
  Predicate2<FP1, FP2> after2(@NotNull BiFunction<FP1, FP2, FR> before)
  {
    return (fp1, fp2) -> before.apply(fp1, fp2).testBy(this);
  }

  /**
   * Get a chained predicate which first applies the given 3-parameter function, then this predicate.
   * The given function has to return a {@link ITuple2} with matching parametric types
   * which are applied in their order to this predicate.
   * @param before function to apply first, has to return a non-{@code null} result so unpacking the parameters
   *               from the tuple is possible
   * @param <FR> function {@code before}'s return type, has to be a 2-tuple with matching types
   * @param <FP1> function {@code before}'s first parameter type, sane for returned predicate
   * @param <FP2> function {@code before}'s second parameter type, sane for returned predicate
   * @param <FP3> function {@code before}'s third parameter type, sane for returned predicate
   * @return chain of the given function and this predicate
   */
  @NotNull
  default <FR extends ITuple2<? extends P1, ? extends P2>, FP1, FP2, FP3>
  Predicate3<FP1, FP2, FP3> after3(@NotNull Function3<FR, FP1, FP2, FP3> before)
  {
    return (fp1, fp2, fp3) -> before.apply(fp1, fp2, fp3).testBy(this);
  }

  /**
   * Get a chained predicate which first applies the given 5-parameter function, then this predicate.
   * The given function has to return a {@link ITuple2} with matching parametric types
   * which are applied in their order to this function.
   * @param before function to apply first, has to return a non-{@code null} result so unpacking the parameters
   *               from the tuple is possible
   * @param <FR> function {@code before}'s return type, has to be a 2-tuple with matching types
   * @param <FP1> function {@code before}'s first parameter type, sane for returned predicate
   * @param <FP2> function {@code before}'s second parameter type, sane for returned predicate
   * @param <FP3> function {@code before}'s third parameter type, sane for returned predicate
   * @param <FP4> function {@code before}'s fourth parameter type, sane for returned predicate
   * @param <FP5> function {@code before}'s fifth parameter type, sane for returned predicate
   * @return chain of the given function and this predicate
   */
  @NotNull
  default <FR extends ITuple2<? extends P1, ? extends P2>, FP1, FP2, FP3, FP4, FP5>
  Predicate5<FP1, FP2, FP3, FP4, FP5> after5(@NotNull Function5<FR, FP1, FP2, FP3, FP4, FP5> before)
  {
    return (fp1, fp2, fp3, fp4, fp5) -> before.apply(fp1, fp2, fp3, fp4, fp5).testBy(this);
  }

  /**
   * Get a chained predicate which first applies the given 6-parameter function, then this predicate.
   * The given function has to return a {@link ITuple2} with matching parametric types
   * which are applied in their order to this function.
   * @param before function to apply first, has to return a non-{@code null} result so unpacking the parameters
   *               from the tuple is possible
   * @param <FR> function {@code before}'s return type, has to be a 2-tuple with matching types
   * @param <FP1> function {@code before}'s first parameter type, sane for returned predicate
   * @param <FP2> function {@code before}'s second parameter type, sane for returned predicate
   * @param <FP3> function {@code before}'s third parameter type, sane for returned predicate
   * @param <FP4> function {@code before}'s fourth parameter type, sane for returned predicate
   * @param <FP5> function {@code before}'s fifth parameter type, sane for returned predicate
   * @param <FP6> function {@code before}'s sixth parameter type, sane for returned predicate
   * @return chain of the given function and this predicate
   */
  @NotNull
  default <FR extends ITuple2<? extends P1, ? extends P2>, FP1, FP2, FP3, FP4, FP5, FP6>
  Predicate6<FP1, FP2, FP3, FP4, FP5, FP6> after6(@NotNull Function6<FR, FP1, FP2, FP3, FP4, FP5, FP6> before)
  {
    return (fp1, fp2, fp3, fp4, fp5, fp6) -> before.apply(fp1, fp2, fp3, fp4, fp5, fp6).testBy(this);
  }

  /**
   * Get a chained predicate which first applies the given 7-parameter function, then this one.
   * The given function has to return a {@link ITuple2} with matching parametric types
   * which are applied in their order to this predicate.
   * @param before function to apply first, has to return a non-{@code null} result so unpacking the parameters
   *               from the tuple is possible
   * @param <FR> function {@code before}'s return type, has to be a 2-tuple with matching types
   * @param <FP1> function {@code before}'s first parameter type, sane for returned predicate
   * @param <FP2> function {@code before}'s second parameter type, sane for returned predicate
   * @param <FP3> function {@code before}'s third parameter type, sane for returned predicate
   * @param <FP4> function {@code before}'s fourth parameter type, sane for returned predicate
   * @param <FP5> function {@code before}'s fifth parameter type, sane for returned predicate
   * @param <FP6> function {@code before}'s sixth parameter type, sane for returned predicate
   * @param <FP7> function {@code before}'s seventh parameter type, sane for returned predicate
   * @return chain of the given function and this predicate
   */
  @NotNull
  default <FR extends ITuple2<? extends P1, ? extends P2>, FP1, FP2, FP3, FP4, FP5, FP6, FP7>
  Predicate7<FP1, FP2, FP3, FP4, FP5, FP6, FP7> after7(@NotNull Function7<FR, FP1, FP2, FP3, FP4, FP5, FP6, FP7> before)
  {
    return (fp1, fp2, fp3, fp4, fp5, fp6, fp7) -> before.apply(fp1, fp2, fp3, fp4, fp5, fp6, fp7).testBy(this);
  }

  /**
   * Get a chained predicate which first applies the given 8-parameter function, then this one.
   * The given function has to return a {@link ITuple2} with matching parametric types
   * which are applied in their order to this predicate.
   * @param before function to apply first, has to return a non-{@code null} result so unpacking the parameters
   *               from the tuple is possible
   * @param <FR> function {@code before}'s return type, has to be a 2-tuple with matching types
   * @param <FP1> function {@code before}'s first parameter type, sane for returned predicate
   * @param <FP2> function {@code before}'s second parameter type, sane for returned predicate
   * @param <FP3> function {@code before}'s third parameter type, sane for returned predicate
   * @param <FP4> function {@code before}'s fourth parameter type, sane for returned predicate
   * @param <FP5> function {@code before}'s fifth parameter type, sane for returned predicate
   * @param <FP6> function {@code before}'s sixth parameter type, sane for returned predicate
   * @param <FP7> function {@code before}'s seventh parameter type, sane for returned predicate
   * @param <FP8> function {@code before}'s seventh parameter type, sane for returned predicate
   * @return chain of the given function and this predicate
   */
  @NotNull
  default <FR extends ITuple2<? extends P1, ? extends P2>, FP1, FP2, FP3, FP4, FP5, FP6, FP7, FP8>
  Predicate8<FP1, FP2, FP3, FP4, FP5, FP6, FP7, FP8> after8(@NotNull Function8<FR, FP1, FP2, FP3, FP4, FP5, FP6, FP7, FP8> before)
  {
    return (fp1, fp2, fp3, fp4, fp5, fp6, fp7, fp8) -> before.apply(fp1, fp2, fp3, fp4, fp5, fp6, fp7, fp8).testBy(this);
  }

  /**
   * Get a chained predicate which first applies the given 9-parameter function, then this one.
   * The given function has to return a {@link ITuple2} with matching parametric types
   * which are applied in their order to this predicate.
   * @param before function to apply first, has to return a non-{@code null} result so unpacking the parameters
   *               from the tuple is possible
   * @param <FR> function {@code before}'s return type, has to be a 2-tuple with matching types
   * @param <FP1> function {@code before}'s first parameter type, sane for returned predicate
   * @param <FP2> function {@code before}'s second parameter type, sane for returned predicate
   * @param <FP3> function {@code before}'s third parameter type, sane for returned predicate
   * @param <FP4> function {@code before}'s fourth parameter type, sane for returned predicate
   * @param <FP5> function {@code before}'s fifth parameter type, sane for returned predicate
   * @param <FP6> function {@code before}'s sixth parameter type, sane for returned predicate
   * @param <FP7> function {@code before}'s seventh parameter type, sane for returned predicate
   * @param <FP8> function {@code before}'s seventh parameter type, sane for returned predicate
   * @param <FP9> function {@code before}'s ninth parameter type, sane for returned predicate
   * @return chain of the given function and this predicate
   */
  @NotNull
  default <FR extends ITuple2<? extends P1, ? extends P2>, FP1, FP2, FP3, FP4, FP5, FP6, FP7, FP8, FP9>
  Predicate9<FP1, FP2, FP3, FP4, FP5, FP6, FP7, FP8, FP9> after9(@NotNull Function9<FR, FP1, FP2, FP3, FP4, FP5, FP6, FP7, FP8, FP9> before)
  {
    return (fp1, fp2, fp3, fp4, fp5, fp6, fp7, fp8, fp9) -> before.apply(fp1, fp2, fp3, fp4, fp5, fp6, fp7, fp8, fp9).testBy(this);
  }
}
