// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.function;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;

import java.util.Objects;
import java.util.function.Function;
import java.util.function.Predicate;

/**
 * A function with one argument which returns a boolean value.
 * <p>
 * Predicates can be chained to simulate boolean logic by
 * {@link #and(Predicate)}, {@link #or(Predicate)},
 * {@link #xor(Predicate)}, and {@link #negate()}.
 * The chaining does not always work as expected from
 * standard Java expressions. Therefore there are
 * versions of the above methods with an underscore suffix
 * which allow to widen the type: {@link #and_(Predicate)},
 * {@link #or_(Predicate)}, {@link #xor_(Predicate)}.
 * <p>
 * There are also static methods which take care of the
 * type fiddling transparently: {@link #and(Predicate, Predicate)},
 *
 * <p>
 * Useful for start values in algorithms or as default values
 * are the two values returned by {@link #alwaysTrue()}
 * (a predicate which always returns {@code true} regardless
 * the argument applied to the {@link #test(Object)} method)
 * and {@link #alwaysFalse()} (same for {@code false}).
 * <p>
 * This also acts as a standard {@link Predicate}.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @param <P> parameter type
 * @see Predicate2
 * @see Predicate3
 * @see Predicate4
 * @see Predicate5
 * @see Predicate6
 * @see Predicate7
 * @see Predicate8
 * @see Predicate9
 */
@FunctionalInterface
public interface Predicate1<P>
        extends Predicate<P>
{
  /**
   * Predicate which always returns true.
   * @deprecated Use {@link #alwaysTrue()} instead as the returned predicate is smarter.
   */
  @Deprecated
  Predicate1<Object> ALWAYS_TRUE = arg -> true;
  /**
   * Predicate which always returns false.
   * @deprecated Use {@link #alwaysFalse()} instead as the returned predicate is smarter.
   */
  @Deprecated
  Predicate1<Object> ALWAYS_FALSE = arg -> false;

  /**
   * Test the argument only if it is not {@code null}, otherwise return the fallback.
   *<p>
   * Using this invocation it is guaranteed that {@link #test(Object)} will always be called with
   * a non-null argument.
   * </p>
   * @param argument argument of this function
   * @param fallback fallback returned if {@code argument} is {@code null}
   * @return the result of applying this predicate to the non-null {@code argument},
   *         or {@code fallback} if {@code argument} is null
   */
  default boolean testNonNull(P argument, boolean fallback)
  {
    return argument == null
            ? fallback
            : test(argument);
  }

  /**
   * Combine this and another predicate with logical and.
   * The result will use shortcut evaluation, so if this
   * predicate already returns {@code false} the other one is
   * not evaluated.
   * @param other other predicate to be chained with logical and
   * @return predicates combined with logical and
   * @see #and(Predicate, Predicate)
   * @see #and_(Predicate)
   */
  @NotNull
  default Predicate1<P> and(@NotNull Predicate<? super P> other)
  {
    return p -> test(p) && other.test(p);
  }

  /**
   * Combine this and another predicate with logical and.
   * The result will use shortcut evaluation, so if this
   * predicate already returns {@code false} the other one is
   * not evaluated.
   * <p>
   * Compared to {@link #and(Predicate)} this method allows
   * to widen the type of this predicate to the one of the other predicate
   * like in standard Jave logical expressions, eg
   * <pre>{@code
   *   Predicate1<Object> nonNull = o -> o != null;
   *   Predicate1<String> notEmpty = s -> !s.isEmpty();
   *   // Predicate1<???> bad = nonNull.and(notEmpty);     // not possible as String is not a super type of Object
   *   Predicate1<String> valid = nonMull.and_(notEmpty);  // not nice, but it works
   * }</pre>
   * @param other other predicate
   * @param <T> type on which the other predicate operates, also the type of
   *           the returned predicate
   * @return predicates combined with logical and
   * @see #and(Predicate, Predicate)
   * @see #and(Predicate)
   */
  @NotNull
  default <T extends P> Predicate1<T> and_(@NotNull Predicate<T> other)
  {
    return p -> test(p) && other.test(p);
  }

  /**
   * Combine this and another predicate with logical or.
   * The result will use shortcut evaluation, so if this
   * predicate already returns {@code true} the other one is
   * not evaluated.
   * @param other other predicate to be chained with logical or
   * @return predicates combined with logical or
   * @see #or(Predicate, Predicate)
   * @see #or_(Predicate)
   */
  @NotNull
  default Predicate1<P> or(@NotNull Predicate<? super P> other)
  {
    return p -> test(p) || other.test(p);
  }

  /**
   * Combine this and another predicate with logical or.
   * The result will use shortcut evaluation, so if this
   * predicate already returns {@code true} the other one is
   * not evaluated.
   * <p>
   * Compared to {@link #or(Predicate)} this method allows
   * to widen the type of this predicate to the one of the other predicate
   * like in standard Jave logical expressions, eg
   * <pre>{@code
   *   Predicate1<Object> isNull = o -> o == null;
   *   Predicate1<String> isEmpty = s -> !s.isEmpty();
   *   // Predicate1<???> bad = pred1.or(pred2);     // not possible as String is not a super type of Object
   *   Predicate1<String> useless = pred1.or_(pred2);  // not nice, but it works
   * }</pre>
   * @param other other predicate
   * @param <T> type on which the other predicate operates, also the type of
   *           the returned predicate
   * @return predicates combined with logical or
   * @see #or(Predicate, Predicate)
   * @see #or(Predicate)
   */
  @NotNull
  default <T extends P> Predicate1<T> or_(@NotNull Predicate<T> other)
  {
    return p -> test(p) || other.test(p);
  }

  /**
   * Combine this and another predicate with logical exclusive or.
   * @param other other predicate to be chained with logical exclusive or
   * @return predicates combined with logical exclusive or
   * @see #xor(Predicate, Predicate)
   * @see #xor_(Predicate)
   */
  @NotNull
  default Predicate1<P> xor(@NotNull Predicate<? super P> other)
  {
    return p -> test(p) ^ other.test(p);
  }

  /**
   * Combine this and another predicate with logical exclusive or.
   * The result will use shortcut evaluation, so if this
   * predicate already returns {@code true} the other one is
   * not evaluated.
   * <p>
   * Compared to {@link #or(Predicate)} this method allows
   * to widen the type of this predicate to the one of the other predicate
   * like in standard Jave logical expressions, eg
   * <pre>{@code
   *   Predicate1<Object> isNull = o -> o == null;
   *   Predicate1<String> isEmpty = s -> !s.isEmpty();
   *   // Predicate1<???> bad = isNull.xor(isEmpty);     // not possible as String is not a super type of Object
   *   Predicate1<String> useless = isNull.xor_(isEmpty);  // bad example, but it works
   * }</pre>
   * @param other other predicate
   * @param <T> type on which the other predicate operates, also the type of
   *           the returned predicate
   * @return predicates combined with logical exclusive or
   * @see #xor(Predicate, Predicate)
   * @see #xor(Predicate)
   */
  @NotNull
  default <T extends P> Predicate1<T> xor_(@NotNull Predicate<T> other)
  {
    return p -> test(p) ^ other.test(p);
  }

  /**
   * Get the negation of this predicate.
   * @return negation of this predicate
   */
  @NotNull
  default Predicate1<P> negate()
  {
    return new Predicate1<P>()
    {
      @Override
      public boolean test(P arg)
      {
        return !Predicate1.this.test(arg);
      }

      @NotNull
      @Override
      public Predicate1<P> negate()
      {
        return Predicate1.this;
      }
    };
  }

  /**
   * Use this predicate on a different type.
   * @param mapper mapper function which converts the different type to one this predicate accepts.
   * @param <S> different type on which the returned predicate operates
   * @return predicate on the new type, base on this one
   */
  @NotNull
  default <S> Predicate1<S> adapted(@NotNull Function<? super S, ? extends P> mapper)
  {
    return s -> Predicate1.this.test(mapper.apply(s));
  }

  /**
   * Use a standard {@link Predicate} as a {@code Predicate1},
   * This is eg useful if your want to use the {@link #xor(Predicate)} method.
   * @param predicate standard predicate
   * @param <T> argument type
   * @return {@code Predicate1} view of given predicate
   */
  @NotNull
  static <T> Predicate1<T> from(@NotNull Predicate<T> predicate)
  {
    return predicate instanceof Predicate1
            ? (Predicate1<T>)predicate
            : predicate::test;
  }

  /**
   * Get a predicate which always evaluates to {@code true}.
   * @param <T> value type of predicate
   * @return predicate always returning {@code true}
   */
  @NotNull
  @SuppressWarnings("unchecked") // predicate ignores the object
  static <T> Predicate1<T> alwaysTrue()
  {
    return (Predicate1<T>)Predicates.TRUE1;
  }

  /**
   * Get a predicate which always evaluates to {@code false}.
   * @param <T> value type of predicate
   * @return predicate always returning {@code false}
   */
  @NotNull
  @SuppressWarnings("unchecked") // predicate ignores the object
  static <T> Predicate1<T> alwaysFalse()
  {
    return (Predicate1<T>)Predicates.FALSE1;
  }

  /**
   * Get a predicate which checks whether a value is {@code null}.
   * @return predicate returning {@code true} if a value is {@code null}
   * @param <T> value type
   */
  @NotNull
  static <T> Predicate1<T> isNull()
  {
    return Objects::isNull;
  }

  /**
   * Get a predicate which checks whether a value is not {@code null}.
   * @return predicate returning {@code true} if a value is not {@code null}
   * @param <T> value type
   */
  @NotNull
  static <T> Predicate1<T> notNull()
  {
    return Objects::nonNull;
  }

  /**
   * Get a predicate which checks deep equality with the given target.
   * <p>
   * Different to {@link Predicate#isEqual(Object)} this internally uses
   * {@link Objects#deepEquals(Object, Object)} to check for equality.
   *
   * @param target target object, possibly {@code null}
   * @param <T> target type
   * @return predicate for deep equality with the given {@code target}
   */
  @NotNull
  static <T> Predicate1<T> isEqual(@Nullable T target)
  {
    return o -> Objects.deepEquals(target, o);
  }

  /**
   * Get a predicate which is the combination of two predicates
   * with a logical and.
   * <p>
   * This method takes care of always creating a predicate of the correct type
   * without having to fiddle with {@link #and(Predicate)} or {@link #and_(Predicate)}
   * depending on the types.
   * <p>
   * The result will use shortcut evaluation, so if this
   * predicate already returns {@code false} the other one is
   * not evaluated.
   *
   * @param pred1 first predicate
   * @param pred2 second predicate
   * @param <T> resulting type, which is the most extending type of both {@code pred1}'s
   *           type and {@code pred2}'s type, so if neither type is extending the
   *           other this method will not compile
   * @return combination of both predicates with logical and
   * @see #and(Predicate)
   * @see #and_(Predicate)
   */
  @NotNull
  static <T> Predicate1<T> and(@NotNull Predicate<? super T> pred1,
                               @NotNull Predicate<? super T> pred2)
  {
    return p -> pred1.test(p) && pred2.test(p);
  }

  /**
   * Get a predicate which is the combination of two predicates
   * with a logical or.
   * <p>
   * This method takes care of always creating a predicate of the correct type
   * without having to fiddle with {@link #or(Predicate)} or {@link #or_(Predicate)}
   * depending on the types.
   * <p>
   * The result will use shortcut evaluation, so if this
   * predicate already returns {@code false} the other one is
   * not evaluated.
   *
   * @param pred1 first predicate
   * @param pred2 second predicate
   * @param <T> resulting type, which is the most extending type of both {@code pred1}'s
   *           type and {@code pred2}'s type, so if neither type is extending the
   *           other this method will not compile
   * @return combination of both predicates with logical or
   * @see #or(Predicate)
   * @see #or_(Predicate)
   */
  @NotNull
  static <T> Predicate1<T> or(@NotNull Predicate<? super T> pred1,
                              @NotNull Predicate<? super T> pred2)
  {
    return p -> pred1.test(p) || pred2.test(p);
  }

  /**
   * Get a predicate which is the combination of two predicates
   * with a logical exclusive or.
   * <p>
   * This method takes care of always creating a predicate of the correct type
   * without having to fiddle with {@link #or(Predicate)} or {@link #or_(Predicate)}
   * depending on the types.
   * <p>
   * The result will use shortcut evaluation, so if this
   * predicate already returns {@code true} the other one is
   * not evaluated.
   *
   * @param pred1 first predicate
   * @param pred2 second predicate
   * @param <T> resulting type, which is the most extending type of both {@code pred1}'s
   *           type and {@code pred2}'s type, so if neither type is extending the
   *           other this method will not compile
   * @return combination of both predicates with logical exclusive or
   * @see #xor(Predicate)
   * @see #xor_(Predicate)
   */
  @NotNull
  static <T> Predicate1<T> xor(@NotNull Predicate<? super T> pred1,
                               @NotNull Predicate<? super T> pred2)
  {
    return p -> pred1.test(p) ^ pred2.test(p);
  }
}
