// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright ©2023-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.function;

import de.caff.annotation.NotNull;
import de.caff.generics.Order;

import java.io.Serializable;
import java.util.Comparator;
import java.util.function.BiPredicate;
import java.util.function.Function;
import java.util.function.ToDoubleFunction;

/**
 * Define an order for two objects of a generic type.
 * An ordering always has to provide the following invariant:
 * {@code order.compare(a, b) == order.compare(b, a).inverse()}
 * for any two values of {@code a} and {@code b}.
 * Note that this also implies {@code order.compare(a,a) == Order.Same}
 * as {@code Order.Same} is the only result which is its own inverse.
 * <p>
 * Ordering is very much the same as a {@link Comparator}, it just
 * has a nicer result. Of course method {@link #asComparator()}
 * converts an ordering into a {@code Comparator}, and static factory
 * method {@link #fromComparator(Comparator)} does the inverse.
 * It is recommended that {@code Comparator}s are serializable for
 * usage with {@link java.util.TreeMap} and similar, and so this is true
 * for orderings. See {@linkplain Serial} to get a general example.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since January 11, 2023
 */
@FunctionalInterface
public interface Ordering<T>
{
  /**
   * Check the ordering of the two values.
   * This method has to fulfill the
   * @param v1 first value
   * @param v2 second value
   * @return order of the two values
   */
  @NotNull
  Order check(T v1, T v2);

  /**
   * Are the two values represented in strictly ascending order?
   * @param v1 first value
   * @param v2 second value
   * @return {@code true} if {@code v1 < v2} according to this ordering<br>
   *         {@code false} if {@code v1 >= v2} according to this ordering
   * @see #descending(Object, Object)
   * @see #ascendingOrSame(Object, Object)
   * @see #descendingOrSame(Object, Object)
   * @see #same(Object, Object)
   * @see #different(Object, Object)
   */
  default boolean ascending(T v1, T v2)
  {
    return check(v1, v2).ascending;
  }

  /**
   * Are the two values represented in ascending order?
   * @param v1 first value
   * @param v2 second value
   * @return {@code true} if {@code v1 <= v2} according to this ordering<br>
   *         {@code false} if {@code v1 > v2} according to this ordering
   * @see #ascending(Object, Object)
   * @see #descending(Object, Object)
   * @see #descendingOrSame(Object, Object)
   * @see #same(Object, Object)
   * @see #different(Object, Object)
   */
  default boolean ascendingOrSame(T v1, T v2)
  {
    return check(v1, v2).ascendingOrSame;
  }

  /**
   * Are the two values represented in strictly descending order?
   * @param v1 first value
   * @param v2 second value
   * @return {@code true} if {@code v1 > v2} according to this ordering<br>
   *         {@code false} if {@code v1 <= v2} according to this ordering
   * @see #ascending(Object, Object)
   * @see #ascendingOrSame(Object, Object)
   * @see #descendingOrSame(Object, Object)
   * @see #same(Object, Object)
   * @see #different(Object, Object)
   */
  default boolean descending(T v1, T v2)
  {
    return check(v1, v2).descending;
  }

  /**
   * Are the two values represented in descending order?
   * @param v1 first value
   * @param v2 second value
   * @return {@code true} if {@code v1 >= v2} according to this ordering<br>
   *         {@code false} if {@code v1 < v2} according to this ordering
   * @see #ascending(Object, Object)
   * @see #descending(Object, Object)
   * @see #ascendingOrSame(Object, Object)
   * @see #same(Object, Object)
   * @see #different(Object, Object)
   */
  default boolean descendingOrSame(T v1, T v2)
  {
    return check(v1, v2).descendingOrSame;
  }

  /**
   * Are the two values considered equal by this order?
   * @param v1 first value
   * @param v2 second value
   * @return {@code true} if {@code v1 == v2} according to this order<br>
   *         {@code false} if {@code v1 != v2} according to this ordering
   * @see #ascending(Object, Object)
   * @see #descending(Object, Object)
   * @see #ascendingOrSame(Object, Object)
   * @see #descendingOrSame(Object, Object)
   * @see #different(Object, Object)
   */
  default boolean same(T v1, T v2)
  {
    return check(v1, v2).same;
  }

  /**
   * Are the two values considered different by this order?
   * @param v1 first value
   * @param v2 second value
   * @return {@code true} if {@code v1 != v2} according to this order<br>
   *         {@code false} if {@code v1 == v2} according to this ordering
   * @see #ascending(Object, Object)
   * @see #descending(Object, Object)
   * @see #ascendingOrSame(Object, Object)
   * @see #descendingOrSame(Object, Object)
   * @see #same(Object, Object)
   */
  default boolean different(T v1, T v2)
  {
    return check(v1, v2).different;
  }


  /**
   * Invert this order.
   * @return inverse order
   */
  @NotNull
  default Ordering<T> inverse()
  {
    return new Serial<T>()
    {
      private static final long serialVersionUID = -4235884649457541199L;

      @Override
      @NotNull
      public Order check(T v1, T v2)
      {
        return Ordering.this.check(v2, v1);
      }

      @NotNull
      @Override
      public Ordering<T> inverse()
      {
        return Ordering.this;
      }
    };
  }

  /**
   * Make this ordering null-safe in handling {@code null} values
   * as coming before all other values.
   * @return ordering sorting {@code null} values first
   */
  default Ordering<T> nullsFirst()
  {
    return (Ordering<T> & Serializable) (v1, v2) -> v1 == null
            ? (v2 == null
                       ? Order.Same
                       : Order.Ascending)
            : (v2 == null
                       ? Order.Descending
                       : Ordering.this.check(v1, v2));
  }

  /**
   * Make this ordering null-safe in handling {@code null} values
   * as coming after all other values.
   * @return ordering sorting {@code null} values last
   */
  default Ordering<T> nullsLast()
  {
    return (Ordering<T> & Serializable) (v1, v2) -> v1 == null
            ? (v2 == null
                       ? Order.Same
                       : Order.Descending)
            : (v2 == null
                       ? Order.Ascending
                       : Ordering.this.check(v1, v2));
  }

  /**
   * Use this ordering as a comparator.
   * @return comparator of {@code Double}
   */
  @NotNull
  default Comparator<T> asComparator()
  {
    return (Comparator<T> & Serializable)(v1, v2) -> Ordering.this.check(v1, v2).comparison;
  }

  /**
   * Combine this ordering with a lower-level ordering.
   * This lower level ordering will only take place if this ordering
   * cannot differ between values and considers them to be
   * {@link Order#Same equal}.
   * @param lowerOrder lower order
   * @return combined order
   */
  @NotNull
  default Ordering<T> andIfSame(@NotNull Ordering<? super T> lowerOrder)
  {
    return (Ordering<T> & Serializable) (v1, v2) -> {
      final Order primaryOrder = Ordering.this.check(v1, v2);
      return primaryOrder == Order.Same
              ? lowerOrder.check(v1, v2)
              : primaryOrder;
    };
  }

  @NotNull
  default <S> Ordering<T> andIfSame(@NotNull Function<? super T, ? extends S> keyExtractor,
                                    @NotNull Ordering<? super S> keyOrdering)
  {
    return andIfSame(ordering(keyExtractor, keyOrdering));
  }

  @NotNull
  default <S extends Comparable<? super S>> Ordering<T> andIfSame(@NotNull Function<? super T, ? extends S> keyExtractor)
  {
    return andIfSame(ordering(keyExtractor));
  }

  @NotNull
  static <S, K> Ordering<S> ordering(@NotNull Function<? super S, ? extends K> keyExtractor,
                                     @NotNull Ordering<? super K> keyOrdering)
  {
    return (Ordering<S> & Serializable) (v1, v2) ->
            keyOrdering.check(keyExtractor.apply(v1),
                              keyExtractor.apply(v2));
  }

  @NotNull
  static <S, K extends Comparable<? super K>> Ordering<S> ordering(@NotNull Function<? super S, ? extends K> keyExtractor)
  {
    return (Ordering<S> & Serializable) (v1, v2) ->
            Order.fromCompare(keyExtractor.apply(v1).compareTo(keyExtractor.apply(v2)));
  }

  @NotNull
  static <S> Ordering<S> orderingDouble(@NotNull ToDoubleFunction<? super S> valueExtractor,
                                        @NotNull DoubleOrdering ordering)
  {
    return (v1, v2) -> ordering.checkDouble(valueExtractor.applyAsDouble(v1),
                                            valueExtractor.applyAsDouble(v2));
  }

  @NotNull
  static <S> Ordering<S> orderingDouble(@NotNull ToDoubleFunction<? super S> valueExtractor)
  {
    return orderingDouble(valueExtractor, DoubleOrdering.STANDARD_ASCENDING);
  }

  // todo: add more combinations w/ primitive orders

  /**
   * Use this ordering as a checker for equality.
   * @return equality checker
   */
  @NotNull
  default BiPredicate<T, T> asEqualityChecker()
  {
    return (v1, v2) -> Ordering.this.check(v1, v2) == Order.Same;
  }

  /**
   * Convert a standard comparator of {@code Double} into a double ordering.
   * @param comparator comparator of {@code Double}
   * @return ordering
   * @param <E> type on which the order is defined
   */
  @NotNull
  static <E> Ordering<E> fromComparator(@NotNull Comparator<E> comparator)
  {
    return new Serial<E>()
    {
      private static final long serialVersionUID = 1262617626311521623L;

      @NotNull
      @Override
      public Order check(E v1, E v2)
      {
        return Order.fromCompare(comparator.compare(v1, v2));
      }

      @NotNull
      @Override
      public Comparator<E> asComparator()
      {
        return comparator;
      }
    };
  }

  /**
   * Get the natural order of a comparable type.
   * @return natural order of the given comparable type
   * @param <E> comparable type
   */
  @NotNull
  static <E extends Comparable<? super E>> Ordering<E> natural()
  {
    return new Serial<E>()
    {
      private static final long serialVersionUID = -7486072987131063309L;

      @NotNull
      @Override
      public Order check(E v1, E v2)
      {
        return Order.fromCompare(v1.compareTo(v2));
      }

      @NotNull
      @Override
      public Comparator<E> asComparator()
      {
        return Comparator.naturalOrder();
      }
    };
  }

  /**
   * Get the inverse of the natural order of a comparable type.
   * @return inverse natural order of the given comparable type
   * @param <E> comparable type
   */
  @NotNull
  static <E extends Comparable<? super E>> Ordering<E> inverseNatural()
  {
    return new Serial<E>()
    {
      private static final long serialVersionUID = -1428976512512032469L;

      @NotNull
      @Override
      public Order check(E v1, E v2)
      {
        return Order.fromCompare(v2.compareTo(v1));
      }

      @NotNull
      @Override
      public Comparator<E> asComparator()
      {
        return (Comparator<E> & Serializable) (v1, v2) -> v2.compareTo(v1);
      }
    };
  }

  /**
   * An ordering which is serializable.
   * It is a recommendation that comparators implement {@code java.io.Serializable}
   * to allow serializing {@code TreeMap}s and similar. If an Ordering is expected to be used
   * as a comparator in such areas implement either this interface, or for lambdas
   * use a cast to force the compiler to make the order serializable:
   * <pre>{@code
   *   return (Ordering<T> & Serializable) (v1, v2) -> v1.compareTo(v2);
   * }</pre>
   * You can also use a {@code (Ordering.Serial)} cast to save a bit of typing,
   * but the above is a nice trick to have in your toolbox.
   */
  interface Serial<E> extends Ordering<E>,
                              java.io.Serializable
  {
  }
}
