// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright ©2023-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.function;

import de.caff.annotation.NotNull;
import de.caff.generics.Order;

import java.io.Serializable;
import java.util.Comparator;

/**
 * Define an order for two primitive long values.
 * <p>
 * Compared to a  {@code Comparator<Long>} this can
 * avoid boxing and unboxing in various situations.
 * For this use the {@link #checkLong(long, long)} method,
 * the {@link #check(Long, Long)} method is only for situations
 * where this is used as a generic ordering (which requires boxing
 * and unboxing).
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since January 23, 2023
 */
@FunctionalInterface
public interface LongOrdering
{
  /**
   * Natural order of ascending {@code long} values.
   */
  LongOrdering ASCENDING = new LongOrdering.Serial()
  {
    private static final long serialVersionUID = -7646031177018732313L;

    @NotNull
    @Override
    public Order checkLong(long v1, long v2)
    {
      return (v1 < v2)
              ? Order.Ascending
              : (v1 > v2
                         ? Order.Descending
                         : Order.Same);
    }

    @NotNull
    @Override
    public LongOrdering inverse()
    {
      return LongOrdering.DESCENDING;
    }

    @NotNull
    @Override
    public Comparator<Long> asLongComparator()
    {
      return Long::compare;
    }
  };
  /** Inverse natural order of {@code long} values. */
  LongOrdering DESCENDING = new LongOrdering.Serial()
  {
    private static final long serialVersionUID = 3026851066329505442L;

    @NotNull
    @Override
    public Order checkLong(long v2, long v1)
    {
      return (v1 < v2)
              ? Order.Ascending
              : (v1 > v2
                         ? Order.Descending
                         : Order.Same);
    }

    @NotNull
    @Override
    public LongOrdering inverse()
    {
      return LongOrdering.ASCENDING;
    }

    @NotNull
    @Override
    public Comparator<Long> asLongComparator()
    {
      return (Comparator<Long> & Serializable) (v1, v2) -> Long.compare(v2, v1);
    }
  };

  /**
   * Natural order of ascending unsigned {@code long} values.
   * This handles long values as if they are unsigned and not signed.
    */
  LongOrdering UNSIGNED_ASCENDING = new LongOrdering.Serial()
  {
    private static final long serialVersionUID = -3495520630473840505L;

    @NotNull
    @Override
    public Order checkLong(long v1, long v2)
    {
      return ASCENDING.checkLong(v1 + Long.MIN_VALUE,
                                 v2 + Long.MIN_VALUE);
    }

    @NotNull
    @Override
    public LongOrdering inverse()
    {
      return LongOrdering.UNSIGNED_DESCENDING;
    }

    @NotNull
    @Override
    public Comparator<Long> asLongComparator()
    {
      return Long::compareUnsigned;
    }
  };

  /**
   * Natural order of ascending unsigned {@code long} values.
   * This handles long values as if they are unsigned and not signed.
    */
  LongOrdering UNSIGNED_DESCENDING = new LongOrdering.Serial()
  {
    private static final long serialVersionUID = 8393916740777317119L;

    @NotNull
    @Override
    public Order checkLong(long v1, long v2)
    {
      return DESCENDING.checkLong(v1 + Long.MIN_VALUE,
                                  v2 + Long.MIN_VALUE);
    }

    @NotNull
    @Override
    public LongOrdering inverse()
    {
      return LongOrdering.UNSIGNED_ASCENDING;
    }

    @NotNull
    @Override
    public Comparator<Long> asLongComparator()
    {
      return (Comparator<Long> & Serializable) (v1, v2) -> Long.compareUnsigned(v2, v1);
    }
  };

  /**
   * Get the order of the two values.
   * @param v1 first value
   * @param v2 second value
   * @return order of the two values
   */
  @NotNull
  Order checkLong(long v1, long v2);

  @NotNull
  default Order check(Long v1, Long v2)
  {
    return checkLong(v1, v2);
  }

  /**
   * Are the two values represented in strictly ascending order?
   * @param v1 first value
   * @param v2 second value
   * @return {@code true} if {@code v1 < v2} according to this ordering<br>
   *         {@code false} if {@code v1 >= v2} according to this ordering
   * @see #descending(long, long)
   * @see #ascendingOrSame(long, long)
   * @see #descendingOrSame(long, long)
   * @see #same(long, long)
   * @see #different(long, long)
   */
  default boolean ascending(long v1, long v2)
  {
    return checkLong(v1, v2).ascending;
  }

  /**
   * Are the two values represented in ascending order?
   * @param v1 first value
   * @param v2 second value
   * @return {@code true} if {@code v1 <= v2} according to this ordering<br>
   *         {@code false} if {@code v1 > v2} according to this ordering
   * @see #ascending(long, long)
   * @see #descending(long, long)
   * @see #descendingOrSame(long, long)
   * @see #same(long, long)
   * @see #different(long, long)
   */
  default boolean ascendingOrSame(long v1, long v2)
  {
    return checkLong(v1, v2).ascendingOrSame;
  }

  /**
   * Are the two values represented in strictly descending order?
   * @param v1 first value
   * @param v2 second value
   * @return {@code true} if {@code v1 > v2} according to this ordering<br>
   *         {@code false} if {@code v1 <= v2} according to this ordering
   * @see #ascending(long, long)
   * @see #ascendingOrSame(long, long)
   * @see #descendingOrSame(long, long)
   * @see #same(long, long)
   * @see #different(long, long)
   */
  default boolean descending(long v1, long v2)
  {
    return checkLong(v1, v2).descending;
  }

  /**
   * Are the two values represented in descending order?
   * @param v1 first value
   * @param v2 second value
   * @return {@code true} if {@code v1 >= v2} according to this ordering<br>
   *         {@code false} if {@code v1 < v2} according to this ordering
   * @see #ascending(long, long)
   * @see #descending(long, long)
   * @see #ascendingOrSame(long, long)
   * @see #same(long, long)
   * @see #different(long, long)
   */
  default boolean descendingOrSame(long v1, long v2)
  {
    return checkLong(v1, v2).descendingOrSame;
  }

  /**
   * Are the two values considered equal by this order?
   * @param v1 first value
   * @param v2 second value
   * @return {@code true} if {@code v1 == v2} according to this order<br>
   *         {@code false} if {@code v1 != v2} according to this ordering
   * @see #ascending(long, long)
   * @see #descending(long, long)
   * @see #ascendingOrSame(long, long)
   * @see #descendingOrSame(long, long)
   * @see #different(long, long)
   */
  default boolean same(long v1, long v2)
  {
    return checkLong(v1, v2).same;
  }

  /**
   * Are the two values considered different by this order?
   * @param v1 first value
   * @param v2 second value
   * @return {@code true} if {@code v1 != v2} according to this order<br>
   *         {@code false} if {@code v1 == v2} according to this ordering
   * @see #ascending(long, long)
   * @see #descending(long, long)
   * @see #ascendingOrSame(long, long)
   * @see #descendingOrSame(long, long)
   * @see #same(long, long)
   */
  default boolean different(long v1, long v2)
  {
    return checkLong(v1, v2).different;
  }

  /**
   * Invert this order.
   * @return inverse order
   */
  @NotNull
  default LongOrdering inverse()
  {
    return new LongOrdering.Serial()
    {
      private static final long serialVersionUID = 3270504639153727362L;

      @Override
      @NotNull
      public Order checkLong(long v1, long v2)
      {
        return LongOrdering.this.checkLong(v2, v1);
      }

      @NotNull
      @Override
      public LongOrdering inverse()
      {
        return LongOrdering.this;
      }
    };
  }

  /**
   * Use this ordering as a comparator.
   * @return comparator of {@code Long}
   */
  @NotNull
  default Comparator<Long> asLongComparator()
  {
    return (Comparator<Long> & Serializable) (v1, v2) -> checkLong(v1, v2).comparison;
  }

  /**
   * View this ordering as a number comparator.
   * @return comparator of {@link Number}
   */
  @NotNull
  default Comparator<Number> asNumberComparator()
  {
    return (Comparator<Number> & Serializable)(v1, v2) -> checkLong(v1.longValue(), v2.longValue()).comparison;
  }

  /**
   * Get this ordering as a generic long ordering. 
   * @return generic ordering for longs
   */
  @NotNull
  default Ordering<Long> asLongOrdering()
  {
    return (Ordering<Long> & Serializable)this::checkLong;
  }

  /**
   * Convert a standard comparator of {@code Double} into a double ordering.
   * @param comparator comparator of {@code Double}
   * @return ordering
   */
  @NotNull
  static LongOrdering fromComparator(@NotNull Comparator<Long> comparator)
  {
    return new LongOrdering.Serial()
    {
      private static final long serialVersionUID = 393576352771473074L;

      @NotNull
      @Override
      public Order checkLong(long v1, long v2)
      {
        return Order.fromCompare(comparator.compare(v1, v2));
      }

      @NotNull
      @Override
      public Comparator<Long> asLongComparator()
      {
        return comparator;
      }
    };
  }

  /**
   * A long ordering which is serializable.
   * It is a recommendation that comparators implement {@code java.io.Serializable}
   * to allow serializing {@code TreeMap}s and similar. If an Ordering is expected to be used
   * as a comparator in such areas implement either this interface, or for lambdas
   * use a cast to force the compiler to make the order serializable:
   * <pre>{@code
   *   return (LongOrdering & Serializable) (v1, v2) -> v1.compareTo(v2);
   * }</pre>
   * You can also use a {@code (LongOrdering.Serial)} cast to save a bit of typing,
   * but the above is a nice trick to have in your toolbox.
   */
  interface Serial extends LongOrdering,
                           java.io.Serializable
  {
  }
}
