// ============================================================================
// File:               IntFunction2
//
// Project:            CAFF
//
// Purpose:            
//
// Author:             Rammi
//
// Copyright Notice:   © 2020-2024  Rammi (rammi@caff.de)
//                     The usage of this source code in commercial or open 
//                     source projects is not allowed without explicit 
//                     permission.
//
// Created:            27.11.20 13:37
//=============================================================================
package de.caff.generics.function;

import de.caff.annotation.NotNull;

import java.util.function.IntBinaryOperator;

/**
 * Function with two primitive integer arguments and a primitive short result.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since November 27, 2020
 */
@FunctionalInterface
public interface IntToShortFunction2
{
  /**
   * Apply this function to the given 2 arguments.
   * @param value1 first argument
   * @param value2 second argument
   * @return the function result
   */
  short applyAsShort(int value1, int value2);

  /**
   * View this as an {@code int} to {@code int} function.
   * @return unary integer operator which will return values between
   *         {@code -32768} to {@code 32767}
   */
  @NotNull
  default IntBinaryOperator asIntToIntFunction()
  {
    return this::applyAsShort;
  }

  /**
   * View this as an {@code int} to unsigned {@code int} function.
   * @return unary integer operator which will return values between
   *         {@code 0} to {@code 65535}
   */
  @NotNull
  default IntBinaryOperator asIntToUnsignedIntFunction()
  {
    return (v1, v2) -> applyAsShort(v1, v2) & 0xFFFF;
  }

  /**
   * View this as an {@code int} to {@code long} function.
   * @return int to long function  which will return values between
   *         {@code -32768L} to {@code 32767L}
   */
  @NotNull
  default IntToLongFunction2 asIntToLongFunction()
  {
    return this::applyAsShort;
  }

  /**
   * View this as an {@code int} to unsigned {@code long} function.
   * @return unary integer operator which will return values between
   *         {@code 0L} to {@code 65535L}
   */
  @NotNull
  default IntToLongFunction2 asIntToUnsignedLongFunction()
  {
    return (v1, v2) -> applyAsShort(v1, v2) & 0xFFFFL;
  }

  /**
   * View this as an {@code int} to {@code double} function.
   * @return int to double function which will return integral values between
   *         {@code -32768.0} to {@code 32767.0}
   */
  @NotNull
  default IntToDoubleFunction2 asIntToDoubleFunction()
  {
    return this::applyAsShort;
  }

  /**
   * View this as an {@code int} to unsigned {@code double} function.
   * @return int to double function  which will return integral values between
   *         {@code 0.0} to {@code 65535.0}
   */
  @NotNull
  default IntToDoubleFunction2 asIntToUnsignedDoubleFunction()
  {
    return (v1, v2) -> applyAsShort(v1, v2) & 0XFFFF;
  }

  /**
   * View this as an {@code int} to {@code char} function which
   * considers the byte values unsigned.
   * @return int to char function which will return values between
   *         {@code (char)0} and {@code (char)65535}
   */
  @NotNull
  default IntToCharFunction2 asIntToCharFunction()
  {
    return (v1, v2) -> (char)(applyAsShort(v1, v2) & 0xFFFF);
  }
}
