// ============================================================================
// File:               IntToByteFunction1
//
// Project:            CAFF
//
// Purpose:            
//
// Author:             Rammi
//
// Copyright Notice:   © 2020-2024  Rammi (rammi@caff.de)
//                     The usage of this source code in commercial or open 
//                     source projects is not allowed without explicit 
//                     permission.
//
// Created:            27.11.20 11:59
//=============================================================================
package de.caff.generics.function;

import de.caff.annotation.NotNull;

import java.util.function.IntToDoubleFunction;
import java.util.function.IntToLongFunction;
import java.util.function.IntUnaryOperator;

/**
 * Function with one primitive integer argument and a primitive short return.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since November 27, 2020
 */
@FunctionalInterface
public interface IntToShortFunction1
{
  /**
   /**
   * Applies this function to the given argument.
   *
   * @param value the function argument
   * @return the function result
   */
  short applyAsShort(int value);

  /**
   * View this as an {@code int} to {@code int} function.
   * @return unary integer operator which will return values between
   *         {@code -32768} and {@code 32767}
   */
  @NotNull
  default IntUnaryOperator asIntToIntFunction()
  {
    return this::applyAsShort;
  }

  /**
   * View this as an {@code int} to unsigned {@code long} function.
   * @return unary integer operator which will return values between
   *         {@code 0} to {@code 65535}
   */
  @NotNull
  default IntUnaryOperator asIntToUnsignedIntFunction()
  {
    return v -> applyAsShort(v) & 0xFFFF;
  }

  /**
   * View this as an {@code int} to {@code long} function.
   * @return int to long function which will return values between
   *         {@code -32768L} and {@code 32767L}
   */
  @NotNull
  default IntToLongFunction asIntToLongFunction()
  {
    return this::applyAsShort;
  }

  /**
   * View this as an {@code int} to unsigned {@code long} function.
   * @return unary integer operator which will return values between
   *         {@code 0L} to {@code 65535L}
   */
  @NotNull
  default IntToLongFunction asIntToUnsignedLongFunction()
  {
    return v -> applyAsShort(v) & 0xFFFFL;
  }

  /**
   * View this as an {@code int} to {@code double} function.
   * @return int to double function which will return integral values between
   *         {@code -32768.0} and {@code 32767.0}
   */
  @NotNull
  default IntToDoubleFunction asIntToDoubleFunction()
  {
    return this::applyAsShort;
  }

  /**
   * View this as an {@code int} to unsigned {@code double} function.
   * @return unary integer operator which will return integral values between
   *         {@code 0.0} to {@code 65535.0}
   */
  @NotNull
  default IntToDoubleFunction asIntToUnsignedDoubleFunction()
  {
    return v -> applyAsShort(v) & 0xFFFF;
  }
}
