// ============================================================================
// File:               IntFunction2
//
// Project:            CAFF
//
// Purpose:            
//
// Author:             Rammi
//
// Copyright Notice:   © 2020-2024  Rammi (rammi@caff.de)
//                     The usage of this source code in commercial or open 
//                     source projects is not allowed without explicit 
//                     permission.
//
// Created:            27.11.20 13:37
//=============================================================================
package de.caff.generics.function;

import de.caff.annotation.NotNull;

import java.util.function.IntBinaryOperator;

/**
 * Function with three primitive integer arguments and a primitive integer result.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since November 27, 2020
 */
@FunctionalInterface
public interface IntToIntFunction3
{
  /**
   * Apply this function to the given 3 arguments.
   * @param value1 first argument
   * @param value2 second argument
   * @param value3 second argument
   * @return the function result
   */
  int applyAsInt(int value1, int value2, int value3);

  /**
   * View this function as one with 2 arguments using a fix first argument.
   * @param value1 fix first argument
   * @return function expecting the remaining 2 arguments in order
   */
  @NotNull
  default IntBinaryOperator partial1(int value1)
  {
    return (v2, v3) -> applyAsInt(value1, v2, v3);
  }

  /**
   * View this function as one with 2 arguments using a fix second argument.
   * @param value2 fix second argument
   * @return function expecting the remaining 2 arguments in order
   */
  @NotNull
  default IntBinaryOperator partial2(int value2)
  {
    return (v1, v3) -> applyAsInt(v1, value2, v3);
  }

  /**
   * View this function as one with 2 arguments using a fix third argument.
   * @param value3 fix third argument
   * @return function expecting the remaining 2 arguments in order
   */
  @NotNull
  default IntBinaryOperator partial3(int value3)
  {
    return (v1, v2) -> applyAsInt(v1, v2, value3);
  }

  /**
   * View this as an {@code int} to {@code long} function.
   * @return int to long function  which will return values between
   *         {@code -2147483648L} to {@code 2147483647L}
   */
  @NotNull
  default IntToLongFunction3 asIntToLongFunction()
  {
    return this::applyAsInt;
  }

  /**
   * View this as an {@code int} to unsigned {@code long} function.
   * @return unary integer operator which will return values between
   *         {@code 0L} to {@code 4294967295L}
   */
  @NotNull
  default IntToLongFunction3 asIntToUnsignedLongFunction()
  {
    return (v1, v2, v3) -> applyAsInt(v1, v2, v3) & 0xFFFF_FFFFL;
  }

  /**
   * View this as an {@code int} to {@code double} function.
   * @return int to double function which will return integral values between
   *         {@code -2147483648.0} to {@code 2147483647.0}
   */
  @NotNull
  default IntToDoubleFunction3 asIntToDoubleFunction()
  {
    return this::applyAsInt;
  }

  /**
   * View this as an {@code int} to unsigned {@code double} function.
   * @return int to double function  which will return integral values between
   *         {@code 0.0} to {@code 4294967295.0}
   */
  @NotNull
  default IntToDoubleFunction3 asIntToUnsignedDoubleFunction()
  {
    return (v1, v2, v3) -> applyAsInt(v1, v2, v3) & 0xFFFF_FFFFL;
  }
}
