// ============================================================================
// File:               IntFunction2
//
// Project:            CAFF
//
// Purpose:            
//
// Author:             Rammi
//
// Copyright Notice:   © 2020-2024  Rammi (rammi@caff.de)
//                     The usage of this source code in commercial or open 
//                     source projects is not allowed without explicit 
//                     permission.
//
// Created:            27.11.20 13:37
//=============================================================================
package de.caff.generics.function;

import de.caff.annotation.NotNull;

import java.util.function.*;

/**
 * Function with two primitive integer arguments and a primitive byte result.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since November 27, 2020
 */
@FunctionalInterface
public interface IntToCharFunction2
{
  /**
   * Apply this function to the given 2 arguments.
   * @param value1 first argument
   * @param value2 second argument
   * @return the function result
   */
  char applyAsChar(int value1, int value2);

  /**
   * View this function as one with 1 argument using a fix first argument.
   * @param value1 fix first argument
   * @return function expecting the remaining argument
   */
  @NotNull
  default IntToCharFunction1 partial1(int value1)
  {
    return v2 -> applyAsChar(value1, v2);
  }

  /**
   * View this function as one with 1 argument using a fix second argument.
   * @param value2 fix second argument
   * @return function expecting the remaining argument
   */
  @NotNull
  default IntToCharFunction1 partial2(int value2)
  {
    return v1 -> applyAsChar(v1, value2);
  }

  /**
   * View this as a 2 {@code int} to {@code short} function.
   * @return unary integer operator which will return values between
   *         {@code -32768} to {@code 32767}
   */
  @NotNull
  default IntToShortFunction2 asIntToShortFunction()
  {
    return (v1, v2) -> (short)applyAsChar(v1, v2);
  }

  /**
   * View this as a 2 {@code int} to {@code int} function.
   * @return unary integer operator which will return values between
   *         {@code 0} to {@code 65535}
   */
  @NotNull
  default IntBinaryOperator asIntToIntFunction()
  {
    return this::applyAsChar;
  }

  /**
   * View this as a 2 {@code int} to {@code long} function.
   * @return int to long function  which will return values between
   *         {@code 0L} to {@code 65535L}
   */
  @NotNull
  default IntToLongFunction2 asIntToLongFunction()
  {
    return this::applyAsChar;
  }

  /**
   * View this as a 2 {@code int} to {@code double} function.
   * @return int to double function which will return integral values between
   *         {@code 0.0} to {@code 65535.0}
   */
  @NotNull
  default IntToDoubleFunction2 asIntToDoubleFunction()
  {
    return this::applyAsChar;
  }
}
