// ============================================================================
// File:               IntFunction2
//
// Project:            CAFF
//
// Purpose:            
//
// Author:             Rammi
//
// Copyright Notice:   © 2020-2024  Rammi (rammi@caff.de)
//                     The usage of this source code in commercial or open 
//                     source projects is not allowed without explicit 
//                     permission.
//
// Created:            27.11.20 13:37
//=============================================================================
package de.caff.generics.function;

import de.caff.annotation.NotNull;

/**
 * Function with three primitive integer arguments and a primitive byte result.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since November 27, 2020
 */
@FunctionalInterface
public interface IntToByteFunction3
{
  /**
   * Apply this function to the given 2 arguments.
   * @param value1 first argument
   * @param value2 second argument
   * @param value3 third argument
   * @return the function result
   */
  byte applyAsByte(int value1, int value2, int value3);

  /**
   * View this function as one with 2 arguments using a fix first argument.
   * @param value1 fix first argument
   * @return function expecting the remaining 2 arguments in order
   */
  @NotNull
  default IntToByteFunction2 partial1(int value1)
  {
    return (v2, v3) -> applyAsByte(value1, v2, v3);
  }

  /**
   * View this function as one with 2 arguments using a fix second argument.
   * @param value2 fix second argument
   * @return function expecting the remaining 2 arguments in order
   */
  @NotNull
  default IntToByteFunction2 partial2(int value2)
  {
    return (v1, v3) -> applyAsByte(v1, value2, v3);
  }

  /**
   * View this function as one with 2 arguments using a fix third argument.
   * @param value3 fix third argument
   * @return function expecting the remaining 2 arguments in order
   */
  @NotNull
  default IntToByteFunction2 partial3(int value3)
  {
    return (v1, v2) -> applyAsByte(v1, v2, value3);
  }

  /**
   * View this as an {@code int} to {@code short} function.
   * @return unary integer operator which will return values between
   *         {@code -128} to {@code 127}
   */
  @NotNull
  default IntToShortFunction3 asIntToShortFunction()
  {
    return this::applyAsByte;
  }

  /**
   * View this as a 2 {@code int} to unsigned {@code short} function.
   * @return unary integer operator which will return values between
   *         {@code 0} to {@code 255}
   */
  @NotNull
  default IntToShortFunction3 asIntToUnsignedShortFunction()
  {
    return (v1, v2, v3) -> (short)(applyAsByte(v1, v2, v3) & 0xFF);
  }

  /**
   * View this as a 2 {@code int} to {@code int} function.
   * @return unary integer operator which will return values between
   *         {@code -128} to {@code 127}
   */
  @NotNull
  default IntToIntFunction3 asIntToIntFunction()
  {
    return this::applyAsByte;
  }

  /**
   * View this as a 2 {@code int} to unsigned {@code int} function.
   * @return unary integer operator which will return values between
   *         {@code 0} to {@code 255}
   */
  @NotNull
  default IntToIntFunction3 asIntToUnsignedIntFunction()
  {
    return (v1, v2, v3) -> applyAsByte(v1, v2, v3) & 0xFF;
  }

  /**
   * View this as a 2 {@code int} to {@code long} function.
   * @return int to long function  which will return values between
   *         {@code -128L} to {@code 127L}
   */
  @NotNull
  default IntToLongFunction3 asIntToLongFunction()
  {
    return this::applyAsByte;
  }

  /**
   * View this as a 2 {@code int} to unsigned {@code long} function.
   * @return unary integer operator which will return values between
   *         {@code 0L} to {@code 255L}
   */
  @NotNull
  default IntToLongFunction3 asIntToUnsignedLongFunction()
  {
    return (v1, v2, v3) -> applyAsByte(v1, v2, v3) & 0xFFL;
  }

  /**
   * View this as a 2 {@code int} to {@code double} function.
   * @return int to double function which will return integral values between
   *         {@code -128.0} to {@code 127.0}
   */
  @NotNull
  default IntToDoubleFunction3 asIntToDoubleFunction()
  {
    return this::applyAsByte;
  }

  /**
   * View this as a 2 {@code int} to unsigned {@code double} function.
   * @return int to double function  which will return integral values between
   *         {@code 0.0} to {@code 255.0}
   */
  @NotNull
  default IntToDoubleFunction3 asIntToUnsignedDoubleFunction()
  {
    return (v1, v2, v3) -> applyAsByte(v1, v2, v3) & 0XFF;
  }

  /**
   * View this as a 2 {@code int} to {@code char} function which
   * considers the byte values unsigned.
   * @return int to char function whih will return values between
   *         {@code (char)0} and {@code (char)255}
   */
  @NotNull
  default IntToCharFunction3 asIntToCharFunction()
  {
    return (v1, v2, v3) -> (char)(applyAsByte(v1, v2, v3) & 0xFF);
  }
}
