// ============================================================================
// File:               IntToByteFunction1
//
// Project:            CAFF
//
// Purpose:            
//
// Author:             Rammi
//
// Copyright Notice:   © 2020-2024  Rammi (rammi@caff.de)
//                     The usage of this source code in commercial or open 
//                     source projects is not allowed without explicit 
//                     permission.
//
// Created:            27.11.20 11:59
//=============================================================================
package de.caff.generics.function;

import de.caff.annotation.NotNull;

import java.util.function.IntToDoubleFunction;
import java.util.function.IntToLongFunction;
import java.util.function.IntUnaryOperator;

/**
 * Function with one primitive integer argument and a primitive byte return.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since November 27, 2020
 */
@FunctionalInterface
public interface IntToByteFunction1
{
  /**
   /**
   * Applies this function to the given argument.
   *
   * @param value the function argument
   * @return the function result
   */
  byte applyAsByte(int value);

  /**
   * View this as an {@code int} to {@code short} function.
   * @return unary integer operator which will return values between
   *         {@code -128} to {@code 127}
   */
  @NotNull
  default IntToShortFunction1 asIntToShortFunction()
  {
    return this::applyAsByte;
  }

  /**
   * View this as an {@code int} to unsigned {@code short} function.
   * @return unary integer operator which will return values between
   *         {@code 0} to {@code 255}
   */
  @NotNull
  default IntToShortFunction1 asIntToUnsignedShortFunction()
  {
    return v -> (short)(applyAsByte(v) & 0xFF);
  }

  /**
   * View this as an {@code int} to {@code int} function.
   * @return unary integer operator which will return values between
   *         {@code -128} to {@code 127}
   */
  @NotNull
  default IntUnaryOperator asIntToIntFunction()
  {
    return this::applyAsByte;
  }

  /**
   * View this as an {@code int} to unsigned {@code int} function.
   * @return unary integer operator which will return values between
   *         {@code 0} to {@code 255}
   */
  @NotNull
  default IntUnaryOperator asIntToUnsignedIntFunction()
  {
    return v -> applyAsByte(v) & 0xFF;
  }

  /**
   * View this as an {@code int} to {@code long} function.
   * @return int to long function  which will return values between
   *         {@code -128L} to {@code 127L}
   */
  @NotNull
  default IntToLongFunction asIntToLongFunction()
  {
    return this::applyAsByte;
  }

  /**
   * View this as an {@code int} to unsigned {@code long} function.
   * @return unary integer operator which will return values between
   *         {@code 0L} to {@code 255L}
   */
  @NotNull
  default IntToLongFunction asIntToUnsignedLongFunction()
  {
    return v -> applyAsByte(v) & 0xFFL;
  }

  /**
   * View this as an {@code int} to {@code double} function.
   * @return int to double function which will return integral values between
   *         {@code -128.0} to {@code 127.0}
   */
  @NotNull
  default IntToDoubleFunction asIntToDoubleFunction()
  {
    return this::applyAsByte;
  }

  /**
   * View this as an {@code int} to unsigned {@code double} function.
   * @return int to double function  which will return integral values between
   *         {@code 0.0} to {@code 255.0}
   */
  @NotNull
  default IntToDoubleFunction asIntToUnsignedDoubleFunction()
  {
    return v -> applyAsByte(v) & 0xFF;
  }

  /**
   * View this as an {@code int} to {@code char} function which
   * considers the byte values unsigned.
   * @return int to char function whih will return values between
   *         {@code (char)0} and {@code (char)255}
   */
  @NotNull
  default IntToCharFunction1 asIntToCharFunction()
  {
    return v -> (char)(applyAsByte(v) & 0xFF);
  }
}
