// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright ©2023-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.function;

import de.caff.annotation.NotNull;
import de.caff.generics.Order;

import java.io.Serializable;
import java.util.Comparator;

/**
 * Define an order for two primitive int values.
 * <p>
 * Compared to a {@code Comparator<Integer>} this can
 * avoid boxing and unboxing in various situations.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since January 23, 2023
 */
@FunctionalInterface
public interface IntOrdering
        extends ByteOrdering,
                ShortOrdering,
                CharOrdering
{
  /**
   * Natural order of ascending {@code int} values.
   */
  IntOrdering ASCENDING = new IntOrdering.Serial()
  {
    private static final long serialVersionUID = -7646031177018732313L;

    @NotNull
    @Override
    public Order checkInt(int v1, int v2)
    {
      return (v1 < v2)
              ? Order.Ascending
              : (v1 > v2
                         ? Order.Descending
                         : Order.Same);
    }

    @NotNull
    @Override
    public IntOrdering inverse()
    {
      return IntOrdering.DESCENDING;
    }

    @NotNull
    @Override
    public Comparator<Integer> asIntComparator()
    {
      return Integer::compare;
    }
  };

  /** Inverse natural order of {@code int} values. */
  IntOrdering DESCENDING = new IntOrdering.Serial()
  {
    private static final long serialVersionUID = 3026851066329505442L;

    @NotNull
    @Override
    public Order checkInt(int v2, int v1)
    {
      return (v1 < v2)
              ? Order.Ascending
              : (v1 > v2
                         ? Order.Descending
                         : Order.Same);
    }

    @NotNull
    @Override
    public IntOrdering inverse()
    {
      return IntOrdering.ASCENDING;
    }

    @NotNull
    @Override
    public Comparator<Integer> asIntComparator()
    {
      return (Comparator<Integer> & Serializable) (v1, v2) -> Integer.compare(v2, v1);
    }
  };

  /**
   * Natural order of ascending unsigned {@code int} values.
   * This handles int values as if they are unsigned and not signed.
   */
  IntOrdering UNSIGNED_ASCENDING = new IntOrdering.Serial()
  {
    private static final long serialVersionUID = -3495520630473840505L;

    @NotNull
    @Override
    public Order checkInt(int v1, int v2)
    {
      return ASCENDING.checkInt(v1 + Integer.MIN_VALUE,
                                 v2 + Integer.MIN_VALUE);
    }

    @NotNull
    @Override
    public IntOrdering inverse()
    {
      return IntOrdering.UNSIGNED_DESCENDING;
    }

    @NotNull
    @Override
    public Comparator<Integer> asIntComparator()
    {
      return Integer::compareUnsigned;
    }
  };

  /**
   * Natural order of ascending unsigned {@code int} values.
   * This handles int values as if they are unsigned and not signed.
   */
  IntOrdering UNSIGNED_DESCENDING = new IntOrdering.Serial()
  {
    private static final long serialVersionUID = 8393916740777317119L;

    @NotNull
    @Override
    public Order checkInt(int v1, int v2)
    {
      return DESCENDING.checkInt(v1 + Integer.MIN_VALUE,
                                  v2 + Integer.MIN_VALUE);
    }

    @NotNull
    @Override
    public IntOrdering inverse()
    {
      return IntOrdering.UNSIGNED_ASCENDING;
    }

    @NotNull
    @Override
    public Comparator<Integer> asIntComparator()
    {
      return (Comparator<Integer> & Serializable) (v1, v2) -> Integer.compareUnsigned(v2, v1);
    }
  };

  /**
   * Get the order of the two values.
   * @param v1 first value
   * @param v2 second value
   * @return order of the two values
   */
  @NotNull
  Order checkInt(int v1, int v2);

  @NotNull
  @Override
  default Order checkByte(byte v1, byte v2)
  {
    return checkInt(v1, v2);
  }

  @NotNull
  @Override
  default Order checkChar(char v1, char v2)
  {
    return checkInt(v1, v2);
  }

  @NotNull
  @Override
  default Order checkShort(short v1, short v2)
  {
    return checkInt(v1, v2);
  }

  /**
   * Are the two values represented in strictly ascending order?
   * @param v1 first value
   * @param v2 second value
   * @return {@code true} if {@code v1 < v2} according to this ordering<br>
   *         {@code false} if {@code v1 >= v2} according to this ordering
   * @see #descending(int, int)
   * @see #ascendingOrSame(int, int)
   * @see #descendingOrSame(int, int)
   * @see #same(int, int)
   * @see #different(int, int)
   */
  default boolean ascending(int v1, int v2)
  {
    return checkInt(v1, v2).ascending;
  }

  /**
   * Are the two values represented in ascending order?
   * @param v1 first value
   * @param v2 second value
   * @return {@code true} if {@code v1 <= v2} according to this ordering<br>
   *         {@code false} if {@code v1 > v2} according to this ordering
   * @see #ascending(int, int)
   * @see #descending(int, int)
   * @see #descendingOrSame(int, int)
   * @see #same(int, int)
   * @see #different(int, int)
   */
  default boolean ascendingOrSame(int v1, int v2)
  {
    return checkInt(v1, v2).ascendingOrSame;
  }

  /**
   * Are the two values represented in strictly descending order?
   * @param v1 first value
   * @param v2 second value
   * @return {@code true} if {@code v1 > v2} according to this ordering<br>
   *         {@code false} if {@code v1 <= v2} according to this ordering
   * @see #ascending(int, int)
   * @see #ascendingOrSame(int, int)
   * @see #descendingOrSame(int, int)
   * @see #same(int, int)
   * @see #different(int, int)
   */
  default boolean descending(int v1, int v2)
  {
    return checkInt(v1, v2).descending;
  }

  /**
   * Are the two values represented in descending order?
   * @param v1 first value
   * @param v2 second value
   * @return {@code true} if {@code v1 >= v2} according to this ordering<br>
   *         {@code false} if {@code v1 < v2} according to this ordering
   * @see #ascending(int, int)
   * @see #descending(int, int)
   * @see #ascendingOrSame(int, int)
   * @see #same(int, int)
   * @see #different(int, int)
   */
  default boolean descendingOrSame(int v1, int v2)
  {
    return checkInt(v1, v2).descendingOrSame;
  }

  /**
   * Are the two values considered equal by this order?
   * @param v1 first value
   * @param v2 second value
   * @return {@code true} if {@code v1 == v2} according to this order<br>
   *         {@code false} if {@code v1 != v2} according to this ordering
   * @see #ascending(int, int)
   * @see #descending(int, int)
   * @see #ascendingOrSame(int, int)
   * @see #descendingOrSame(int, int)
   * @see #different(int, int)
   */
  default boolean same(int v1, int v2)
  {
    return checkInt(v1, v2).same;
  }

  /**
   * Are the two values considered different by this order?
   * @param v1 first value
   * @param v2 second value
   * @return {@code true} if {@code v1 != v2} according to this order<br>
   *         {@code false} if {@code v1 == v2} according to this ordering
   * @see #ascending(int, int)
   * @see #descending(int, int)
   * @see #ascendingOrSame(int, int)
   * @see #descendingOrSame(int, int)
   * @see #same(int, int)
   */
  default boolean different(int v1, int v2)
  {
    return checkInt(v1, v2).different;
  }

  /**
   * Invert this order.
   * @return inverse order
   */
  @NotNull
  default IntOrdering inverse()
  {
    return new IntOrdering.Serial()
    {
      private static final long serialVersionUID = 3270504639153727362L;

      @Override
      @NotNull
      public Order checkInt(int v1, int v2)
      {
        return IntOrdering.this.checkInt(v2, v1);
      }

      @NotNull
      @Override
      public IntOrdering inverse()
      {
        return IntOrdering.this;
      }
    };
  }

  /**
   * Use this ordering as a comparator.
   * @return comparator of {@code Int}
   */
  @NotNull
  default Comparator<Integer> asIntComparator()
  {
    return (Comparator<Integer> & Serializable) (v1, v2) -> checkInt(v1, v2).comparison;
  }

  /**
   * View this ordering as a number comparator.
   * @return comparator of {@link Number}
   */
  @NotNull
  default Comparator<Number> asNumberComparator()
  {
    return (Comparator<Number> & Serializable)(v1, v2) -> checkInt(v1.intValue(), v2.intValue()).comparison;
  }

  /**
   * Get this ordering as a generic int ordering. 
   * @return generic ordering for ints
   */
  @NotNull
  default Ordering<Integer> asIntOrdering()
  {
    return (Ordering<Integer> & Serializable)this::checkInt;
  }

  /**
   * Convert a standard comparator of {@code Double} into a double ordering.
   * @param comparator comparator of {@code Double}
   * @return ordering
   */
  @NotNull
  static IntOrdering fromComparator(@NotNull Comparator<Integer> comparator)
  {
    return new IntOrdering.Serial()
    {
      private static final long serialVersionUID = 393576352771473074L;

      @NotNull
      @Override
      public Order checkInt(int v1, int v2)
      {
        return Order.fromCompare(comparator.compare(v1, v2));
      }

      @NotNull
      @Override
      public Comparator<Integer> asIntComparator()
      {
        return comparator;
      }
    };
  }

  /**
   * An int ordering which is serializable.
   * It is a recommendation that comparators implement {@code java.io.Serializable}
   * to allow serializing {@code TreeMap}s and similar. If an Ordering is expected to be used
   * as a comparator in such areas implement either this interface, or for lambdas
   * use a cast to force the compiler to make the order serializable:
   * <pre>{@code
   *   return (IntOrdering & Serializable) (v1, v2) -> v1.compareTo(v2);
   * }</pre>
   * You can also use a {@code (IntOrdering.Serial)} cast to save a bit of typing,
   * but the above is a nice trick to have in your toolbox.
   */
  interface Serial extends IntOrdering,
                           java.io.Serializable
  {
  }
}
