// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2019-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.function;

import de.caff.annotation.NotNull;

import java.util.function.IntBinaryOperator;

/**
 * Binary operator with two int arguments and a int result.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since October 31, 2019
 */
@FunctionalInterface
public interface IntOperator2
        extends IntBinaryOperator
{
  /**
   * PLUS ({@code +}) operator.
   */
  IntOperator2 PLUS = (x1, x2) -> (x1 + x2);
  /**
   * MINUS ({@code -}) operator.
   */
  IntOperator2 MINUS = (x1, x2) -> (x1 - x2);
  /**
   * MULTIPLICATION ({@code *}) operator.
   */
  IntOperator2 MULT = (x1, x2) -> (x1 * x2);
  /**
   * DIVISION ({@code /}) operator.
   */
  IntOperator2 DIV = (x1, x2) -> (x1 / x2);
  /**
   * MODULO ({@code %}) operator.
   */
  IntOperator2 MOD = (x1, x2) -> (x1 % x2);

  /**
   * Get a partially applied function.
   * @param argument value applied for first argument
   * @return partially evaluated function
   */
  @NotNull
  default IntOperator1 partialLeft(int argument)
  {
    return arg -> applyAsInt(argument, arg);
  }

  /**
   * Get a partially applied function.
   * @param argument value applied for second argument
   * @return partially evaluated function
   */
  @NotNull
  default IntOperator1 partialRight(int argument)
  {
    return arg -> applyAsInt(arg, argument);
  }

  /**
   * Create a function which applies this function and feeds the result into the next function.
   * @param after function called with the result of this function
   * @return chained function
   */
  @NotNull
  default IntOperator2 andThen(@NotNull IntOperator1 after)
  {
    return (arg1, arg2) -> after.applyAsInt(applyAsInt(arg1, arg2));
  }

  /**
   * Make a standard unary int operator usable as {@code IntOperator2}.
   * This is useful if you want to make use of the extended functionality of this implementation.
   * @param operator operator to be used as a {@code IntOperator2}
   * @return {@code IntOperator2} view of the incoming operator, or possibly just a cast if 
   *         {@code operator} already has the correct type
   */
  @NotNull
  static IntOperator2 from(@NotNull IntBinaryOperator operator)
  {
    return operator instanceof IntOperator2
            ? (IntOperator2) operator
            : operator::applyAsInt;
  }
}
