// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2019-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.function;

import de.caff.annotation.NotNull;

import java.util.function.IntUnaryOperator;

/**
 * Unary operator with a int argument and result.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since October 31, 2019
 */
@FunctionalInterface
public interface IntOperator1
        extends IntUnaryOperator
{
  /**
   * Integer operator NEGATE.
   * Beware: as with usual NEGATE not only {@code 0} also {@link Integer#MIN_VALUE} is its own negative.
   */
  IntOperator1 NEGATE = x -> -x;
  /**
   * Integer operator BIT FLIP ({@code ~}).
   */
  IntOperator1 BIT_FLIP = x -> ~x;

  /**
   * Create a zero argument function from this one by applying a constant argument.
   * Although useless for pure functions it can be useful in cases where the zero
   * argument function is a factory (and therefore not pure).
   * @param argument constant argument
   * @return zero argument function
   */
  @NotNull
  default IntOperator0 partial(int argument)
  {
    return () -> applyAsInt(argument);
  }

  /**
   * Create a function which applies this function and feeds the result into the next function.
   * @param after function called with the result of this function
   * @return chained function
   */
  @NotNull
  default IntOperator1 andThen(@NotNull IntOperator1 after)
  {
    return (arg) -> after.applyAsInt(applyAsInt(arg));
  }

  /**
   * Make a standard unary int operator usable as {@code IntOperator1}.
   * This is useful if you want to make use of the extended functionality of this implementation.
   * @param operator operator to be used as a {@code IntOperator1}
   * @return {@code IntOperator1} view of the incoming operator, or possibly just a cast if
   *         {@code operator} already has the correct type
   */
  @NotNull
  static IntOperator1 from(@NotNull IntUnaryOperator operator)
  {
    return operator instanceof IntOperator1
            ? (IntOperator1) operator
            : operator::applyAsInt;
  }
}
