// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.function;

import de.caff.annotation.NotNull;
import de.caff.generics.Types;
import de.caff.generics.tuple.ITuple9;

import java.util.function.BiFunction;
import java.util.function.Function;

/**
 * Function with nine parameters.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @param <R> return type
 * @param <P1> parameter type 1
 * @param <P2> parameter type 2
 * @param <P3> parameter type 3
 * @param <P4> parameter type 4
 * @param <P5> parameter type 5
 * @param <P6> parameter type 6
 * @param <P7> parameter type 7
 * @param <P8> parameter type 8
 * @param <P9> parameter type 9
 * @see Function0
 * @see Function1
 * @see Function2
 * @see Function3
 * @see Function4
 * @see Function5
 * @see Function6
 * @see Function7
 * @see Function8
 */
@FunctionalInterface
public interface Function9<R, P1, P2, P3, P4, P5, P6, P7, P8, P9>
        extends Function1<R, ITuple9<? extends P1, ? extends P2, ? extends P3,
                                     ? extends P4, ? extends P5, ? extends P6,
                                     ? extends P7, ? extends P8, ? extends P9>>
{
  /**
   * Run the function.
   * @param arg1 function argument 1
   * @param arg2 function argument 2
   * @param arg3 function argument 3
   * @param arg4 function argument 4
   * @param arg5 function argument 5
   * @param arg6 function argument 6
   * @param arg7 function argument 7
   * @param arg8 function argument 8
   * @param arg9 function argument 9
   * @return return value
   */
  R apply(P1 arg1, P2 arg2, P3 arg3, P4 arg4, P5 arg5, P6 arg6, P7 arg7, P8 arg8, P9 arg9);

  /**
   * Expand a tuple into the arguments of this function.
   * @param tuple9 tuple to expand
   * @return the result of calling this function with the arguments set from the tuple
   */
  @Override
  default R apply(@NotNull ITuple9<? extends P1, ? extends P2, ? extends P3, ? extends P4, ? extends P5, ? extends P6, ? extends P7, ? extends P8, ? extends P9> tuple9)
  {
    return tuple9.invoke(this);
  }

  /**
   * Apply this function, but take care to create a fallback.
   * Thiw will return the fallback when {@link #apply(Object, Object, Object, Object, Object, Object, Object, Object, Object)} returns {@code null}.
   * @param arg1 first argument of this function
   * @param arg2 second argument of this function
   * @param arg3 third argument of this function
   * @param arg4 fourth argument of this function
   * @param arg5 fifth argument of this function
   * @param arg6 sixth argument of this function
   * @param arg7 seventh argument of this function
   * @param arg8 eighth argument of this function
   * @param arg9 ninth argument of this function
   * @param fallback fallback value returned if function returns {@code null}
   * @return the non-null result of calling {@link #apply(Object, Object, Object, Object, Object, Object, Object, Object, Object)}, or {@code fallback}
   *         if the result was {@code null}
   */
  @NotNull
  default R applyOrDefault(P1 arg1, P2 arg2, P3 arg3, P4 arg4, P5 arg5, P6 arg6, P7 arg7, P8 arg8, P9 arg9, @NotNull R fallback)
  {
    return Types.notNull(apply(arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8, arg9), fallback);
  }

  /**
   * Get a partially applied function.
   * @param argument value applied for first argument
   * @return partially evaluated function
   */
  @NotNull
  default Function8<R, P2, P3, P4, P5, P6, P7, P8, P9> partial1(P1 argument)
  {
    return (arg2, arg3, arg4, arg5, arg6, arg7, arg8, arg9) -> apply(argument, arg2, arg3, arg4, arg5, arg6, arg7, arg8, arg9);
  }

  /**
   * Get a partially applied function.
   * @param argument value applied for second argument
   * @return partially evaluated function
   */
  @NotNull
  default Function8<R, P1, P3, P4, P5, P6, P7, P8, P9> partial2(P2 argument)
  {
    return (arg1, arg3, arg4, arg5, arg6, arg7, arg8, arg9) -> apply(arg1, argument, arg3, arg4, arg5, arg6, arg7, arg8, arg9);
  }

  /**
   * Get a partially applied function.
   * @param argument value applied for third argument
   * @return partially evaluated function
   */
  @NotNull
  default Function8<R, P1, P2, P4, P5, P6, P7, P8, P9> partial3(P3 argument)
  {
    return (arg1, arg2, arg4, arg5, arg6, arg7, arg8, arg9) -> apply(arg1, arg2, argument, arg4, arg5, arg6, arg7, arg8, arg9);
  }

  /**
   * Get a partially applied function.
   * @param argument value applied for fourth argument
   * @return partially evaluated function
   */
  @NotNull
  default Function8<R, P1, P2, P3, P5, P6, P7, P8, P9> partial4(P4 argument)
  {
    return (arg1, arg2, arg3, arg5, arg6, arg7, arg8, arg9) -> apply(arg1, arg2, arg3, argument, arg5, arg6, arg7, arg8, arg9);
  }

  /**
   * Get a partially applied function.
   * @param argument value applied for fifth argument
   * @return partially evaluated function
   */
  @NotNull
  default Function8<R, P1, P2, P3, P4, P6, P7, P8, P9> partial5(P5 argument)
  {
    return (arg1, arg2, arg3, arg4, arg6, arg7, arg8, arg9) -> apply(arg1, arg2, arg3, arg4, argument, arg6, arg7, arg8, arg9);
  }

  /**
   * Get a partially applied function.
   * @param argument value applied for sixth argument
   * @return partially evaluated function
   */
  @NotNull
  default Function8<R, P1, P2, P3, P4, P5, P7, P8, P9> partial6(P6 argument)
  {
    return (arg1, arg2, arg3, arg4, arg5, arg7, arg8, arg9) -> apply(arg1, arg2, arg3, arg4, arg5, argument, arg7, arg8, arg9);
  }

  /**
   * Get a partially applied function.
   * @param argument value applied for seventh argument
   * @return partially evaluated function
   */
  @NotNull
  default Function8<R, P1, P2, P3, P4, P5, P6, P8, P9> partial7(P7 argument)
  {
    return (arg1, arg2, arg3, arg4, arg5, arg6, arg8, arg9) -> apply(arg1, arg2, arg3, arg4, arg5, arg6, argument, arg8, arg9);
  }

  /**
   * Get a partially applied function.
   * @param argument value applied for eighth argument
   * @return partially evaluated function
   */
  @NotNull
  default Function8<R, P1, P2, P3, P4, P5, P6, P7, P9> partial8(P8 argument)
  {
    return (arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg9) -> apply(arg1, arg2, arg3, arg4, arg5, arg6, arg7, argument, arg9);
  }

  /**
   * Get a partially applied function.
   * @param argument value applied for eighth argument
   * @return partially evaluated function
   */
  @NotNull
  default Function8<R, P1, P2, P3, P4, P5, P6, P7, P8> partial9(P9 argument)
  {
    return (arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8) -> apply(arg1, arg2, arg3, arg4, arg5, arg6, arg7, arg8, argument);
  }

  /**
   * Get a chained function which first applies this function
   * and then the given function to the returned value.
   * @param after function to apply after this one
   * @param <T> result type of returned function
   * @return function chain
   */
  @NotNull
  default <T> Function9<T, P1, P2, P3, P4, P5, P6, P7, P8, P9> andThen(@NotNull Function<? super R, ? extends T> after)
  {
    return (v1, v2, v3, v4, v5, v6, v7, v8, v9) -> after.apply(apply(v1, v2, v3, v4, v5, v6, v7, v8, v9));
  }

  /**
   * Get a chained function which first applies the given function, then this one.
   * The given function has to return a {@link ITuple9} with matching parametric types
   * which are applied in their order to this function.
   * @param before function to apply first, has to return a non-{@code null} result so unpacking the parameters
   *               from the tuple is possible
   * @param <FR> function {@code before}'s return type, has to be a 9-tuple with matching types
   * @param <FP> function {@code before}'s parameter type, same for returned function
   * @return function chain
   */
  @NotNull
  default <FR extends ITuple9<? extends P1, ? extends P2, ? extends P3, ? extends P4, ? extends P5, ? extends P6, ? extends P7, ? extends P8, ? extends P9>, FP>
  Function1<R, FP> after1(@NotNull Function<FP, FR> before)
  {
    return fp -> before.apply(fp).invoke(this);
  }

  /**
   * Get a chained function which first applies the given 2-parameter function, then this one.
   * The given function has to return a {@link ITuple9} with matching parametric types
   * which are applied in their order to this function.
   * @param before function to apply first, has to return a non-{@code null} result so unpacking the parameters
   *               from the tuple is possible
   * @param <FR> function {@code before}'s return type, has to be a 9-tuple with matching types
   * @param <FP1> function {@code before}'s first parameter type, sane for returned function
   * @param <FP2> function {@code before}'s second parameter type, same for returned function
   * @return function chain
   */
  @NotNull
  default <FR extends ITuple9<? extends P1, ? extends P2, ? extends P3, ? extends P4, ? extends P5, ? extends P6, ? extends P7, ? extends P8, ? extends P9>, FP1, FP2>
  Function2<R, FP1, FP2> after2(@NotNull BiFunction<FP1, FP2, FR> before)
  {
    return (fp1, fp2) -> before.apply(fp1, fp2).invoke(this);
  }

  /**
   * Get a chained function which first applies the given 3-parameter function, then this one.
   * The given function has to return a {@link ITuple9} with matching parametric types
   * which are applied in their order to this function.
   * @param before function to apply first, has to return a non-{@code null} result so unpacking the parameters
   *               from the tuple is possible
   * @param <FR> function {@code before}'s return type, has to be a 9-tuple with matching types
   * @param <FP1> function {@code before}'s first parameter type, sane for returned function
   * @param <FP2> function {@code before}'s second parameter type, sane for returned function
   * @param <FP3> function {@code before}'s third parameter type, sane for returned function
   * @return function chain
   */
  @NotNull
  default <FR extends ITuple9<? extends P1, ? extends P2, ? extends P3, ? extends P4, ? extends P5, ? extends P6, ? extends P7, ? extends P8, ? extends P9>, FP1, FP2, FP3>
  Function3<R, FP1, FP2, FP3> after3(@NotNull Function3<FR, FP1, FP2, FP3> before)
  {
    return (fp1, fp2, fp3) -> before.apply(fp1, fp2, fp3).invoke(this);
  }

  /**
   * Get a chained function which first applies the given 5-parameter function, then this one.
   * The given function has to return a {@link ITuple9} with matching parametric types
   * which are applied in their order to this function.
   * @param before function to apply first, has to return a non-{@code null} result so unpacking the parameters
   *               from the tuple is possible
   * @param <FR> function {@code before}'s return type, has to be a 9-tuple with matching types
   * @param <FP1> function {@code before}'s first parameter type, sane for returned function
   * @param <FP2> function {@code before}'s second parameter type, sane for returned function
   * @param <FP3> function {@code before}'s third parameter type, sane for returned function
   * @param <FP4> function {@code before}'s fourth parameter type, sane for returned function
   * @param <FP5> function {@code before}'s fifth parameter type, sane for returned function
   * @return function chain
   */
  @NotNull
  default <FR extends ITuple9<? extends P1, ? extends P2, ? extends P3, ? extends P4, ? extends P5, ? extends P6, ? extends P7, ? extends P8, ? extends P9>, FP1, FP2, FP3, FP4, FP5>
  Function5<R, FP1, FP2, FP3, FP4, FP5> after5(@NotNull Function5<FR, FP1, FP2, FP3, FP4, FP5> before)
  {
    return (fp1, fp2, fp3, fp4, fp5) -> before.apply(fp1, fp2, fp3, fp4, fp5).invoke(this);
  }

  /**
   * Get a chained function which first applies the given 6-parameter function, then this one.
   * The given function has to return a {@link ITuple9} with matching parametric types
   * which are applied in their order to this function.
   * @param before function to apply first, has to return a non-{@code null} result so unpacking the parameters
   *               from the tuple is possible
   * @param <FR> function {@code before}'s return type, has to be a 9-tuple with matching types
   * @param <FP1> function {@code before}'s first parameter type, sane for returned function
   * @param <FP2> function {@code before}'s second parameter type, sane for returned function
   * @param <FP3> function {@code before}'s third parameter type, sane for returned function
   * @param <FP4> function {@code before}'s fourth parameter type, sane for returned function
   * @param <FP5> function {@code before}'s fifth parameter type, sane for returned function
   * @param <FP6> function {@code before}'s sixth parameter type, sane for returned function
   * @return function chain
   */
  @NotNull
  default <FR extends ITuple9<? extends P1, ? extends P2, ? extends P3, ? extends P4, ? extends P5, ? extends P6, ? extends P7, ? extends P8, ? extends P9>, FP1, FP2, FP3, FP4, FP5, FP6>
  Function6<R, FP1, FP2, FP3, FP4, FP5, FP6> after6(@NotNull Function6<FR, FP1, FP2, FP3, FP4, FP5, FP6> before)
  {
    return (fp1, fp2, fp3, fp4, fp5, fp6) -> before.apply(fp1, fp2, fp3, fp4, fp5, fp6).invoke(this);
  }

  /**
   * Get a chained function which first applies the given 7-parameter function, then this one.
   * The given function has to return a {@link ITuple9} with matching parametric types
   * which are applied in their order to this function.
   * @param before function to apply first, has to return a non-{@code null} result so unpacking the parameters
   *               from the tuple is possible
   * @param <FR> function {@code before}'s return type, has to be a 9-tuple with matching types
   * @param <FP1> function {@code before}'s first parameter type, sane for returned function
   * @param <FP2> function {@code before}'s second parameter type, sane for returned function
   * @param <FP3> function {@code before}'s third parameter type, sane for returned function
   * @param <FP4> function {@code before}'s fourth parameter type, sane for returned function
   * @param <FP5> function {@code before}'s fifth parameter type, sane for returned function
   * @param <FP6> function {@code before}'s sixth parameter type, sane for returned function
   * @param <FP7> function {@code before}'s seventh parameter type, sane for returned function
   * @return function chain
   */
  @NotNull
  default <FR extends ITuple9<? extends P1, ? extends P2, ? extends P3, ? extends P4, ? extends P5, ? extends P6, ? extends P7, ? extends P8, ? extends P9>, FP1, FP2, FP3, FP4, FP5, FP6, FP7>
  Function7<R, FP1, FP2, FP3, FP4, FP5, FP6, FP7> after7(@NotNull Function7<FR, FP1, FP2, FP3, FP4, FP5, FP6, FP7> before)
  {
    return (fp1, fp2, fp3, fp4, fp5, fp6, fp7) -> before.apply(fp1, fp2, fp3, fp4, fp5, fp6, fp7).invoke(this);
  }

  /**
   * Get a chained function which first applies the given 8-parameter function, then this one.
   * The given function has to return a {@link ITuple9} with matching parametric types
   * which are applied in their order to this function.
   * @param before function to apply first, has to return a non-{@code null} result so unpacking the parameters
   *               from the tuple is possible
   * @param <FR> function {@code before}'s return type, has to be a 9-tuple with matching types
   * @param <FP1> function {@code before}'s first parameter type, sane for returned function
   * @param <FP2> function {@code before}'s second parameter type, sane for returned function
   * @param <FP3> function {@code before}'s third parameter type, sane for returned function
   * @param <FP4> function {@code before}'s fourth parameter type, sane for returned function
   * @param <FP5> function {@code before}'s fifth parameter type, sane for returned function
   * @param <FP6> function {@code before}'s sixth parameter type, sane for returned function
   * @param <FP7> function {@code before}'s seventh parameter type, sane for returned function
   * @param <FP8> function {@code before}'s seventh parameter type, sane for returned function
   * @return function chain
   */
  @NotNull
  default <FR extends ITuple9<? extends P1, ? extends P2, ? extends P3, ? extends P4, ? extends P5, ? extends P6, ? extends P7, ? extends P8, ? extends P9>, FP1, FP2, FP3, FP4, FP5, FP6, FP7, FP8>
  Function8<R, FP1, FP2, FP3, FP4, FP5, FP6, FP7, FP8> after8(@NotNull Function8<FR, FP1, FP2, FP3, FP4, FP5, FP6, FP7, FP8> before)
  {
    return (fp1, fp2, fp3, fp4, fp5, fp6, fp7, fp8) -> before.apply(fp1, fp2, fp3, fp4, fp5, fp6, fp7, fp8).invoke(this);
  }

  /**
   * Get a chained function which first applies the given 9-parameter function, then this one.
   * The given function has to return a {@link ITuple9} with matching parametric types
   * which are applied in their order to this function.
   * @param before function to apply first, has to return a non-{@code null} result so unpacking the parameters
   *               from the tuple is possible
   * @param <FR> function {@code before}'s return type, has to be a 9-tuple with matching types
   * @param <FP1> function {@code before}'s first parameter type, sane for returned function
   * @param <FP2> function {@code before}'s second parameter type, sane for returned function
   * @param <FP3> function {@code before}'s third parameter type, sane for returned function
   * @param <FP4> function {@code before}'s fourth parameter type, sane for returned function
   * @param <FP5> function {@code before}'s fifth parameter type, sane for returned function
   * @param <FP6> function {@code before}'s sixth parameter type, sane for returned function
   * @param <FP7> function {@code before}'s seventh parameter type, sane for returned function
   * @param <FP8> function {@code before}'s seventh parameter type, sane for returned function
   * @param <FP9> function {@code before}'s ninth parameter type, sane for returned function
   * @return function chain
   */
  @NotNull
  default <FR extends ITuple9<? extends P1, ? extends P2, ? extends P3, ? extends P4, ? extends P5, ? extends P6, ? extends P7, ? extends P8, ? extends P9>, FP1, FP2, FP3, FP4, FP5, FP6, FP7, FP8, FP9>
  Function9<R, FP1, FP2, FP3, FP4, FP5, FP6, FP7, FP8, FP9> after9(@NotNull Function9<FR, FP1, FP2, FP3, FP4, FP5, FP6, FP7, FP8, FP9> before)
  {
    return (fp1, fp2, fp3, fp4, fp5, fp6, fp7, fp8, fp9) -> before.apply(fp1, fp2, fp3, fp4, fp5, fp6, fp7, fp8, fp9).invoke(this);
  }
}
