// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.function;

import de.caff.annotation.NotNull;
import de.caff.generics.Types;
import de.caff.generics.WrappedFragileException;
import de.caff.generics.tuple.ITuple3;

import java.util.function.BiFunction;
import java.util.function.Function;

/**
 * Function with three parameters.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @param <R> return type
 * @param <P1> parameter type 1
 * @param <P2> parameter type 2
 * @param <P3> parameter type 3
 * @see Function0
 * @see Function1
 * @see Function2
 * @see Function4
 * @see Function5
 * @see Function6
 * @see Function7
 * @see Function8
 * @see Function9
 */
@FunctionalInterface
public interface Function3<R, P1, P2, P3>
        extends Function1<R, ITuple3<? extends P1, ? extends P2, ? extends P3>>
{
  /**
   * Run the function.
   * @param arg1 function argument 1
   * @param arg2 function argument 2
   * @param arg3 function argument 3
   * @return return value
   */
  R apply(P1 arg1, P2 arg2, P3 arg3);

  /**
   * Expand a tuple into the arguments of this function.
   * @param tuple3 tuple to expand
   * @return the result of calling this function with the arguments set from the tuple
   */
  @Override
  default R apply(@NotNull ITuple3<? extends P1, ? extends P2, ? extends P3> tuple3)
  {
    return tuple3.invoke(this);
  }

  /**
   * Apply this function, but take care to create a fallback.
   * This will return the fallback when {@link #apply(Object, Object, Object)} returns {@code null}.
   * @param arg1 first argument of this function
   * @param arg2 second argument of this function
   * @param arg3 third argument of this function
   * @param fallback fallback value returned if function returns {@code null}
   * @return the non-null result of calling {@link #apply(Object, Object, Object)}, or {@code fallback}
   *         if the result was {@code null}
   */
  @NotNull
  default R applyOrDefault(P1 arg1, P2 arg2, P3 arg3, @NotNull R fallback)
  {
    return Types.notNull(apply(arg1, arg2, arg3), fallback);
  }

  /**
   * Get a partially applied function.
   * @param argument value applied for first argument
   * @return partially evaluated function
   */
  @NotNull
  default Function2<R, P2, P3> partialLeft(P1 argument)
  {
    return (arg2, arg3) -> apply(argument, arg2, arg3);
  }

  /**
   * Get a partially applied function.
   * This is the same as {@link #partialLeft(Object)}, but with a name which is more
   * consistent with functions having more arguments.
   * @param argument value applied for first argument
   * @return partially evaluated function
   */
  @NotNull
  default Function2<R, P2, P3> partial1(P1 argument)
  {
    return (arg2, arg3) -> apply(argument, arg2, arg3);
  }

  /**
   * Get a partially applied function.
   * @param argument value applied for second argument
   * @return partially evaluated function
   */
  @NotNull
  default Function2<R, P1, P3> partialMid(P2 argument)
  {
    return (arg1, arg3) -> apply(arg1, argument, arg3);
  }

  /**
   * Get a partially applied function.
   * This is the same as {@link #partialMid(Object)}, but with a name which is more
   * consistent with functions having more arguments.
   * @param argument value applied for second argument
   * @return partially evaluated function
   */
  @NotNull
  default Function2<R, P1, P3> partial2(P2 argument)
  {
    return (arg1, arg3) -> apply(arg1, argument, arg3);
  }

  /**
   * Get a partially applied function.
   * @param argument value applied for third argument
   * @return partially evaluated function
   */
  @NotNull
  default Function2<R, P1, P2> partialRight(P3 argument)
  {
    return (arg1, arg2) -> apply(arg1, arg2, argument);
  }

  /**
   * Get a partially applied function.
   * This is the same as {@link #partialRight(Object)}, but with a name which is more
   * consistent with functions having more arguments.
   * @param argument value applied for third argument
   * @return partially evaluated function
   */
  @NotNull
  default Function2<R, P1, P2> partial3(P3 argument)
  {
    return (arg1, arg2) -> apply(arg1, arg2, argument);
  }

  /**
   * Get a chained function which first applies this function
   * and then the given function to the returned value.
   * @param after function to apply after this one
   * @param <T> result type of returned function
   * @return function chain
   */
  @NotNull
  default <T> Function3<T, P1, P2, P3> andThen(@NotNull Function<? super R, ? extends T> after)
  {
    return (v1, v2, v3) -> after.apply(apply(v1, v2, v3));
  }

  /**
   * Get a chained function which first applies the given function, then this one.
   * The given function has to return a {@link ITuple3} with matching parametric types
   * which are applied in their order to this function.
   * @param before function to apply first, has to return a non-{@code null} result so unpacking the parameters
   *               from the tuple is possible
   * @param <FR> function {@code before}'s return type, has to be a 3-tuple with matching types
   * @param <FP> function {@code before}'s parameter type, same for returned function
   * @return function chain
   */
  @NotNull
  default <FR extends ITuple3<? extends P1, ? extends P2, ? extends P3>, FP>
  Function1<R, FP> after1(@NotNull Function<FP, FR> before)
  {
    return fp -> before.apply(fp).invoke(this);
  }

  /**
   * Get a chained function which first applies the given 2-parameter function, then this one.
   * The given function has to return a {@link ITuple3} with matching parametric types
   * which are applied in their order to this function.
   * @param before function to apply first, has to return a non-{@code null} result so unpacking the parameters
   *               from the tuple is possible
   * @param <FR> function {@code before}'s return type, has to be a 3-tuple with matching types
   * @param <FP1> function {@code before}'s first parameter type, sane for returned function
   * @param <FP2> function {@code before}'s second parameter type, same for returned function
   * @return function chain
   */
  @NotNull
  default <FR extends ITuple3<? extends P1, ? extends P2, ? extends P3>, FP1, FP2>
  Function2<R, FP1, FP2> after2(@NotNull BiFunction<FP1, FP2, FR> before)
  {
    return (fp1, fp2) -> before.apply(fp1, fp2).invoke(this);
  }

  /**
   * Get a chained function which first applies the given 3-parameter function, then this one.
   * The given function has to return a {@link ITuple3} with matching parametric types
   * which are applied in their order to this function.
   * @param before function to apply first, has to return a non-{@code null} result so unpacking the parameters
   *               from the tuple is possible
   * @param <FR> function {@code before}'s return type, has to be a 3-tuple with matching types
   * @param <FP1> function {@code before}'s first parameter type, sane for returned function
   * @param <FP2> function {@code before}'s second parameter type, sane for returned function
   * @param <FP3> function {@code before}'s third parameter type, sane for returned function
   * @return function chain
   */
  @NotNull
  default <FR extends ITuple3<? extends P1, ? extends P2, ? extends P3>, FP1, FP2, FP3>
  Function3<R, FP1, FP2, FP3> after3(@NotNull Function3<FR, FP1, FP2, FP3> before)
  {
    return (fp1, fp2, fp3) -> before.apply(fp1, fp2, fp3).invoke(this);
  }

  /**
   * Get a chained function which first applies the given 4-parameter function, then this one.
   * The given function has to return a {@link ITuple3} with matching parametric types
   * which are applied in their order to this function.
   * @param before function to apply first, has to return a non-{@code null} result so unpacking the parameters
   *               from the tuple is possible
   * @param <FR> function {@code before}'s return type, has to be a 3-tuple with matching types
   * @param <FP1> function {@code before}'s first parameter type, sane for returned function
   * @param <FP2> function {@code before}'s second parameter type, sane for returned function
   * @param <FP3> function {@code before}'s third parameter type, sane for returned function
   * @param <FP4> function {@code before}'s fourth parameter type, sane for returned function
   * @return function chain
   */
  @NotNull
  default <FR extends ITuple3<? extends P1, ? extends P2, ? extends P3>, FP1, FP2, FP3, FP4>
  Function4<R, FP1, FP2, FP3, FP4> after4(@NotNull Function4<FR, FP1, FP2, FP3, FP4> before)
  {
    return (fp1, fp2, fp3, fp4) -> before.apply(fp1, fp2, fp3, fp4).invoke(this);
  }

  /**
   * Get a chained function which first applies the given 5-parameter function, then this one.
   * The given function has to return a {@link ITuple3} with matching parametric types
   * which are applied in their order to this function.
   * @param before function to apply first, has to return a non-{@code null} result so unpacking the parameters
   *               from the tuple is possible
   * @param <FR> function {@code before}'s return type, has to be a 3-tuple with matching types
   * @param <FP1> function {@code before}'s first parameter type, sane for returned function
   * @param <FP2> function {@code before}'s second parameter type, sane for returned function
   * @param <FP3> function {@code before}'s third parameter type, sane for returned function
   * @param <FP4> function {@code before}'s fourth parameter type, sane for returned function
   * @param <FP5> function {@code before}'s fifth parameter type, sane for returned function
   * @return function chain
   */
  @NotNull
  default <FR extends ITuple3<? extends P1, ? extends P2, ? extends P3>, FP1, FP2, FP3, FP4, FP5>
  Function5<R, FP1, FP2, FP3, FP4, FP5> after5(@NotNull Function5<FR, FP1, FP2, FP3, FP4, FP5> before)
  {
    return (fp1, fp2, fp3, fp4, fp5) -> before.apply(fp1, fp2, fp3, fp4, fp5).invoke(this);
  }

  /**
   * Get a chained function which first applies the given 6-parameter function, then this one.
   * The given function has to return a {@link ITuple3} with matching parametric types
   * which are applied in their order to this function.
   * @param before function to apply first, has to return a non-{@code null} result so unpacking the parameters
   *               from the tuple is possible
   * @param <FR> function {@code before}'s return type, has to be a 3-tuple with matching types
   * @param <FP1> function {@code before}'s first parameter type, sane for returned function
   * @param <FP2> function {@code before}'s second parameter type, sane for returned function
   * @param <FP3> function {@code before}'s third parameter type, sane for returned function
   * @param <FP4> function {@code before}'s fourth parameter type, sane for returned function
   * @param <FP5> function {@code before}'s fifth parameter type, sane for returned function
   * @param <FP6> function {@code before}'s sixth parameter type, sane for returned function
   * @return function chain
   */
  @NotNull
  default <FR extends ITuple3<? extends P1, ? extends P2, ? extends P3>, FP1, FP2, FP3, FP4, FP5, FP6>
  Function6<R, FP1, FP2, FP3, FP4, FP5, FP6> after6(@NotNull Function6<FR, FP1, FP2, FP3, FP4, FP5, FP6> before)
  {
    return (fp1, fp2, fp3, fp4, fp5, fp6) -> before.apply(fp1, fp2, fp3, fp4, fp5, fp6).invoke(this);
  }

  /**
   * Get a chained function which first applies the given 7-parameter function, then this one.
   * The given function has to return a {@link ITuple3} with matching parametric types
   * which are applied in their order to this function.
   * @param before function to apply first, has to return a non-{@code null} result so unpacking the parameters
   *               from the tuple is possible
   * @param <FR> function {@code before}'s return type, has to be a 3-tuple with matching types
   * @param <FP1> function {@code before}'s first parameter type, sane for returned function
   * @param <FP2> function {@code before}'s second parameter type, sane for returned function
   * @param <FP3> function {@code before}'s third parameter type, sane for returned function
   * @param <FP4> function {@code before}'s fourth parameter type, sane for returned function
   * @param <FP5> function {@code before}'s fifth parameter type, sane for returned function
   * @param <FP6> function {@code before}'s sixth parameter type, sane for returned function
   * @param <FP7> function {@code before}'s seventh parameter type, sane for returned function
   * @return function chain
   */
  @NotNull
  default <FR extends ITuple3<? extends P1, ? extends P2, ? extends P3>, FP1, FP2, FP3, FP4, FP5, FP6, FP7>
  Function7<R, FP1, FP2, FP3, FP4, FP5, FP6, FP7> after7(@NotNull Function7<FR, FP1, FP2, FP3, FP4, FP5, FP6, FP7> before)
  {
    return (fp1, fp2, fp3, fp4, fp5, fp6, fp7) -> before.apply(fp1, fp2, fp3, fp4, fp5, fp6, fp7).invoke(this);
  }

  /**
   * Get a chained function which first applies the given 8-parameter function, then this one.
   * The given function has to return a {@link ITuple3} with matching parametric types
   * which are applied in their order to this function.
   * @param before function to apply first, has to return a non-{@code null} result so unpacking the parameters
   *               from the tuple is possible
   * @param <FR> function {@code before}'s return type, has to be a 3-tuple with matching types
   * @param <FP1> function {@code before}'s first parameter type, sane for returned function
   * @param <FP2> function {@code before}'s second parameter type, sane for returned function
   * @param <FP3> function {@code before}'s third parameter type, sane for returned function
   * @param <FP4> function {@code before}'s fourth parameter type, sane for returned function
   * @param <FP5> function {@code before}'s fifth parameter type, sane for returned function
   * @param <FP6> function {@code before}'s sixth parameter type, sane for returned function
   * @param <FP7> function {@code before}'s seventh parameter type, sane for returned function
   * @param <FP8> function {@code before}'s seventh parameter type, sane for returned function
   * @return function chain
   */
  @NotNull
  default <FR extends ITuple3<? extends P1, ? extends P2, ? extends P3>, FP1, FP2, FP3, FP4, FP5, FP6, FP7, FP8>
  Function8<R, FP1, FP2, FP3, FP4, FP5, FP6, FP7, FP8> after8(@NotNull Function8<FR, FP1, FP2, FP3, FP4, FP5, FP6, FP7, FP8> before)
  {
    return (fp1, fp2, fp3, fp4, fp5, fp6, fp7, fp8) -> before.apply(fp1, fp2, fp3, fp4, fp5, fp6, fp7, fp8).invoke(this);
  }

  /**
   * Get a chained function which first applies the given 9-parameter function, then this one.
   * The given function has to return a {@link ITuple3} with matching parametric types
   * which are applied in their order to this function.
   * @param before function to apply first, has to return a non-{@code null} result so unpacking the parameters
   *               from the tuple is possible
   * @param <FR> function {@code before}'s return type, has to be a 3-tuple with matching types
   * @param <FP1> function {@code before}'s first parameter type, sane for returned function
   * @param <FP2> function {@code before}'s second parameter type, sane for returned function
   * @param <FP3> function {@code before}'s third parameter type, sane for returned function
   * @param <FP4> function {@code before}'s fourth parameter type, sane for returned function
   * @param <FP5> function {@code before}'s fifth parameter type, sane for returned function
   * @param <FP6> function {@code before}'s sixth parameter type, sane for returned function
   * @param <FP7> function {@code before}'s seventh parameter type, sane for returned function
   * @param <FP8> function {@code before}'s seventh parameter type, sane for returned function
   * @param <FP9> function {@code before}'s ninth parameter type, sane for returned function
   * @return function chain
   */
  @NotNull
  default <FR extends ITuple3<? extends P1, ? extends P2, ? extends P3>, FP1, FP2, FP3, FP4, FP5, FP6, FP7, FP8, FP9>
  Function9<R, FP1, FP2, FP3, FP4, FP5, FP6, FP7, FP8, FP9> after9(@NotNull Function9<FR, FP1, FP2, FP3, FP4, FP5, FP6, FP7, FP8, FP9> before)
  {
    return (fp1, fp2, fp3, fp4, fp5, fp6, fp7, fp8, fp9) -> before.apply(fp1, fp2, fp3, fp4, fp5, fp6, fp7, fp8, fp9).invoke(this);
  }

  /**
   * Convert a 3-argument function which might throw a checked exception into
   * one which does throw an unchecked exception.
   * The returned function will instead throw an unchecked {@link WrappedFragileException} for any
   * checked exception thrown during {@link #apply(Object, Object, Object)}.
   * @param fragileFunction function throwing a checked exception
   * @param <E> exception type
   * @param <RR> result type of incoming and outgoing function.
   * @param <PP1> type of first parameter of returned functions
   * @param <PP2> type of second parameter of returned functions
   * @param <PP3> type of third parameter of returned functions
   * @return non-fragile function
   */
  @NotNull
  static <E extends Exception, RR, PP1, PP2, PP3>
  Function3<RR, PP1, PP2, PP3>
  nonFragile(@NotNull FragileFunction3<? extends RR, E, ? super PP1, ? super PP2, ? super PP3> fragileFunction)
  {
    return (p1, p2, p3) -> {
      try {
        return fragileFunction.apply(p1, p2, p3);
      } catch (Exception x) {
        throw new WrappedFragileException(x);
      }
    };
  }

  /**
   * Convert a 3-argument function which might throw a checked exception into
   * one which will return a default value instead of throwing an exception.
   * Please note that the exception is lost in this case.
   * @param fragileFunction function throwing a checked exception
   * @param fallback fallback returned when {@code fragileFunction} has thrown an exception
   * @param <E> exception type
   * @param <RR> result type of incoming and outgoing function.
   * @param <PP1> type of first parameter of returned functions
   * @param <PP2> type of second parameter of returned functions
   * @param <PP3> type of third parameter of returned functions
   * @return non-fragile function
   */
  @NotNull
  static <E extends Exception, RR, PP1, PP2, PP3>
  Function3<RR, PP1, PP2, PP3>
  nonFragile(@NotNull FragileFunction3<? extends RR, E, ? super PP1, ? super PP2, ? super PP3> fragileFunction,
             RR fallback)
  {
    return (p1, p2, p3) -> {
      try {
        return fragileFunction.apply(p1, p2, p3);
      } catch (Exception x) {
        return fallback;
      }
    };
  }

  /**
   * Convert a 3-argument function which might throw a checked exception into
   * one for which you can decide what happens with the exception.
   * The exception handler allows
   * <ul>
   *   <li>throw an unchecked exception instead (compare {@link #nonFragile(FragileFunction1)})</li>
   *   <li>provide a fallback value (compare {@link #nonFragile(FragileFunction1, Object)})</li>
   *   <li>include logging in one of the above</li>
   *   <li>and more...</li>
   * </ul>
   * @param fragileFunction function throwing a checked exception
   * @param exceptionHandler exception handler. Its return value will be used as return value of the returned function.
   * @param <E> exception type
   * @param <RR> result type of incoming and outgoing function.
   * @param <PP1> type of first parameter of returned functions
   * @param <PP2> type of second parameter of returned functions
   * @param <PP3> type of third parameter of returned functions
   * @return non-fragile function
   */
  @NotNull
  @SuppressWarnings("unchecked") // compiler should take care that only the correct checked exception is used
  static <E extends Exception, RR, PP1, PP2, PP3>
  Function3<RR, PP1, PP2, PP3>
  nonFragileX(@NotNull FragileFunction3<? extends RR, E, ? super PP1, ? super PP2, ? super PP3> fragileFunction,
              @NotNull Function4<? extends RR, ? super E, ? super PP1, ? super PP2, ? super PP3> exceptionHandler)
  {
    return (p1, p2, p3) -> {
      try {
        return fragileFunction.apply(p1, p2, p3);
      } catch (Exception x) {
        return exceptionHandler.apply((E)x, p1, p2, p3);
      }
    };
  }
}
