// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.function;

import de.caff.annotation.NotNull;
import de.caff.generics.Types;
import de.caff.generics.WrappedFragileException;
import de.caff.generics.tuple.ITuple2;

import java.util.function.BiFunction;
import java.util.function.Function;

/**
 * Function with two parameters.
 * <p>
 * This is basically the same as {@link java.util.function.BiFunction},
 * but more extensible. As it extends {@code BiFunction} it can be used
 * anywhere as a replacement.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @param <R> return type
 * @param <P1> parameter type 1
 * @param <P2> parameter type 2
 * @see Function0
 * @see Function1
 * @see Function3
 * @see Function4
 * @see Function5
 * @see Function6
 * @see Function7
 * @see Function8
 * @see Function9
 */
@FunctionalInterface
public interface Function2<R, P1, P2>
        extends BiFunction<P1, P2, R>,
                Function1<R, ITuple2<? extends P1, ? extends P2>>
{
  /**
   * Expand a tuple into the arguments of this function.
   * @param tuple2 tuple to expand, not {@code null}. Use {@link Types#notNull(Object, Object)} or
   *               {@link java.util.Objects#requireNonNull(Object)} if necessary,
   *               or have a look at {@link Types#notNull(Object, Object)}
   * @return the result of calling this function with the arguments set from the tuple
   */
  @Override
  default R apply(@NotNull ITuple2<? extends P1, ? extends P2> tuple2)
  {
    return tuple2.invoke(this);
  }

  /**
   * Apply this function, but take care to create a fallback.
   * Thiw will return the fallback when {@link #apply(Object, Object)} returns {@code null}.
   * @param arg1 first argument of this function
   * @param arg2 second argument of this function
   * @param fallback fallback value returned if function returns {@code null}
   * @return the non-null result of calling {@link #apply(Object, Object)}, or {@code fallback}
   *         if the result was {@code null}
   */
  @NotNull
  default R applyOrDefault(P1 arg1, P2 arg2, @NotNull R fallback)
  {
    return Types.notNull(apply(arg1, arg2), fallback);
  }

  /**
   * Get a partially applied function.
   * @param argument value applied for first argument
   * @return partially evaluated function
   */
  @NotNull
  default Function1<R, P2> partialLeft(P1 argument)
  {
    return arg -> apply(argument, arg);
  }

  /**
   * Get a partially applied function where the first argument is fix.
   * This is the same as {@link #partialLeft(Object)} but named more consistently.
   * @param argument value applied for first argument
   * @return partially evaluated function
   */
  @NotNull
  default Function1<R, P2> partial1(P1 argument)
  {
    return arg -> apply(argument, arg);
  }

  /**
   * Get a partially applied function.
   * @param argument value applied for second argument
   * @return partially evaluated function
   */
  @NotNull
  default Function1<R, P1> partialRight(P2 argument)
  {
    return arg -> apply(arg, argument);
  }

  /**
   * Get a partially applied function where the second argument is fix.
   * @param argument value applied for second argument
   * @return partially evaluated function
   */
  @NotNull
  default Function1<R, P1> partial2(P2 argument)
  {
    return arg -> apply(arg, argument);
  }

  /**
   * Get a chained function which first applies this function
   * and then the given function to the returned value.
   * @param after function to apply after this one
   * @param <T> result type of returned function
   * @return function chain
   */
  @NotNull
  default <T> Function2<T,P1,P2> andThen(@NotNull Function<? super R, ? extends T> after)
  {
    return (v1, v2) -> after.apply(apply(v1, v2));
  }

  /**
   * Get a chained function which first applies the given function, then this one.
   * The given function has to return a {@link ITuple2} with matching parametric types
   * which are applied in their order to this function.
   * @param before function to apply first, has to return a non-{@code null} result so unpacking the parameters
   *               from the tuple is possible
   * @param <FR> function {@code before}'s return type, has to be a 2-tuple with matching types
   * @param <FP> function {@code before}'s parameter type, same for returned function
   * @return function chain
   */
  @NotNull
  default <FR extends ITuple2<? extends P1, ? extends P2>, FP>
  Function1<R, FP> after1(@NotNull Function<FP, FR> before)
  {
    return fp -> before.apply(fp).invoke(this);
  }

  /**
   * Get a chained function which first applies the given 2-parameter function, then this one.
   * The given function has to return a {@link ITuple2} with matching parametric types
   * which are applied in their order to this function.
   * @param before function to apply first, has to return a non-{@code null} result so unpacking the parameters
   *               from the tuple is possible
   * @param <FR> function {@code before}'s return type, has to be a 2-tuple with matching types
   * @param <FP1> function {@code before}'s first parameter type, sane for returned function
   * @param <FP2> function {@code before}'s second parameter type, same for returned function
   * @return function chain
   */
  @NotNull
  default <FR extends ITuple2<? extends P1, ? extends P2>, FP1, FP2>
  Function2<R, FP1, FP2> after2(@NotNull BiFunction<FP1, FP2, FR> before)
  {
    return (fp1, fp2) -> before.apply(fp1, fp2).invoke(this);
  }

  /**
   * Get a chained function which first applies the given 3-parameter function, then this one.
   * The given function has to return a {@link ITuple2} with matching parametric types
   * which are applied in their order to this function.
   * @param before function to apply first, has to return a non-{@code null} result so unpacking the parameters
   *               from the tuple is possible
   * @param <FR> function {@code before}'s return type, has to be a 2-tuple with matching types
   * @param <FP1> function {@code before}'s first parameter type, sane for returned function
   * @param <FP2> function {@code before}'s second parameter type, sane for returned function
   * @param <FP3> function {@code before}'s third parameter type, sane for returned function
   * @return function chain
   */
  @NotNull
  default <FR extends ITuple2<? extends P1, ? extends P2>, FP1, FP2, FP3>
  Function3<R, FP1, FP2, FP3> after3(@NotNull Function3<FR, FP1, FP2, FP3> before)
  {
    return (fp1, fp2, fp3) -> before.apply(fp1, fp2, fp3).invoke(this);
  }

  /**
   * Get a chained function which first applies the given 5-parameter function, then this one.
   * The given function has to return a {@link ITuple2} with matching parametric types
   * which are applied in their order to this function.
   * @param before function to apply first, has to return a non-{@code null} result so unpacking the parameters
   *               from the tuple is possible
   * @param <FR> function {@code before}'s return type, has to be a 2-tuple with matching types
   * @param <FP1> function {@code before}'s first parameter type, sane for returned function
   * @param <FP2> function {@code before}'s second parameter type, sane for returned function
   * @param <FP3> function {@code before}'s third parameter type, sane for returned function
   * @param <FP4> function {@code before}'s fourth parameter type, sane for returned function
   * @param <FP5> function {@code before}'s fifth parameter type, sane for returned function
   * @return function chain
   */
  @NotNull
  default <FR extends ITuple2<? extends P1, ? extends P2>, FP1, FP2, FP3, FP4, FP5>
  Function5<R, FP1, FP2, FP3, FP4, FP5> after5(@NotNull Function5<FR, FP1, FP2, FP3, FP4, FP5> before)
  {
    return (fp1, fp2, fp3, fp4, fp5) -> before.apply(fp1, fp2, fp3, fp4, fp5).invoke(this);
  }

  /**
   * Get a chained function which first applies the given 6-parameter function, then this one.
   * The given function has to return a {@link ITuple2} with matching parametric types
   * which are applied in their order to this function.
   * @param before function to apply first, has to return a non-{@code null} result so unpacking the parameters
   *               from the tuple is possible
   * @param <FR> function {@code before}'s return type, has to be a 2-tuple with matching types
   * @param <FP1> function {@code before}'s first parameter type, sane for returned function
   * @param <FP2> function {@code before}'s second parameter type, sane for returned function
   * @param <FP3> function {@code before}'s third parameter type, sane for returned function
   * @param <FP4> function {@code before}'s fourth parameter type, sane for returned function
   * @param <FP5> function {@code before}'s fifth parameter type, sane for returned function
   * @param <FP6> function {@code before}'s sixth parameter type, sane for returned function
   * @return function chain
   */
  @NotNull
  default <FR extends ITuple2<? extends P1, ? extends P2>, FP1, FP2, FP3, FP4, FP5, FP6>
  Function6<R, FP1, FP2, FP3, FP4, FP5, FP6> after6(@NotNull Function6<FR, FP1, FP2, FP3, FP4, FP5, FP6> before)
  {
    return (fp1, fp2, fp3, fp4, fp5, fp6) -> before.apply(fp1, fp2, fp3, fp4, fp5, fp6).invoke(this);
  }

  /**
   * Get a chained function which first applies the given 7-parameter function, then this one.
   * The given function has to return a {@link ITuple2} with matching parametric types
   * which are applied in their order to this function.
   * @param before function to apply first, has to return a non-{@code null} result so unpacking the parameters
   *               from the tuple is possible
   * @param <FR> function {@code before}'s return type, has to be a 2-tuple with matching types
   * @param <FP1> function {@code before}'s first parameter type, sane for returned function
   * @param <FP2> function {@code before}'s second parameter type, sane for returned function
   * @param <FP3> function {@code before}'s third parameter type, sane for returned function
   * @param <FP4> function {@code before}'s fourth parameter type, sane for returned function
   * @param <FP5> function {@code before}'s fifth parameter type, sane for returned function
   * @param <FP6> function {@code before}'s sixth parameter type, sane for returned function
   * @param <FP7> function {@code before}'s seventh parameter type, sane for returned function
   * @return function chain
   */
  @NotNull
  default <FR extends ITuple2<? extends P1, ? extends P2>, FP1, FP2, FP3, FP4, FP5, FP6, FP7>
  Function7<R, FP1, FP2, FP3, FP4, FP5, FP6, FP7> after7(@NotNull Function7<FR, FP1, FP2, FP3, FP4, FP5, FP6, FP7> before)
  {
    return (fp1, fp2, fp3, fp4, fp5, fp6, fp7) -> before.apply(fp1, fp2, fp3, fp4, fp5, fp6, fp7).invoke(this);
  }

  /**
   * Get a chained function which first applies the given 8-parameter function, then this one.
   * The given function has to return a {@link ITuple2} with matching parametric types
   * which are applied in their order to this function.
   * @param before function to apply first, has to return a non-{@code null} result so unpacking the parameters
   *               from the tuple is possible
   * @param <FR> function {@code before}'s return type, has to be a 2-tuple with matching types
   * @param <FP1> function {@code before}'s first parameter type, sane for returned function
   * @param <FP2> function {@code before}'s second parameter type, sane for returned function
   * @param <FP3> function {@code before}'s third parameter type, sane for returned function
   * @param <FP4> function {@code before}'s fourth parameter type, sane for returned function
   * @param <FP5> function {@code before}'s fifth parameter type, sane for returned function
   * @param <FP6> function {@code before}'s sixth parameter type, sane for returned function
   * @param <FP7> function {@code before}'s seventh parameter type, sane for returned function
   * @param <FP8> function {@code before}'s seventh parameter type, sane for returned function
   * @return function chain
   */
  @NotNull
  default <FR extends ITuple2<? extends P1, ? extends P2>, FP1, FP2, FP3, FP4, FP5, FP6, FP7, FP8>
  Function8<R, FP1, FP2, FP3, FP4, FP5, FP6, FP7, FP8> after8(@NotNull Function8<FR, FP1, FP2, FP3, FP4, FP5, FP6, FP7, FP8> before)
  {
    return (fp1, fp2, fp3, fp4, fp5, fp6, fp7, fp8) -> before.apply(fp1, fp2, fp3, fp4, fp5, fp6, fp7, fp8).invoke(this);
  }

  /**
   * Get a chained function which first applies the given 9-parameter function, then this one.
   * The given function has to return a {@link ITuple2} with matching parametric types
   * which are applied in their order to this function.
   * @param before function to apply first, has to return a non-{@code null} result so unpacking the parameters
   *               from the tuple is possible
   * @param <FR> function {@code before}'s return type, has to be a 2-tuple with matching types
   * @param <FP1> function {@code before}'s first parameter type, sane for returned function
   * @param <FP2> function {@code before}'s second parameter type, sane for returned function
   * @param <FP3> function {@code before}'s third parameter type, sane for returned function
   * @param <FP4> function {@code before}'s fourth parameter type, sane for returned function
   * @param <FP5> function {@code before}'s fifth parameter type, sane for returned function
   * @param <FP6> function {@code before}'s sixth parameter type, sane for returned function
   * @param <FP7> function {@code before}'s seventh parameter type, sane for returned function
   * @param <FP8> function {@code before}'s seventh parameter type, sane for returned function
   * @param <FP9> function {@code before}'s ninth parameter type, sane for returned function
   * @return function chain
   */
  @NotNull
  default <FR extends ITuple2<? extends P1, ? extends P2>, FP1, FP2, FP3, FP4, FP5, FP6, FP7, FP8, FP9>
  Function9<R, FP1, FP2, FP3, FP4, FP5, FP6, FP7, FP8, FP9> after9(@NotNull Function9<FR, FP1, FP2, FP3, FP4, FP5, FP6, FP7, FP8, FP9> before)
  {
    return (fp1, fp2, fp3, fp4, fp5, fp6, fp7, fp8, fp9) -> before.apply(fp1, fp2, fp3, fp4, fp5, fp6, fp7, fp8, fp9).invoke(this);
  }

  /**
   * Make a standard {@link BiFunction} usable as a {@code Function2}.
   * This is eg useful if you want to make use of partial evaluation.
   * @param bifunc bifunction
   * @param <RR>   result type
   * @param <PP1>  first parameter type
   * @param <PP2>  second parameter type
   * @return {@code Function2} view of the given bifunction
   */
  @NotNull
  static <RR,PP1,PP2> Function2<RR, PP1, PP2> from(@NotNull BiFunction<PP1, PP2, RR> bifunc)
  {
    return bifunc instanceof Function2
            ? (Function2<RR, PP1, PP2>)bifunc
            : bifunc::apply;
  }

  /**
   * Convert a 2-argument function which might throw a checked exception into
   * one which does throw an unchecked exception.
   * The returned function will instead throw an unchecked {@link WrappedFragileException} for any
   * checked exception thrown during {@link #apply(Object, Object)}.
   * @param fragileFunction function throwing a checked exception
   * @param <E> exception type
   * @param <RR> result type of incoming and outgoing function.
   * @param <PP1> type of first parameter of returned functions
   * @param <PP2> type of second parameter of returned functions
   * @return non-fragile function
   */
  @NotNull
  static <E extends Exception, RR, PP1, PP2>
  Function2<RR, PP1, PP2>
  nonFragile(@NotNull FragileFunction2<? extends RR, E, ? super PP1, ? super PP2> fragileFunction)
  {
    return (p1, p2) -> {
      try {
        return fragileFunction.apply(p1, p2);
      } catch (Exception x) {
        throw new WrappedFragileException(x);
      }
    };
  }

  /**
   * Convert a 2-argument function which might throw a checked exception into
   * one which will return a default value instead of throwing an exception.
   * Please note that the exception is lost in this case.
   * @param fragileFunction function throwing a checked exception
   * @param fallback fallback returned when {@code fragileFunction} has thrown an exception
   * @param <E> exception type
   * @param <RR> result type of incoming and outgoing function.
   * @param <PP1> type of first parameter of returned functions
   * @param <PP2> type of second parameter of returned functions
   * @return non-fragile function
   */
  @NotNull
  static <E extends Exception, RR, PP1, PP2>
  Function2<RR, PP1, PP2>
  nonFragile(@NotNull FragileFunction2<? extends RR, E, ? super PP1, ? super PP2> fragileFunction,
             RR fallback)
  {
    return (p1, p2) -> {
      try {
        return fragileFunction.apply(p1, p2);
      } catch (Exception x) {
        return fallback;
      }
    };
  }

  /**
   * Convert a 2-argument function which might throw a checked exception into
   * one for which you can decide what happens with the exception.
   * The exception handler allows
   * <ul>
   *   <li>throw an unchecked exception instead (compare {@link #nonFragile(FragileFunction1)})</li>
   *   <li>provide a fallback value (compare {@link #nonFragile(FragileFunction1, Object)})</li>
   *   <li>include logging in one of the above</li>
   *   <li>and more...</li>
   * </ul>
   * @param fragileFunction function throwing a checked exception
   * @param exceptionHandler exception handler. Its return value will be used as return value of the returned function.
   * @param <E> exception type
   * @param <RR> result type of incoming and outgoing function.
   * @param <PP1> type of first parameter of returned functions
   * @param <PP2> type of second parameter of returned functions
   * @return non-fragile function
   */
  @NotNull
  @SuppressWarnings("unchecked") // compiler should take care that only the correct checked exception is used
  static <E extends Exception, RR, PP1, PP2>
  Function2<RR, PP1, PP2>
  nonFragileX(@NotNull FragileFunction2<? extends RR, E, ? super PP1, ? super PP2> fragileFunction,
              @NotNull Function3<? extends RR, ? super E, ? super PP1, ? super PP2> exceptionHandler)
  {
    return (p1, p2) -> {
      try {
        return fragileFunction.apply(p1, p2);
      } catch (Exception x) {
        return exceptionHandler.apply((E)x, p1, p2);
      }
    };
  }
}
