// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.function;

import de.caff.annotation.NotNull;
import de.caff.generics.Types;
import de.caff.generics.WrappedFragileException;

import java.util.function.BiFunction;
import java.util.function.Function;
import java.util.function.Predicate;

/**
 * Function with one parameter.
 * <p>
 * This is basically the same as {@link java.util.function.Function},
 * but is more extensible. As it is extending {@code Function} it
 * can be used anywhere as a replacement.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @param <R> return type
 * @param <P> parameter type
 * @see Function0
 * @see Function2
 * @see Function3
 * @see Function4
 * @see Function5
 * @see Function6
 * @see Function7
 * @see Function8
 * @see Function9
 */
@FunctionalInterface
public interface Function1<R, P>
        extends Function<P, R>
{
  /**
   * Apply this function, but take care to create a fallback.
   * This will return the fallback when {@link #apply(Object)} returns {@code null}.
   * @param argument argument of this function
   * @param fallback fallback value returned if function returns {@code null}
   * @return the non-null result of calling {@link #apply(Object)}, or {@code fallback}
   *         if the result was {@code null}
   * @see #applyNonNull(Object, Object)
   */
  @NotNull
  default R applyOrDefault(P argument, @NotNull R fallback)
  {
    return Types.notNull(apply(argument), fallback);
  }

  /**
   * Apply this function if the argument is not {@code null}, otherwise return the fallback.
   * The difference to {@link #applyOrDefault(Object, Object)} is that this function is not
   * applied when {@code argument} is {@code null}.
   *<p>
   * Using this invocation it is guaranteed that {@link #apply(Object)} will always be called with
   * a non-null argument.
   * </p>
   * @param argument argument of this function
   * @param fallback fallback returned if {@code argument} is {@code null}
   * @return the result of applying this function to the non-null {@code argument},
   *         or {@code fallback} if {@code argument} is null
   */
  default R applyNonNull(P argument, R fallback)
  {
    return argument == null
            ? fallback
            : apply(argument);
  }

  /**
   * Get a function which calls a fallback function if this function
   * returns {@code null}.
   * @param fallback fallback function called if this function returns {@code null} for a given parameter
   * @return combined function providing a fallback
   * @see #fallback(Function1, Function1)
   * @see #applyOrDefault(Object, Object)
   */
  @NotNull
  default Function1<R, P> withFallback(@NotNull Function1<? extends R, ? super P> fallback)
  {
    return fallback(this, fallback);
  }

  /**
   * Create a zero argument function from this one by applying a constant argument.
   * Although useless for pure functions it can be useful in cases where the zero
   * argument function is a factory (and therefore not pure).
   * @param argument constant argument
   * @return zero argument function
   */
  @NotNull
  default Function0<R> partial(P argument)
  {
    return () -> apply(argument);
  }

  /**
   * Get a chained function which first applies the
   * given function and then this one.
   * @param before function to apply first
   * @param <T> parameter type of returned function
   * @return function chain
   */
  @NotNull
  default <T> Function1<R,T> after(@NotNull Function<? super T, ? extends P> before)
  {
    return v -> apply(before.apply(v));
  }

  /**
   * Get a chained function which first applies this function
   * and then the given function.
   * @param after function to apply after this one
   * @param <T> result type of returned function
   * @return function chain
   */
  @NotNull
  default <T> Function1<T,P> andThen(@NotNull Function<? super R, ? extends T> after)
  {
    return (P v) -> after.apply(apply(v));
  }

  /**
   * Create a normal function from a fragile function.
   * This will throw an {@link EvaluationException} (an unchecked runtime exception) with
   * the original exception as the cause.
   * @param ff    fragile function
   * @param <TT>  result type
   * @param <PP>  parameter type
   * @param <EE>  exception type
   * @return normal function
   */
  @NotNull
  static <TT, PP, EE extends Exception> Function1<TT, PP> from(@NotNull FragileFunction1<TT, EE, PP> ff)
  {
    return arg -> {
      try {
        return ff.apply(arg);
      } catch (Exception e) {
        throw new EvaluationException("Caught exception during function evaluation!",
                                      e);
      }
    };
  }

  /**
   * Make a {@link Function} usable as a {@code Function1}.
   * @param func standard function
   * @param <RR> result type
   * @param <PP> parameter type
   * @return {@code Function1} view of the given function
   */
  @NotNull
  static <RR, PP> Function1<RR, PP> from(@NotNull Function<PP, RR> func)
  {
    return func instanceof Function1
            ? (Function1<RR, PP>)func
            : func::apply;
  }

  /**
   * Get a function which is a combination of two functions
   * which calls the first function and returns its result.
   * Only if the first function returns {@code null} it will
   * call the second function and return its result.
   *
   * @param func1 first function
   * @param func2 second function
   * @param <PP> parameter type of returned function, which is the most extending
   *            type of both {@code func1}'s parameter type and {@code func2}'s
   *            parameter type, so if neither type is extending the other this method
   *            will not compile
   * @param <RR> result type of returned function, which is the most restricting type
   *            of of both {@code func1}'s result type and {@code func2}'s
   *            result type, so if neither type is extending the other this method
   *            will not compile
   * @return combination of both function where the second one is only called if the first returns {@code null}
   * @see #fallback(Predicate, Function1, Function1)
   */
  @NotNull
  static <RR, PP> Function1<RR, PP> fallback(@NotNull Function1<? extends RR, ? super PP> func1,
                                             @NotNull Function1<? extends RR, ? super PP> func2)
  {
    return p -> {
      final RR result = func1.apply(p);
      return result != null
              ? result
              : func2.apply(p);
    };
  }

  /**
   * Get a function which is a combination of two functions
   * which calls the first function and returns its result.
   * Only if the first function returns a value which is invalid it will
   * call the second function and return its result.
   *
   * @param validCheck check which determines the validity of the return of
   *                   {@code func1}, if the check returns {@code false}
   *                   {@code func2} will be used to calculate the result
   *                   of the returned function
   * @param func1 first function
   * @param func2 second function
   * @param <PP> parameter type of returned function, which is the most extending
   *            type of both {@code func1}'s parameter type and {@code func2}'s
   *            parameter type, so if neither type is extending the other this method
   *            will not compile
   * @param <RR> result type of returned function, which is the most restricting type
   *            of of both {@code func1}'s result type and {@code func2}'s
   *            result type, so if neither type is extending the other this method
   *            will not compile
   * @return combination of both functions where the second one is only called if the first returns
   *         a value which the {@code fallbackCheck} identifies as invalid
   */
  @NotNull
  static <RR, PP> Function1<RR, PP> fallback(@NotNull Predicate<? super RR> validCheck,
                                             @NotNull Function1<? extends RR, ? super PP> func1,
                                             @NotNull Function1<? extends RR, ? super PP> func2)
  {
    return p -> {
      final RR result = func1.apply(p);
      return validCheck.test(result)
              ? result
              : func2.apply(p);
    };
  }

  /**
   * Convert a 1-argument function which might throw a checked exception into
   * one which does throw an unchecked exception.
   * The returned function will instead throw an unchecked {@link WrappedFragileException} for any
   * checked exception thrown during {@link #apply(Object)}.
   * @param fragileFunction function throwing a checked exception
   * @param <E> exception type
   * @param <RR> result type of incoming and outgoing function.
   * @param <PP> type of parameter of both functions
   * @return non-fragile function
   * @throws WrappedFragileException wrapping the original exception from {@code fragileFunction}
   */
  @NotNull
  static <E extends Exception, RR, PP> Function1<RR, PP> nonFragile(@NotNull FragileFunction1<? extends RR, E, ? super PP> fragileFunction)
  {
    return p -> {
      try {
        return fragileFunction.apply(p);
      } catch (Exception x) {
        throw new WrappedFragileException(x, "Caught hidden fragile exception!");
      }
    };
  }

  /**
   * Convert a 1-argument function which might throw a checked exception into
   * one which will return a default value instead of throwing an exception.
   * Please note that the exception is lost in this case.
   * @param fragileFunction function throwing a checked exception
   * @param fallback fallback returned when {@code fragileFunction} has thrown an exception
   * @param <E> exception type
   * @param <RR> result type of incoming and outgoing function.
   * @param <PP> type of parameter of both functions
   * @return non-fragile function
   */
  @NotNull
  static <E extends Exception, RR, PP> Function1<RR, PP> nonFragile(@NotNull FragileFunction1<? extends RR, E, ? super PP> fragileFunction,
                                                                    RR fallback)
  {
    return p -> {
      try {
        return fragileFunction.apply(p);
      } catch (Exception x) {
        return fallback;
      }
    };
  }

  /**
   * Convert a 1-argument function which might throw a checked exception into
   * one for which you can decide what happens with the exception.
   * The exception handler allows
   * <ul>
   *   <li>throw an unchecked exception instead (compare {@link #nonFragile(FragileFunction1)})</li>
   *   <li>provide a fallback value (compare {@link #nonFragile(FragileFunction1, Object)})</li>
   *   <li>include logging in one of the above</li>
   *   <li>and more...</li>
   * </ul>
   * @param fragileFunction function throwing a checked exception
   * @param exceptionHandler exception handler. Its return value will be used as return value of the returned function.
   * @param <E> exception type
   * @param <RR> result type of incoming and outgoing function.
   * @param <PP> type of parameter of both functions
   * @return non-fragile function
   */
  @NotNull
  @SuppressWarnings("unchecked") // compiler should take care that only the correct checked exception is used
  static <E extends Exception, RR, PP> Function1<RR, PP> nonFragileX(@NotNull FragileFunction1<? extends RR, E, ? super PP> fragileFunction,
                                                                     @NotNull BiFunction<? super E, ? super PP, ? extends RR> exceptionHandler)
  {
    return p -> {
      try {
        return fragileFunction.apply(p);
      } catch (Exception x) {
        return exceptionHandler.apply((E)x, p);
      }
    };
  }
}
