// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.function;

import de.caff.annotation.NotNull;
import de.caff.generics.WrappedFragileException;

import java.util.function.Function;
import java.util.function.Supplier;

/**
 * Function with no parameter.
 * As a pure function this is useless, but there are cases where this is useful
 * like factories.
 * <p>
 * The standard Java equivalent is {@link java.util.function.Supplier}, but this
 * implementation is more extensible. As it is exending {@code Supplier} it can
 * be used anywhere where a supplier is used.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @param <R> return type
 * @see Function1
 * @see Function2
 * @see Function3
 * @see Function4
 * @see Function5
 * @see Function6
 * @see Function7           
 * @see Function8
 * @see Function9
 */
@FunctionalInterface
public interface Function0<R>
        extends Supplier<R>
{
  /**
   * Function which always returns a {@code null} object.
   * Use {@link #alwaysNull()} instead as it handles the return type
   * transparently.
   */
  @NotNull
  Function0<Object> ALWAYS_NULL = () -> null;

  /**
   * Run the function.
   * @return return value
   */
  default R apply()
  {
    return get();
  }

  /**
   * Get a chained function which first applies this function
   * and then the given function to the returned value.
   * @param after function to apply after this one
   * @param <T> result type of returned function
   * @return function chain
   */
  @NotNull
  default <T> Function0<T> andThen(@NotNull Function<? super R, ? extends T> after)
  {
    return () -> after.apply(get());
  }

  /**
   * Make this zero-argument function appear as a function with one ignored argument.
   * This is occasionally useful when a function with an argument is requested, but you want
   * to provide a predefined zero-arg function.
   * @param <P1> argument type of returned function, ignored
   * @return 1-argument function which behaves like this function be ignoring the argument
   */
  @NotNull
  default <P1> Function1<R, P1> ignore1()
  {
    return p -> apply();
  }

  /**
   * Get a zero-argument function which always returns {@code null}.
   * @param <T> result type of returned function
   * @return function always returning {@code null}
   */
  @NotNull
  @SuppressWarnings("unchecked") // as null is compatible with all objects
  static <T> Function0<T> alwaysNull()
  {
    return (Function0<T>)ALWAYS_NULL;
  }

  /**
   * Make a {@link Supplier} usable as a {@code Function0}.
   * This is eg useful if you want to use {@link #andThen(Function)}.
   * @param supplier supplier
   * @param <T> supplier return type
   * @return {@code Function0} view of the given supplier
   */
  @NotNull
  static <T> Function0<T> from(@NotNull Supplier<T> supplier)
  {
    return  supplier instanceof Function0
            ? (Function0<T>)supplier
            : supplier::get;
  }

  /**
   * Convert a 0-argument function which might throw a checked exception into
   * one which does throw an unchecked exception.
   * The returned function will instead throw an unchecked {@link WrappedFragileException} for any
   * checked exception thrown during {@link #apply()}.
   * @param fragileFunction function throwing a checked exception
   * @param <E> exception type
   * @param <RR> result type of incoming and outgoing function.
   * @return non-fragile function
   */
  @NotNull
  static <E extends Exception, RR> Function0<RR> nonFragile(@NotNull FragileFunction0<? extends RR, E> fragileFunction)
  {
    return () -> {
      try {
        return fragileFunction.apply();
      } catch (Exception x) {
        throw new WrappedFragileException(x);
      }
    };
  }

  /**
   * Convert a 0-argument function which might throw a checked exception into
   * one which will return a default value instead of throwing an exception.
   * Please note that the exception is lost in this case.
   * @param fragileFunction function throwing a checked exception
   * @param fallback fallback returned when {@code fragileFunction} has thrown an exception
   * @param <E> exception type
   * @param <RR> result type of incoming and outgoing function.
   * @return non-fragile function
   */
  @NotNull
  static <E extends Exception, RR> Function0<RR> nonFragile(@NotNull FragileFunction0<? extends RR, E> fragileFunction,
                                                            RR fallback)
  {
    return () -> {
      try {
        return fragileFunction.apply();
      } catch (Exception x) {
        return fallback;
      }
    };
  }

  /**
   * Convert a 0-argument function which might throw a checked exception into
   * one for which you can decide what happens with the exception.
   * The exception handler allows
   * <ul>
   *   <li>throw an unchecked exception instead (compare {@link #nonFragile(FragileFunction0)})</li>
   *   <li>provide a fallback value (compare {@link #nonFragile(FragileFunction0, Object)})</li>
   *   <li>include logging in one of the above</li>
   *   <li>and more...</li>
   * </ul>
   * @param fragileFunction function throwing a checked exception
   * @param exceptionHandler exception handler. Its return value will be used as return value of the returned function.
   * @param <E> exception type
   * @param <RR> result type of incoming and outgoing function.
   * @return non-fragile function
   */
  @NotNull
  @SuppressWarnings("unchecked") // compiler should take care that only the correct checked exception is used
  static <E extends Exception, RR> Function0<RR> nonFragileX(@NotNull FragileFunction0<? extends RR, E> fragileFunction,
                                                             @NotNull Function<? super E, ? extends RR> exceptionHandler)
  {
    return () -> {
      try {
        return fragileFunction.apply();
      } catch (Exception x) {
        return exceptionHandler.apply((E)x);
      }
    };
  }
}
