// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.function;

import de.caff.annotation.NotNull;
import de.caff.generics.WrappedFragileException;
import de.caff.generics.tuple.ITuple3;
import de.caff.generics.tuple.ITuple4;

import java.util.function.Function;

/**
 * Function without return and with four parameters.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @param <P1> first parameter type
 * @param <P2> second parameter type
 * @param <P3> third parameter type
 * @param <P4> fourth parameter type
 * @see Procedure0
 * @see Procedure1
 * @see Procedure2
 * @see Procedure4
 * @see Procedure5
 * @see Procedure6
 * @see Procedure7
 * @see Procedure8
 * @see Procedure9
 */
@FunctionalInterface
public interface FragileProcedure4<E extends Exception, P1, P2, P3, P4>
        extends FragileProcedure1<E, ITuple4<? extends P1, ? extends P2, ? extends P3, ? extends P4>>
{
  /**
   * Procedure with three arguments which does nothing.
   * Use {@link #empty()} instead.
   */
  @NotNull
  FragileProcedure4<?, ?, ?, ?, ?> NULL = (FragileProcedure4<?, ?, ?, ?, ?>)(a, b, c, d) -> {};

  /**
   * Run the procedure.
   * @param arg1 first procedure argument
   * @param arg2 second procedure argument
   * @param arg3 third procedure argument
   * @param arg4 fourth procedure argument
   * @throws E exception
   */
  void apply(P1 arg1, P2 arg2, P3 arg3, P4 arg4) throws E;

  @Override
  default void apply(@NotNull ITuple4<? extends P1, ? extends P2, ? extends P3, ? extends P4> tuple4) throws E
  {
    apply(tuple4._1(), tuple4._2(), tuple4._3(), tuple4._4());
  }

  /**
   * Convert this to a non-fragile procedure which does not throw a checked exception.
   * This will instead throw an unchecked {@link WrappedFragileException} for any
   * checked exception thrown during {@link #apply(Object, Object, Object, Object)}.
   * @return non-fragile function
   * @see Procedure4#nonFragile(FragileProcedure4)
   */
  @NotNull
  default Procedure4<P1, P2, P3, P4> nonFragile()
  {
    return Procedure4.nonFragile(this);
  }

  /**
   * Convert this to a non-fragile procedure which does not a checked exception.
   * This will instead throw an unchecked {@link WrappedFragileException} for any
   * checked exception thrown during {@link #apply(Object, Object, Object, Object)}.
   * @param exceptionHandler handler called when an exception is encountered.
   *                         It may either ignore it, throw an unchecked exception,
   *                         or collect the exceptions
   * @return non-fragile function
   * @see Procedure4#nonFragileX(FragileProcedure4, Procedure5)
   */
  @NotNull
  default Procedure4<P1, P2, P3, P4> nonFragileX(@NotNull Procedure5<? super E, ? super P1, ? super P2, ? super P3, ? super P4> exceptionHandler)
  {
    return Procedure4.nonFragileX(this, exceptionHandler);
  }

  /**
   * Get a chained procedure which first applies the given function, then this procedure.
   * The given function has to return a {@link ITuple3} with matching parametric types
   * which are applied in their order to this procedure.
   * @param before function to apply first, has to return a non-{@code null} result so unpacking the parameters
   *               from the tuple is possible
   * @param <FR> function {@code before}'s return type, has to be a 3-tuple with matching types
   * @param <FP> function {@code before}'s parameter type, same for returned procedure
   * @return chain of function and this procedure
   */
  @NotNull
  default <FR extends ITuple4<? extends P1, ? extends P2, ? extends P3, ? extends P4>, FP>
  FragileProcedure1<E, FP> after1(@NotNull Function<FP, FR> before)
  {
    return fp -> apply(before.apply(fp));
  }

  /**
   * Get a partially applied procedure.
   * @param argument value applied for first argument
   * @return partially evaluated procedure
   */
  @NotNull
  default FragileProcedure3<E, P2, P3, P4> partial1(P1 argument)
  {
    return (arg2, arg3, arg4) -> FragileProcedure4.this.apply(argument, arg2, arg3, arg4);
  }

  /**
   * Get a partially applied procedure.
   * @param argument value applied for second argument
   * @return partially evaluated procedure
   */
  @NotNull
  default FragileProcedure3<E, P1, P3, P4> partial2(P2 argument)
  {
    return (arg1, arg3, arg4) -> FragileProcedure4.this.apply(arg1, argument, arg3, arg4);
  }

  /**
   * Get a partially applied procedure.
   * @param argument value applied for third argument
   * @return partially evaluated procedure
   */
  @NotNull
  default FragileProcedure3<E, P1, P2, P4> partial3(P3 argument)
  {
    return (arg1, arg2, arg4) -> FragileProcedure4.this.apply(arg1, arg2, argument, arg4);
  }

  /**
   * Get a partially applied procedure.
   * @param argument value applied for fourth argument
   * @return partially evaluated procedure
   */
  @NotNull
  default FragileProcedure3<E, P1, P2, P3> partial4(P4 argument)
  {
    return (arg1, arg2, arg3) -> FragileProcedure4.this.apply(arg1, arg2, arg3, argument);
  }

  /**
   * Return a procedure which does nothing.
   * This just returns {@link #NULL}, but the compiler will accept it smoothly.
   * @param <X>  exception type
   * @param <T1> first argument type
   * @param <T2> second argument type
   * @param <T3> third argument type
   * @param <T4> fourth argument type
   * @return do-nothing procedure
   */
  @NotNull
  @SuppressWarnings("unchecked") // do nothing is always same
  static <X extends Exception, T1, T2, T3, T4> FragileProcedure4<X, T1, T2, T3, T4> empty()
  {
    return (FragileProcedure4<X, T1, T2, T3, T4>)NULL;
  }
}
