// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.function;

import de.caff.annotation.NotNull;
import de.caff.generics.WrappedFragileException;
import de.caff.generics.tuple.ITuple2;

import java.util.function.BiConsumer;
import java.util.function.Function;

/**
 * Function without return and with two parameters.
 * This provides the same functional interface as modern Java's {@link BiConsumer BiConsumer},
 * but in a more extensible way.
 * As it is extending {@link BiConsumer} it can be used anywhere
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @param <P1> first parameter type
 * @param <P2> second parameter type
 * @see Procedure0
 * @see Procedure1
 * @see Procedure3
 * @see Procedure4
 * @see Procedure5
 * @see Procedure6
 * @see Procedure7
 * @see Procedure8
 * @see Procedure9
 */
@FunctionalInterface
public interface FragileProcedure2<E extends Exception, P1, P2>
        extends FragileProcedure1<E, ITuple2<? extends P1, ? extends P2>>
{
  /**
   * Procedure with two arguments which does nothing.
   * Use {@link #empty()} instead.
   */
  @NotNull
  FragileProcedure2<?, ?, ?> NULL = (FragileProcedure2<Exception, Object, Object>)(a, b) -> {};

  /**
   * Expand a tuple into the arguments of this procedure.
   * @param tuple2 tuple to expand
   * @throws E on errors
   */
  @Override
  default void apply(@NotNull ITuple2<? extends P1, ? extends P2> tuple2)
          throws E
  {
    apply(tuple2._1(), tuple2._2());
  }

  /**
   * Run the function.
   * @param arg1 first function argument
   * @param arg2 second function argument
   * @throws E on errors
   */
  void apply(P1 arg1, P2 arg2) throws E;

  /**
   * Convert this to a non-fragile procedure which does not throw a checked exception.
   * This will instead throw an unchecked {@link WrappedFragileException} for any
   * checked exception thrown during {@link #apply(Object, Object)}.
   * @return non-fragile function
   * @see Procedure2#nonFragile(FragileProcedure2)
   */
  @NotNull
  default Procedure2<P1, P2> nonFragile()
  {
    return Procedure2.nonFragile(this);
  }

  /**
   * Convert this to a non-fragile procedure which does not a checked exception.
   * This will instead throw an unchecked {@link WrappedFragileException} for any
   * checked exception thrown during {@link #apply(Object, Object)}.
   * @param exceptionHandler handler called when an exception is encountered.
   *                         It may either ignore it, throw an unchecked exception,
   *                         or collect the exceptions
   * @return non-fragile function
   * @see Procedure2#nonFragileX(FragileProcedure2, Procedure3)
   */
  @NotNull
  default Procedure2<P1, P2> nonFragileX(@NotNull Procedure3<? super E, ? super P1, ? super P2> exceptionHandler)
  {
    return Procedure2.nonFragileX(this, exceptionHandler);
  }

  /**
   * Get a partially applied procedure.
   * @param argument value applied for first argument
   * @return partially evaluated procedure
   */
  @NotNull
  default FragileProcedure1<E, P2> partialLeft(P1 argument)
  {
    return arg -> apply(argument, arg);
  }

  /**
   * Get a partially applied procedure.
   * This is the same as {@link #partialLeft(Object)}, but with a name which is more
   * consistent with functions having more arguments.
   * @param argument value applied for first argument
   * @return partially evaluated procedure
   */
  @NotNull
  default FragileProcedure1<E, P2> partial1(P1 argument)
  {
    return arg -> apply(argument, arg);
  }

  /**
   * Get a partially applied procedure.
   * @param argument value applied for second argument
   * @return partially evaluated procedure
   */
  @NotNull
  default FragileProcedure1<E, P1> partialRight(P2 argument)
  {
    return arg -> apply(arg, argument);
  }

  /**
   * Get a partially applied procedure.
   * This is the same as {@link #partialRight(Object)}, but with a name which is more
   * consistent with functions having more arguments.
   * @param argument value applied for second argument
   * @return partially evaluated procedure
   */
  @NotNull
  default FragileProcedure1<E, P1> partial2(P2 argument)
  {
    return arg -> apply(arg, argument);
  }

  /**
   * Get a chained procedure which first applies the given function, then this procedure.
   * The given function has to return a {@link ITuple2} with matching parametric types
   * which are applied in their order to this procedure.
   * @param before function to apply first, has to return a non-{@code null} result so unpacking the parameters
   *               from the tuple is possible
   * @param <FR> function {@code before}'s return type, has to be a 2-tuple with matching types
   * @param <FP> function {@code before}'s parameter type, same for returned procedure
   * @return chain of function and this procedure
   */
  @NotNull
  default <FR extends ITuple2<? extends P1, ? extends P2>, FP>
  FragileProcedure1<E, FP> after1(@NotNull Function<FP, FR> before)
  {
    return fp -> apply(before.apply(fp));
  }

  /**
   * Return a procedure which does nothing.
   * This just returns {@link #NULL}, but the compiler will accept it smoothly.
   * @param <X> exception type
   * @param <T1> first argument type
   * @param <T2> second argument type
   * @return do-nothing procedure
   */
  @NotNull
  @SuppressWarnings("unchecked") // do nothing is always same
  static <X extends Exception, T1, T2> FragileProcedure2<X, T1, T2> empty()
  {
    return (FragileProcedure2<X, T1, T2>)NULL;
  }
}
