// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.function;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;
import de.caff.generics.WrappedFragileException;

import java.util.function.BiConsumer;
import java.util.function.Consumer;

/**
 * Function without return and with one parameter.
 * This provides the same functional interface as modern Java's {@link Consumer Consumer},
 * but in a more extensible way.
 * As it is extending {@code Consumer} it can be used anywhere a {@code Consumer} is expected.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @param <P> parameter type
 * @param <E> exception type
 * @see FragileProcedure0
 * @see FragileProcedure2
 * @see FragileProcedure3
 * @see Procedure1
 */
@FunctionalInterface
public interface FragileProcedure1<E extends Exception, P>
{
  /**
   * Procedure with one argument which does nothing, not even throwing.
   * Use {@link #empty()} instead.
   */
  @NotNull
  FragileProcedure1<?, ?> NULL = (FragileProcedure1<Exception, Object>)arg -> {};

  /**
   * Run the function.
   * @param arg function argument
   * @throws E exception
   */
  void apply(P arg) throws E;

  /**
   * Convert this to a non-fragile procedure which does not throw a checked exception.
   * This will instead throw an unchecked {@link WrappedFragileException} for any
   * checked exception thrown during {@link #apply(Object)}.
   * @return non-fragile function
   * @see Procedure1#nonFragile(FragileProcedure1)
   */
  @NotNull
  default Procedure1<P> nonFragile()
  {
    return Procedure1.nonFragile(this);
  }

  /**
   * Convert this to a non-fragile procedure which does not a checked exception.
   * This will instead throw an unchecked {@link WrappedFragileException} for any
   * checked exception thrown during {@link #apply(Object)}.
   * @param exceptionHandler handler called when an exception is encountered.
   *                         It may either ignore it, throw an unchecked exception,
   *                         or collect the exceptions
   * @return non-fragile function
   * @see Procedure1#nonFragileX(FragileProcedure1, BiConsumer)
   */
  @NotNull
  default Procedure1<P> nonFragileX(@NotNull BiConsumer<? super E, ? super P> exceptionHandler)
  {
    return Procedure1.nonFragileX(this, exceptionHandler);
  }

  /**
   * Apply this procedure only if the argument is not {@code null}.
   *<p>
   * Using this invocation it is guaranteed that {@link #apply(Object)} will always be called with
   * a non-null argument.
   * </p>
   * <p>
   *   Use {@link de.caff.generics.Types#notNull(Object, Object)} if yo want to invoke this
   *   with a default argument when {@code argument} is {@code null}.
   * </p>
   * @param argument argument of this function, possibly {@code null}
   * @throws E exception
   */
  default void applyNonNull(@Nullable P argument)
          throws E
  {
    if (argument != null) {
      apply(argument);
    }
  }

  /**
   * Return a partial evaluation of this procedure.
   *
   * @param argument constant argument use for the evaluation
   * @return zero argument procedure
   */
  @NotNull
  default FragileProcedure0<E> partial(P argument)
  {
    return () -> apply(argument);
  }

  /**
   * Return a procedure which does nothing, not even throw.
   * This just returns {@link #NULL}, but the compiler will accept it smoothly.
   * @param <X> exception type
   * @param <T> argument type
   * @return do-nothing procedure
   */
  @NotNull
  @SuppressWarnings("unchecked") // do nothing is always same
  static <X extends Exception, T> FragileProcedure1<X, T> empty()
  {
    return (FragileProcedure1<X, T>)NULL;
  }
}
