// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.function;

import de.caff.annotation.NotNull;
import de.caff.generics.WrappedFragileException;

import java.util.function.Consumer;

/**
 * Function without return and with no paramete, but a possible expectionr.
 * In a pure functional world this is useless, but in the real world it can be useful sometimes.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @see Procedure0
 * @see FragileProcedure1
 * @see FragileProcedure2
 * @see FragileProcedure3
 * @param <E> exception thrown by the procedure
 */
@FunctionalInterface
public interface FragileProcedure0<E extends Exception>
{
  /**
   * Fragile procedure with no arguments which does nothing, not even throw.
   * Use {@link #empty()} instead.
   */
  @NotNull
  FragileProcedure0<?>  NULL = () -> {};

  /**
   * Run the function.
   * @throws E exception
   */
  void apply() throws E;

  /**
   * Convert this to a non-fragile procedure which does not throw a checked exception.
   * This will instead throw an unchecked {@link WrappedFragileException} for any
   * checked exception thrown during {@link #apply()}.
   * @return non-fragile function
   * @see Procedure0#nonFragile(FragileProcedure0)
   */
  @NotNull
  default Procedure0 nonFragile()
  {
    return Procedure0.nonFragile(this);
  }

  /**
   * Convert this to a non-fragile procedure which does not a checked exception.
   * This will instead throw an unchecked {@link WrappedFragileException} for any
   * checked exception thrown during {@link #apply()}.
   * @param exceptionHandler handler called when an exception is encountered.
   *                         It may either ignore it, throw an unchecked exception,
   *                         or collect the exceptions
   * @return non-fragile function
   * @see Procedure0#nonFragileX(FragileProcedure0, Consumer)
   */
  @NotNull
  default Procedure0 nonFragileX(@NotNull Consumer<? super E> exceptionHandler)
  {
    return Procedure0.nonFragileX(this, exceptionHandler);
  }

  /**
   * Return a zero argument procedure which does nothing, not even throw.
   * @param <X> exception type
   * @return empty do-nothing procedure
   */
  @NotNull
  @SuppressWarnings("unchecked") // as NULL is ignorant of types
  static <X extends Exception> FragileProcedure0<X> empty()
  {
    return (FragileProcedure0<X>)NULL;
  }
}
