// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.function;

import de.caff.annotation.NotNull;
import de.caff.generics.tuple.ITuple2;

import java.util.function.BiPredicate;

/**
 * A function with two arguments which returns a boolean value,
 * and might throw a checked exception.
 * <p>
 * Compared to the standard Java {@link BiPredicate}
 * this functional interface allows for checked exceptions.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @param <E> exception type
 * @param <P1> parameter type 1
 * @param <P2> parameter type 2
 * @see FragilePredicate1
 */
@FunctionalInterface
public interface FragilePredicate2<E extends Exception, P1, P2>
        extends FragilePredicate1<E, ITuple2<? extends P1, ? extends P2>>
{
  /**
   * Decide whether the given combination of arguments evaluates to {@code true}.
   * @param arg1 first argument
   * @param arg2 second argument
   * @return {@code true} if the combination of the given arguments evaluates to {@code true},
   *         {@code false} otherwise
   * @throws E checked exception thrown by implementation
   */
  boolean test(P1 arg1, P2 arg2) throws E;

  /**
   * Expand a tuple into the arguments of this predicate.
   * @param tuple2 tuple to expand
   * @return the result of testing the expended eleeents of the tuple
   * @throws E checked exception thrown by implementation
   */
  @Override
  default boolean test(@NotNull ITuple2<? extends P1, ? extends P2> tuple2)
          throws E
  {
    return test(tuple2._1(), tuple2._2());
  }

  /**
   * Combine this and another predicate with logical and.
   * The result will use shortcut evaluation, so if this
   * predicate already returns {@code false} the other one is
   * not evaluated.
   *
   * @param other the other predicate to be chained with logical and
   * @return predicates combined with logical and
   */
  @NotNull
  default FragilePredicate2<E, P1, P2> and(@NotNull BiPredicate<? super P1, ? super P2> other)
  {
    return (p1, p2) -> test(p1, p2) && other.test(p1, p2);
  }

  /**
   * Combine this and another predicate with logical or.
   * The result will use shortcut evaluation, so if this
   * predicate already returns {@code true} the other one is
   * not evaluated.
   *
   * @param other other predicate to be chained with logical or
   * @return predicates combined with logical or
   */
  @NotNull
  default FragilePredicate2<E, P1, P2> or(@NotNull BiPredicate<? super P1, ? super P2> other)
  {
    return (p1, p2) -> test(p1, p2) || other.test(p1, p2);
  }

  /**
   * Combine this and another predicate with logical exclusive or.
   *
   * @param other other predicate to be chained with logical exclusive or
   * @return predicates combined with logical exclusive or
   */
  @NotNull
  default FragilePredicate2<E, P1, P2> xor(@NotNull BiPredicate<? super P1, ? super P2> other)
  {
    return (p1, p2) -> test(p1, p2) ^ other.test(p1, p2);
  }

  /**
   * Get the negation of this predicate.
   * @return negation of this predicate
   */
  @NotNull
  default FragilePredicate2<E, P1, P2> negate()
  {
    return new FragilePredicate2<E, P1, P2>()
    {
      @Override
      public boolean test(P1 arg1, P2 arg2) throws E
      {
        return !FragilePredicate2.this.test(arg1, arg2);
      }

      @NotNull
      @Override
      public FragilePredicate2<E, P1, P2> negate()
      {
        return FragilePredicate2.this;
      }
    };
  }

  /**
   * Get a partially applied predicate using a fix first argument.
   * @param argument value applied for first argument
   * @return partially evaluated predicate
   */
  @NotNull
  default FragilePredicate1<E, P2> partial1(P1 argument)
  {
    return arg -> test(argument, arg);
  }

  /**
   * Get a partially applied predicate using a fix second argument.
   * @param argument value applied for second argument
   * @return partially evaluated predicate
   */
  @NotNull
  default FragilePredicate1<E, P1> partial2(P2 argument)
  {
    return arg -> test(arg, argument);
  }

  /**
   * Get a two argument predicate which expects the arguments in reverse order.
   * @return predicate with reversed order of arguments
   */
  @NotNull
  default FragilePredicate2<E, P2, P1> reverseOrder()
  {
    return (arg2, arg1) -> test(arg1, arg2);
  }

  /**
   * Get a predicate which is the combination of two predicates
   * with a logical and.
   * <p>
   * The result will use shortcut evaluation, so if this
   * predicate already returns {@code false} the other one is
   * not evaluated.
   *
   * @param pred1 first predicate
   * @param pred2 second predicate
   * @param <T1> resulting type of first argument,
   *            which is the most extending type of both {@code pred1}'s
   *            first argument type and {@code pred2}'s first argument type,
   *            so if neither type is extending the
   *            other this method will not compile
   * @param <T2> resulting type of second argument, see {@code <T1>} for the fine print
   * @param <X> combined exception type
   * @return combination of both predicates with logical and
   * @see #and(BiPredicate)
   */
  @NotNull
  static <X extends Exception, T1, T2> FragilePredicate2<X, T1, T2> and(@NotNull FragilePredicate2<? extends X, ? super T1, ? super T2> pred1,
                                                                        @NotNull FragilePredicate2<? extends X, ? super T1, ? super T2> pred2)
  {
    return (p1, p2) -> pred1.test(p1, p2) && pred2.test(p1, p2);
  }

  /**
   * Get a predicate which is the combination of two predicates
   * with a logical or.
   * <p>
   * The result will use shortcut evaluation, so if this
   * predicate already returns {@code true} the other one is
   * not evaluated.
   *
   * @param pred1 first predicate
   * @param pred2 second predicate
   * @param <T1> resulting type of first argument,
   *            which is the most extending type of both {@code pred1}'s
   *            first argument type and {@code pred2}'s first argument type,
   *            so if neither type is extending the
   *            other this method will not compile
   * @param <T2> resulting type of second argument, see {@code <T1>} for the fine print
   * @param <X> combined exception type
   * @return combination of both predicates with logical or
   * @see #or(BiPredicate)
   */
  @NotNull
  static <X extends Exception, T1, T2> FragilePredicate2<X, T1, T2> or(@NotNull FragilePredicate2<? extends X, ? super T1, ? super T2> pred1,
                                                                       @NotNull FragilePredicate2<? extends X, ? super T1, ? super T2> pred2)
  {
    return (p1, p2) -> pred1.test(p1, p2) || pred2.test(p1, p2);
  }

  /**
   * Get a predicate which is the combination of two predicates
   * with a logical exclusive or.
   * <p>
   * The result will use shortcut evaluation, so if this
   * predicate already returns {@code true} the other one is
   * not evaluated.
   *
   * @param pred1 first predicate
   * @param pred2 second predicate
   * @param <T1> resulting type of first argument,
   *            which is the most extending type of both {@code pred1}'s
   *            first argument type and {@code pred2}'s first argument type,
   *            so if neither type is extending the
   *            other this method will not compile
   * @param <T2> resulting type of second argument, see {@code <T1>} for the fine print
   * @param <X> combined exception type
   * @return combination of both predicates with logical exclusive or
   * @see #xor(BiPredicate)
   */
  @NotNull
  static <X extends Exception, T1, T2> FragilePredicate2<X, T1, T2> xor(@NotNull FragilePredicate2<? extends X, ? super T1, ? super T2> pred1,
                                                                        @NotNull FragilePredicate2<? extends X, ? super T1, ? super T2> pred2)
  {
    return (p1, p2) -> pred1.test(p1, p2) ^ pred2.test(p1, p2);
  }
}
