// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.function;

import de.caff.annotation.NotNull;

import java.util.function.Function;
import java.util.function.Predicate;

/**
 * A function with one argument which returns a boolean value,
 * and throws a checked exception.
 * <p>
 * Fragile predicates can be chained to simulate boolean logic by
 * {@link #and(Predicate)}, {@link #or(Predicate)},
 * {@link #xor(Predicate)}, and {@link #negate()}.
 * The chaining does not always work as expected from
 * standard Java expressions. Therefore, there are
 * versions of the above methods with an underscore suffix
 * which allow to widen the type: {@link #and_(Predicate)},
 * {@link #or_(Predicate)}, {@link #xor_(Predicate)}.
 * <p>
 * There are also static methods which take care of the
 * type fiddling transparently: {@link #and(FragilePredicate1, FragilePredicate1)},
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @param <E> exception type
 * @param <P> parameter type
 * @see FragilePredicate2
 */
@FunctionalInterface
public interface FragilePredicate1<E extends Exception, P>
{
  /**
   * Test the argument and check whether it evaluates.
   * @param arg argument to check
   * @return {@code true} if the argument evaluates to {@code true},<br>
   *         {@code false} if not
   * @throws E checked exception thrown by implementation
   */
  boolean test(P arg) throws E;

  /**
   * Test the argument only if it is not {@code null}, otherwise return the fallback.
   *<p>
   * Using this invocation it is guaranteed that {@link #test(Object)} will always be called with
   * a non-null argument.
   * </p>
   * @param argument argument of this function
   * @param fallback fallback returned if {@code argument} is {@code null}
   * @return the result of applying this predicate to the non-null {@code argument},
   *         or {@code fallback} if {@code argument} is null
   * @throws E checked exception thrown by implementation
   */
  default boolean testNonNull(P argument, boolean fallback) throws E
  {
    return argument == null
            ? fallback
            : test(argument);
  }

  /**
   * Combine this and another predicate with logical and.
   * The result will use shortcut evaluation, so if this
   * predicate already returns {@code false} the other one is
   * not evaluated.
   * @param other other predicate to be chained with logical and
   * @return predicates combined with logical and
   * @see #and(FragilePredicate1, FragilePredicate1)
   * @see #and_(Predicate)
   */
  @NotNull
  default FragilePredicate1<E, P> and(@NotNull Predicate<? super P> other)
  {
    return p -> test(p) && other.test(p);
  }

  /**
   * Combine this and another predicate with logical and.
   * The result will use shortcut evaluation, so if this
   * predicate already returns {@code false} the other one is
   * not evaluated.
   * <p>
   * Compared to {@link #and(Predicate)} this method allows
   * to widen the type of this predicate to the one of the other predicate
   * like in standard Jave logical expressions, eg
   * <pre>{@code
   *   Predicate1<Object> nonNull = o -> o != null;
   *   Predicate1<String> notEmpty = s -> !s.isEmpty();
   *   // Predicate1<???> bad = nonNull.and(notEmpty);     // not possible as String is not a super type of Object
   *   Predicate1<String> valid = nonMull.and_(notEmpty);  // not nice, but it works
   * }</pre>
   * @param other other predicate
   * @param <T> type on which the other predicate operates, also the type of
   *           the returned predicate
   * @return predicates combined with logical and
   * @see #and(FragilePredicate1, FragilePredicate1)
   * @see #and(Predicate)
   */
  @NotNull
  default <T extends P> FragilePredicate1<E, T> and_(@NotNull Predicate<T> other)
  {
    return p -> test(p) && other.test(p);
  }

  /**
   * Combine this and another predicate with logical or.
   * The result will use shortcut evaluation, so if this
   * predicate already returns {@code true} the other one is
   * not evaluated.
   * @param other the other predicate to be chained with logical or
   * @return predicates combined with logical or
   * @see #or(FragilePredicate1, FragilePredicate1)
   * @see #or_(Predicate)
   */
  @NotNull
  default FragilePredicate1<E, P> or(@NotNull Predicate<? super P> other)
  {
    return p -> test(p) || other.test(p);
  }

  /**
   * Combine this and another predicate with logical or.
   * The result will use shortcut evaluation, so if this
   * predicate already returns {@code true} the other one is
   * not evaluated.
   * <p>
   * Compared to {@link #or(Predicate)} this method allows
   * to widen the type of this predicate to the one of the other predicate
   * like in standard Jave logical expressions, eg
   * <pre>{@code
   *   Predicate1<Object> isNull = o -> o == null;
   *   Predicate1<String> isEmpty = s -> !s.isEmpty();
   *   // Predicate1<???> bad = pred1.or(pred2);     // not possible as String is not a super type of Object
   *   Predicate1<String> useless = pred1.or_(pred2);  // not nice, but it works
   * }</pre>
   * @param other other predicate
   * @param <T> type on which the other predicate operates, also the type of
   *           the returned predicate
   * @return predicates combined with logical or
   * @see #or(FragilePredicate1, FragilePredicate1)
   * @see #or(Predicate)
   */
  @NotNull
  default <T extends P> FragilePredicate1<E, T> or_(@NotNull Predicate<T> other)
  {
    return p -> test(p) || other.test(p);
  }

  /**
   * Combine this and another predicate with logical exclusive or.
   * @param other the other predicate to be chained with logical exclusive or
   * @return predicates combined with logical exclusive or
   * @see #xor(FragilePredicate1, FragilePredicate1)
   * @see #xor_(Predicate)
   */
  @NotNull
  default FragilePredicate1<E, P> xor(@NotNull Predicate<? super P> other)
  {
    return p -> test(p) ^ other.test(p);
  }

  /**
   * Combine this and another predicate with logical exclusive or.
   * The result will use shortcut evaluation, so if this
   * predicate already returns {@code true} the other one is
   * not evaluated.
   * <p>
   * Compared to {@link #or(Predicate)} this method allows
   *  widening the type of this predicate to the one of the other predicate
   * like in standard Jave logical expressions, eg
   * <pre>{@code
   *   Predicate1<Object> isNull = o -> o == null;
   *   Predicate1<String> isEmpty = s -> !s.isEmpty();
   *   // Predicate1<???> bad = isNull.xor(isEmpty);     // not possible as String is not a super type of Object
   *   Predicate1<String> useless = isNull.xor_(isEmpty);  // bad example, but it works
   * }</pre>
   * @param other other predicate
   * @param <T> type on which the other predicate operates, also the type of
   *           the returned predicate
   * @return predicates combined with logical exclusive or
   * @see #xor(FragilePredicate1, FragilePredicate1)
   * @see #xor(Predicate)
   */
  @NotNull
  default <T extends P> FragilePredicate1<E, T> xor_(@NotNull Predicate<T> other)
  {
    return p -> test(p) ^ other.test(p);
  }

  /**
   * Get the negation of this predicate.
   * @return negation of this predicate
   */
  @NotNull
  default FragilePredicate1<E, P> negate()
  {
    return new FragilePredicate1<E, P>()
    {
      @Override
      public boolean test(P arg) throws E
      {
        return !FragilePredicate1.this.test(arg);
      }

      @NotNull
      @Override
      public FragilePredicate1<E, P> negate()
      {
        return FragilePredicate1.this;
      }
    };
  }

  /**
   * Use this predicate on a different type.
   * @param mapper mapper function which converts the different type to one this predicate accepts.
   * @param <S> different type on which the returned predicate operates
   * @return predicate on the new type, base on this one
   */
  @NotNull
  default <S> FragilePredicate1<E, S> adapted(@NotNull Function<? super S, ? extends P> mapper)
  {
    return s -> FragilePredicate1.this.test(mapper.apply(s));
  }

  /**
   * Get a predicate which is the combination of two predicates
   * with a logical and.
   * <p>
   * This method takes care of always creating a predicate of the correct type
   * without having to fiddle with {@link #and(Predicate)} or {@link #and_(Predicate)}
   * depending on the types.
   * <p>
   * The result will use shortcut evaluation, so if this
   * predicate already returns {@code false} the other one is
   * not evaluated.
   *
   * @param pred1 first predicate
   * @param pred2 second predicate
   * @param <T> resulting type, which is the most extending type of both {@code pred1}'s
   *           type and {@code pred2}'s type, so if neither type is extending the
   *           other this method will not compile
   * @param <X> common exception type
   * @return combination of both predicates with logical and
   * @see #and(Predicate)
   * @see #and_(Predicate)
   */
  @NotNull
  static <T, X extends Exception> FragilePredicate1<X, T> and(@NotNull FragilePredicate1<? extends X, ? super T> pred1,
                                                              @NotNull FragilePredicate1<? extends X, ? super T> pred2)
  {
    return p -> pred1.test(p) && pred2.test(p);
  }

  /**
   * Get a predicate which is the combination of two predicates
   * with a logical or.
   * <p>
   * This method takes care of always creating a predicate of the correct type
   * without having to fiddle with {@link #or(Predicate)} or {@link #or_(Predicate)}
   * depending on the types.
   * <p>
   * The result will use shortcut evaluation, so if this
   * predicate already returns {@code false} the other one is
   * not evaluated.
   *
   * @param pred1 first predicate
   * @param pred2 second predicate
   * @param <T> resulting type, which is the most extending type of both {@code pred1}'s
   *           type and {@code pred2}'s type, so if neither type is extending the
   *           other this method will not compile
   * @param <X> common exception type
   * @return combination of both predicates with logical or
   * @see #or(Predicate)
   * @see #or_(Predicate)
   */
  @NotNull
  static <T, X extends Exception> FragilePredicate1<X, T> or(@NotNull FragilePredicate1<? extends X, ? super T> pred1,
                                                             @NotNull FragilePredicate1<? extends X, ? super T> pred2)
  {
    return p -> pred1.test(p) || pred2.test(p);
  }

  /**
   * Get a predicate which is the combination of two predicates
   * with a logical exclusive or.
   * <p>
   * This method takes care of always creating a predicate of the correct type
   * without having to fiddle with {@link #or(Predicate)} or {@link #or_(Predicate)}
   * depending on the types.
   * <p>
   * The result will use shortcut evaluation, so if this
   * predicate already returns {@code true} the other one is
   * not evaluated.
   *
   * @param pred1 first predicate
   * @param pred2 second predicate
   * @param <T> resulting type, which is the most extending type of both {@code pred1}'s
   *           type and {@code pred2}'s type, so if neither type is extending the
   *           other this method will not compile
   * @param <X> common exception type
   * @return combination of both predicates with logical exclusive or
   * @see #xor(Predicate)
   * @see #xor_(Predicate)
   */
  @NotNull
  static <T, X extends Exception> FragilePredicate1<X, T> xor(@NotNull FragilePredicate1<? extends X, ? super T> pred1,
                                                              @NotNull FragilePredicate1<? extends X, ? super T> pred2)
  {
    return p -> pred1.test(p) ^ pred2.test(p);
  }
}
