// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================

package de.caff.generics.function;

import de.caff.annotation.NotNull;
import de.caff.generics.WrappedFragileException;
import de.caff.generics.tuple.ITuple4;

/**
 * Function with four parameters, possibly throwing a checked exception.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @param <R> return type
 * @param <E> exception type
 * @param <P1> first parameter type
 * @param <P2> second parameter type
 * @param <P3> third parameter type
 * @param <P4> fourth parameter type
 */
@FunctionalInterface
public interface FragileFunction4<R, E extends Exception, P1, P2, P3, P4>
        extends FragileFunction1<R, E, ITuple4<? extends P1, ? extends P2, ? extends P3, ? extends P4>>
{
  /**
   * Run the function.
   * @param arg1 first argument
   * @param arg2 second argument
   * @param arg3 third argument
   * @param arg4 fourth argument
   * @return return value
   * @throws E typed exception
   */
  R apply(P1 arg1, P2 arg2, P3 arg3, P4 arg4) throws E;

  /**
   * Expand a tuple into the arguments of this function.
   * @param tuple4 tuple to expand
   * @return the result of calling this function with the arguments from the expanded tuple
   * @throws E typed exception
   */
  @Override
  default R apply(@NotNull ITuple4<? extends P1, ? extends P2, ? extends P3, ? extends P4> tuple4) throws E
  {
    return apply(tuple4._1(), tuple4._2(), tuple4._3(), tuple4._4());
  }

  /**
   * Convert this to a non-fragile function which does not throw a checked exception.
   * This will instead throw an unchecked {@link WrappedFragileException} for any
   * checked exception thrown during {@link #apply(Object, Object, Object, Object)}.
   * @return non-fragile function
   * @see Function4#nonFragile(FragileFunction4)
   */
  @NotNull
  default Function4<R, P1, P2, P3, P4> nonFragile()
  {
    return Function4.nonFragile(this);
  }

  /**
   * Convert this to a non-fragile function which does not throw a checked exception.
   * This will instead return the {@code fallback} value.
   * Please note that in this case the exception is completely lost!
   * @param fallback fallback value
   * @return non-fragile function
   * @see Function4#nonFragile(FragileFunction4, Object)
   */
  @NotNull
  default Function4<R, P1, P2, P3, P4> nonFragile(R fallback)
  {
    return Function4.nonFragile(this, fallback);
  }

  /**
   * Convert this to a non-fragile function which does not throw a checked exception.
   * @param exceptionHandler handler which can handle the exception how it sees fit.
   *                         E.g. throw an unchecked exception (compare {@link #nonFragile()},
   *                         return a fallback (compare {@link #nonFragile(Object)}),
   *                         or do the latter but also log or collect the exception
   * @return non-fragile function
   * @see Function4#nonFragileX(FragileFunction4, Function5)
   */
  @NotNull
  default Function4<R, P1, P2, P3, P4> nonFragileX(@NotNull Function5<? extends R, ? super E, ? super P1, ? super P2, ? super P3, ? super P4> exceptionHandler)
  {
    return Function4.nonFragileX(this, exceptionHandler);
  }

  /**
   * Get a partially applied function.
   * @param argument value applied for first argument
   * @return partially evaluated function
   */
  @NotNull
  default FragileFunction3<R, E, P2, P3, P4> partial1(P1 argument)
  {
    return (P2 arg2, P3 arg3, P4 arg4) -> FragileFunction4.this.apply(argument, arg2, arg3, arg4);
  }

  /**
   * Get a partially applied function.
   * @param argument value applied for second argument
   * @return partially evaluated function
   */
  @NotNull
  default FragileFunction3<R, E, P1, P3, P4> partial2(P2 argument)
  {
    return (arg1, arg3, arg4) -> FragileFunction4.this.apply(arg1, argument, arg3, arg4);
  }

  /**
   * Get a partially applied function.
   * @param argument value applied for third argument
   * @return partially evaluated function
   */
  @NotNull
  default FragileFunction3<R, E, P1, P2, P4> partial3(P3 argument)
  {
    return (arg1, arg2, arg4) -> FragileFunction4.this.apply(arg1, arg2, argument, arg4);
  }

  /**
   * Get a partially applied function.
   * @param argument value applied for fourth argument
   * @return partially evaluated function
   */
  @NotNull
  default FragileFunction3<R, E, P1, P2, P3> partial4(P4 argument)
  {
    return (arg1, arg2, arg3) -> FragileFunction4.this.apply(arg1, arg2, arg3, argument);
  }
}
