// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================

package de.caff.generics.function;

import de.caff.annotation.NotNull;
import de.caff.generics.WrappedFragileException;
import de.caff.generics.tuple.ITuple2;

import java.util.function.BiFunction;

/**
 * Function with two parameters, possibly throwing an exception.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @param <R> return type
 * @param <E> exception type
 * @param <P1> first parameter type
 * @param <P2> second parameter type
 */
@FunctionalInterface
public interface FragileFunction2<R, E extends Exception, P1, P2>
  extends FragileFunction1<R, E, ITuple2<? extends P1, ? extends P2>>
{
  /**
   * Run the function.
   * @param arg1 first argument
   * @param arg2 second argument
   * @return return value
   * @throws E typed exception
   */
  R apply(P1 arg1, P2 arg2) throws E;

  /**
   * Expand a tuple into the arguments of this function.
   * @param tuple2 tuple to expand
   * @return the result of calling this function with the arguments from the expanded tuple
   * @throws E typed exception
   */
  @Override
  default R apply(@NotNull ITuple2<? extends P1, ? extends P2> tuple2) throws E
  {
    return apply(tuple2._1(), tuple2._2());
  }

  /**
   * Convert this to a non-fragile function which does not throw a checked exception.
   * This will instead throw an unchecked {@link WrappedFragileException} for any
   * checked exception thrown during {@link #apply(Object, Object)}.
   * @return non-fragile function
   * @see Function2#nonFragile(FragileFunction2)
   */
  @NotNull
  default Function2<R, P1, P2> nonFragile()
  {
    return Function2.nonFragile(this);
  }

  /**
   * Convert this to a non-fragile function which does not throw a checked exception.
   * This will instead return the {@code fallback} value.
   * Please note that in this case the exception is completely lost!
   * @param fallback fallback value
   * @return non-fragile function
   * @see Function2#nonFragile(FragileFunction2, Object)
   */
  @NotNull
  default Function2<R, P1, P2> nonFragile(R fallback)
  {
    return Function2.nonFragile(this, fallback);
  }

  /**
   * Convert this to a non-fragile function which does not throw a checked exception.
   * @param exceptionHandler handler which can handle the exception how it sees fit.
   *                         E.g. throw an unchecked exception (compare {@link #nonFragile()},
   *                         return a fallback (compare {@link #nonFragile(Object)}),
   *                         or do the latter but also log or collect the exception
   * @return non-fragile function
   * @see Function2#nonFragileX(FragileFunction1, BiFunction)
   */
  @NotNull
  default Function2<R, P1, P2> nonFragileX(@NotNull Function3<? extends R, ? super E, ? super P1, ? super P2> exceptionHandler)
  {
    return Function2.nonFragileX(this, exceptionHandler);
  }

  /**
   * Get a partially applied function.
   * @param argument value applied for first argument
   * @return partially evaluated function
   */
  @NotNull
  default FragileFunction1<R, E, P2> partial1(P1 argument)
  {
    return arg2 -> FragileFunction2.this.apply(argument, arg2);
  }

  /**
   * Get a partially applied function.
   * @param argument value applied for second argument
   * @return partially evaluated function
   */
  @NotNull
  default FragileFunction1<R, E, P1> partial2(P2 argument)
  {
    return arg1 -> FragileFunction2.this.apply(arg1, argument);
  }
}
