// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================

package de.caff.generics.function;

import de.caff.annotation.NotNull;
import de.caff.generics.WrappedFragileException;

import java.util.function.BiFunction;

/**
 * Function with one parameter, possibly throwing an exception.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @param <R> return type
 * @param <E> exception type
 * @param <P> parameter type
 */
@FunctionalInterface
public interface FragileFunction1<R, E extends Exception, P>
{
  /**
   * Run the function.
   * @param arg function argument
   * @return return value
   * @throws E typed exception
   */
  R apply(P arg) throws E;

  /**
   * Convert this to a non-fragile function which does not throw a checked exception.
   * This will instead throw an unchecked {@link WrappedFragileException} for any
   * checked exception thrown during {@link #apply(Object)}.
   * @return non-fragile function
   * @see Function1#nonFragile(FragileFunction1)
   */
  @NotNull
  default Function1<R, P> nonFragile()
  {
    return Function1.nonFragile(this);
  }

  /**
   * Convert this to a non-fragile function which does not throw a checked exception.
   * This will instead return the {@code fallback} value.
   * Please note that in this case the exception is completely lost!
   * @param fallback fallback value
   * @return non-fragile function
   * @see Function1#nonFragile(FragileFunction1, Object)
   */
  @NotNull
  default Function1<R, P> nonFragile(R fallback)
  {
    return Function1.nonFragile(this, fallback);
  }

  /**
   * Convert this to a non-fragile function which does not throw a checked exception.
   * @param exceptionHandler handler which can handle the exception how it sees fit.
   *                         E.g. throw an unchecked exception (compare {@link #nonFragile()},
   *                         return a fallback (compare {@link #nonFragile(Object)}),
   *                         or do the latter but also log or collect the exception
   * @return non-fragile function
   * @see Function1#nonFragileX(FragileFunction1, BiFunction)
   */
  @NotNull
  default Function1<R, P> nonFragileX(@NotNull BiFunction<? super E, ? super P, ? extends R> exceptionHandler)
  {
    return Function1.nonFragileX(this, exceptionHandler);
  }
}
