// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright ©2023-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.function;

import de.caff.annotation.NotNull;
import de.caff.generics.Order;
import de.caff.generics.Primitives;

import java.io.Serializable;
import java.util.Comparator;

/**
 * Define an order for two primitive float values.
 * <p>
 * Compared to a  {@code Comparator<Float>} this can
 * avoid boxing and unboxing in various situations.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since January 11, 2023
 */
@FunctionalInterface
public interface FloatOrdering
{
  /**
   * Natural order of ascending {@code float} values.
   * Besides collapsing all NaN values and ordering them between
   * everything else this ordering is nearer to IEEE 784 than
   * standard {@link Float#compare(float, float)}.
   *
   * @see Primitives#order(float, float)
   * @see #STANDARD_ASCENDING
   */
  FloatOrdering NATURAL_ASCENDING = new FloatOrdering.Serial()
  {
    private static final long serialVersionUID = 3025885466446256167L;

    @NotNull
    @Override
    public Order checkFloat(float v1, float v2)
    {
      return Primitives.order(v1, v2);
    }

    @NotNull
    @Override
    public Comparator<Float> asFloatComparator()
    {
      return (Comparator<Float> & Serializable)Primitives::compare;
    }

    @NotNull
    @Override
    public FloatOrdering inverse()
    {
      return NATURAL_DESCENDING;
    }
  };

  /**
   * Natural order of descending {@code float} values.
   * Besides collapsing all NaN values and ordering them before
   * everything else this ordering is nearer to IEEE 784 than
   * standard {@link Float#compare(float, float)}.
   * @see Primitives#order(float, float)
   * @see #STANDARD_DESCENDING
   */
  FloatOrdering NATURAL_DESCENDING = new FloatOrdering.Serial()
  {
    private static final long serialVersionUID = 3025885466446256167L;

    @NotNull
    @Override
    public Order checkFloat(float v1, float v2)
    {
      return Primitives.order(v2, v1);
    }

    @NotNull
    @Override
    public Comparator<Float> asFloatComparator()
    {
      return (Comparator<Float> & Serializable)(v1, v2) -> Primitives.compare(v2, v1);
    }

    @NotNull
    @Override
    public FloatOrdering inverse()
    {
      return NATURAL_ASCENDING;
    }
  };

  /**
   * Standard Java order of ascending {@code float} values.
   * Please note that this order is broken as it considers
   * {@code -0.0 < 0.0} which is not correct, for primitive
   * {@code float} values {@code -0.0 == 0.0} is valid.
   * @see #NATURAL_ASCENDING
   */
  FloatOrdering STANDARD_ASCENDING = new FloatOrdering.Serial()
  {
    private static final long serialVersionUID = 3025885466446256167L;

    @NotNull
    @Override
    public Order checkFloat(float v1, float v2)
    {
      if (v1 < v2) {
        return Order.Ascending;
      }
      if (v1 > v2) {
        return Order.Descending;
      }

      final int bits1 = Float.floatToIntBits(v1);
      final int bits2 = Float.floatToIntBits(v2);

      return bits1 == bits2
              ? Order.Same
              : (bits1 < bits2
                         ? Order.Ascending
                         : Order.Descending);

    }

    @NotNull
    @Override
    public FloatOrdering inverse()
    {
      return FloatOrdering.STANDARD_DESCENDING;
    }

    @NotNull
    @Override
    public Comparator<Float> asFloatComparator()
    {
      return Float::compare;
    }
  };

  /**
   * Inverse standard Java order of {@code float} values.
   * Please note that this order is broken as it considers
   * {@code -0.0 < 0.0} which is not correct, for primitive
   * {@code float} values {@code -0.0 == 0.0} is valid.
   */
  FloatOrdering STANDARD_DESCENDING = new FloatOrdering.Serial()
  {
    private static final long serialVersionUID = 8490254886442973329L;

    @NotNull
    @Override
    public Order checkFloat(float v2, float v1)
    {  // code copied for efficiency
      if (v1 < v2) {
        return Order.Ascending;
      }
      if (v1 > v2) {
        return Order.Descending;
      }

      final int bits1 = Float.floatToIntBits(v1);
      final int bits2 = Float.floatToIntBits(v2);

      return bits1 == bits2
              ? Order.Same
              : (bits1 < bits2
                         ? Order.Ascending
                         : Order.Descending);
    }

    @NotNull
    @Override
    public FloatOrdering inverse()
    {
      return FloatOrdering.STANDARD_ASCENDING;
    }

    @NotNull
    @Override
    public Comparator<Float> asFloatComparator()
    {
      return (v1, v2) -> Float.compare(v2, v1);
    }
  };

  /**
   * Get the order of the two values.
   * @param v1 first value
   * @param v2 second value
   * @return order of the two values
   */
  @NotNull
  Order checkFloat(float v1, float v2);

  /**
   * Are the two values represented in strictly ascending order?
   * @param v1 first value
   * @param v2 second value
   * @return {@code true} if {@code v1 < v2} according to this ordering<br>
   *         {@code false} if {@code v1 >= v2} according to this ordering
   * @see #descending(float, float)
   * @see #ascendingOrSame(float, float)
   * @see #descendingOrSame(float, float)
   * @see #same(float, float)
   * @see #different(float, float)
   */
  default boolean ascending(float v1, float v2)
  {
    return checkFloat(v1, v2).ascending;
  }

  /**
   * Are the two values represented in ascending order?
   * @param v1 first value
   * @param v2 second value
   * @return {@code true} if {@code v1 <= v2} according to this ordering<br>
   *         {@code false} if {@code v1 > v2} according to this ordering
   * @see #ascending(float, float)
   * @see #descending(float, float)
   * @see #descendingOrSame(float, float)
   * @see #same(float, float)
   * @see #different(float, float)
   */
  default boolean ascendingOrSame(float v1, float v2)
  {
    return checkFloat(v1, v2).ascendingOrSame;
  }

  /**
   * Are the two values represented in strictly descending order?
   * @param v1 first value
   * @param v2 second value
   * @return {@code true} if {@code v1 > v2} according to this ordering<br>
   *         {@code false} if {@code v1 <= v2} according to this ordering
   * @see #ascending(float, float)
   * @see #ascendingOrSame(float, float)
   * @see #descendingOrSame(float, float)
   * @see #same(float, float)
   * @see #different(float, float)
   */
  default boolean descending(float v1, float v2)
  {
    return checkFloat(v1, v2).descending;
  }

  /**
   * Are the two values represented in descending order?
   * @param v1 first value
   * @param v2 second value
   * @return {@code true} if {@code v1 >= v2} according to this ordering<br>
   *         {@code false} if {@code v1 < v2} according to this ordering
   * @see #ascending(float, float)
   * @see #descending(float, float)
   * @see #ascendingOrSame(float, float)
   * @see #same(float, float)
   * @see #different(float, float)
   */
  default boolean descendingOrSame(float v1, float v2)
  {
    return checkFloat(v1, v2).descendingOrSame;
  }

  /**
   * Are the two values considered equal by this order?
   * @param v1 first value
   * @param v2 second value
   * @return {@code true} if {@code v1 == v2} according to this order<br>
   *         {@code false} if {@code v1 != v2} according to this ordering
   * @see #ascending(float, float)
   * @see #descending(float, float)
   * @see #ascendingOrSame(float, float)
   * @see #descendingOrSame(float, float)
   * @see #different(float, float)
   */
  default boolean same(float v1, float v2)
  {
    return checkFloat(v1, v2).same;
  }

  /**
   * Are the two values considered different by this order?
   * @param v1 first value
   * @param v2 second value
   * @return {@code true} if {@code v1 != v2} according to this order<br>
   *         {@code false} if {@code v1 == v2} according to this ordering
   * @see #ascending(float, float)
   * @see #descending(float, float)
   * @see #ascendingOrSame(float, float)
   * @see #descendingOrSame(float, float)
   * @see #same(float, float)
   */
  default boolean different(float v1, float v2)
  {
    return checkFloat(v1, v2).different;
  }

  /**
   * Invert this order.
   * @return inverse order
   */
  @NotNull
  default FloatOrdering inverse()
  {
    return new FloatOrdering()
    {
      @Override
      @NotNull
      public Order checkFloat(float v1, float v2)
      {
        return FloatOrdering.this.checkFloat(v2, v1);
      }

      @NotNull
      @Override
      public FloatOrdering inverse()
      {
        return FloatOrdering.this;
      }
    };
  }

  /**
   * Use this ordering as a comparator.
   * @return comparator of {@code Float}
   */
  @NotNull
  default Comparator<Float> asFloatComparator()
  {
    return (v1, v2) -> checkFloat(v1, v2).comparison;
  }

  /**
   * View this ordering as a number comparator.
   * @return comparator of {@link Number}
   */
  @NotNull
  default Comparator<Number> asNumberComparator()
  {
    return (Comparator<Number> & Serializable)(v1, v2) -> checkFloat(v1.floatValue(), v2.floatValue()).comparison;
  }

  /**
   * Get this ordering as a generic float ordering. 
   * @return generic ordering for floats
   */
  @NotNull
  default Ordering<Float> asFloatOrdering()
  {
    return (Ordering<Float> & Serializable)this::checkFloat;
  }

  /**
   * Convert a standard comparator of {@code Double} into a double ordering.
   * @param comparator comparator of {@code Double}
   * @return ordering
   */
  @NotNull
  static FloatOrdering fromComparator(@NotNull Comparator<Float> comparator)
  {
    return (v1, v2) -> Order.fromCompare(comparator.compare(v1, v2));
  }

  /**
   * A float ordering which is serializable.
   * It is a recommendation that comparators implement {@code java.io.Serializable}
   * to allow serializing {@code TreeMap}s and similar. If an Ordering is expected to be used
   * as a comparator in such areas implement either this interface, or for lambdas
   * use a cast to force the compiler to make the order serializable:
   * <pre>{@code
   *   return (ByteOrdering & Serializable) (v1, v2) -> v1.compareTo(v2);
   * }</pre>
   * You can also use a {@code (ByteOrdering.Serial)} cast to save a bit of typing,
   * but the above is a nice trick to have in your toolbox.
   */
  interface Serial extends FloatOrdering,
                           java.io.Serializable
  {
  }
}
