// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2019-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.function;

import de.caff.annotation.NotNull;

/**
 * Binary operator with two float arguments and a float result.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since October 31, 2019
 */
@FunctionalInterface
public interface FloatOperator2
{
  /**
   * Float operator which returns the sum of two arguments:
   * {@code a + b}.
   */
  FloatOperator2 PLUS = (a, b) -> a + b;
  /**
   * Float operator which returns the difference of two arguments:
   * {@code a - b}
   */
  FloatOperator2 MINUS = (a, b) -> a - b;
  /**
   * Float operator which returns the product of two arguments:
   * {@code a * b}
   */
  FloatOperator2 MULT = (a, b) -> a * b;
  /**
   * Float operator which returns the division result of two arguments:
   * {@code a / b}
   */
  FloatOperator2 DIV = (a, b) -> a / b;
  /**
   * Float operator which returns the modulo result of two arguments:
   * {@code a % b}.
   */
  FloatOperator2 MOD = (a, b) -> a % b;
  /**
   * Apply the function.
   * @param arg1 first argument
   * @param arg2 second argument
   * @return result of the function
   */
  float applyAsFloat(float arg1, float arg2);

  /**
   * Get a partially applied function.
   * @param argument value applied for first argument
   * @return partially evaluated function
   */
  @NotNull
  default FloatOperator1 partialLeft(float argument)
  {
    return arg -> applyAsFloat(argument, arg);
  }

  /**
   * Get a partially applied function.
   * @param argument value applied for second argument
   * @return partially evaluated function
   */
  @NotNull
  default FloatOperator1 partialRight(float argument)
  {
    return arg -> applyAsFloat(arg, argument);
  }

  /**
   * Create a function which applies this function and feeds the result into the next function.
   * @param after function called with the result of this function
   * @return chained function
   */
  @NotNull
  default FloatOperator2 andThen(@NotNull FloatOperator1 after)
  {
    return (arg1, arg2) -> after.applyAsFloat(applyAsFloat(arg1, arg2));
  }
}
